﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include "../kern_Platform.h"
#include "../kern_Kernel.h"
#include "../kern_Utility.h"
#include "../kern_InterruptManagerSelect.h"
#include "../kern_KInterruptEvent.h"
#include "../kern_KProcess.h"
#include "../kern_KHandleTable.h"
#include "../kern_KScheduler.h"
#include "../kern_CPUSelect.h"
#include "../kern_KDeviceAddressSpace.h"
#include "kern_SystemControl.h"
#include "kern_SvcHandlers.h"

namespace nn { namespace kern { namespace svc {
namespace {
const uint64_t DeviceAddressSpaceAlignMask = (1ull << 22) - 1;

Result
SvcCreateInterruptEvent(nn::svc::Handle* pReadHandle, int32_t name, nn::svc::InterruptType type)
{
    Result result;

    switch (type)
    {
    case nn::svc::InterruptType_Edge:
    case nn::svc::InterruptType_Level:
        break;
    default:
        return nn::svc::ResultInvalidEnum();
    }

    KProcess& process = GetCurrentProcess();

    if (!process.IsPermitIsPermittedInterrupt(name))
    {
        return nn::svc::ResultNotFound();
    }

    KHandleTable& handleTable = process.GetHandleTable();

    KInterruptEvent* pEvent = KInterruptEvent::Create();
    if (pEvent)
    {
        result = pEvent->Initialize(name, type);

        if (result.IsSuccess())
        {
            result = KInterruptEvent::Register(pEvent);
            if (result.IsSuccess())
            {
                result = handleTable.Add(pReadHandle, pEvent);
            }
        }
        pEvent->Close();
    }
    else
    {
        result = nn::svc::ResultOutOfResource();
    }

    return result;
}

Result SvcCreateDeviceAddressSpace(nn::svc::Handle* pOut, uint64_t spaceAddress, uint64_t spaceSize)
{
    if (spaceAddress % NN_KERN_FINEST_PAGE_SIZE != 0)
    {
        return nn::svc::ResultInvalidRegion();
    }
    if ((spaceSize % NN_KERN_FINEST_PAGE_SIZE) != 0 || spaceSize == 0)
    {
        return nn::svc::ResultInvalidRegion();
    }
    if (!(spaceAddress < spaceAddress + spaceSize))
    {
        return nn::svc::ResultInvalidRegion();
    }

    KDeviceAddressSpace* pDeviceAddressSpace = KDeviceAddressSpace::Create();
    if (!pDeviceAddressSpace)
    {
        return nn::svc::ResultOutOfResource();
    }

    KScopedAutoObject<KDeviceAddressSpace> autoCloser(pDeviceAddressSpace);

    Result result = pDeviceAddressSpace->Initialize(spaceAddress, spaceSize);
    if (result.IsFailure())
    {
        return result;
    }

    result = KDeviceAddressSpace::Register(pDeviceAddressSpace);
    if (result.IsSuccess())
    {
        result = GetCurrentProcess().GetHandleTable().Add(pOut, pDeviceAddressSpace);
    }

    return result;
}

Result SvcAttachDeviceAddressSpace(nn::svc::DeviceName deviceName, nn::svc::Handle handle)
{
    KDeviceAddressSpace* pDeviceAddressSpace = GetCurrentProcess().GetHandleTable().GetObject<KDeviceAddressSpace>(handle);

    if (!pDeviceAddressSpace)
    {
        return nn::svc::ResultInvalidHandle();
    }
    KScopedAutoObject<KDeviceAddressSpace> autoCloser(pDeviceAddressSpace);

    return pDeviceAddressSpace->Attach(deviceName);
}

Result SvcDetachDeviceAddressSpace(nn::svc::DeviceName deviceName, nn::svc::Handle handle)
{
    KDeviceAddressSpace* pDeviceAddressSpace = GetCurrentProcess().GetHandleTable().GetObject<KDeviceAddressSpace>(handle);

    if (!pDeviceAddressSpace)
    {
        return nn::svc::ResultInvalidHandle();
    }
    KScopedAutoObject<KDeviceAddressSpace> autoCloser(pDeviceAddressSpace);

    return pDeviceAddressSpace->Detach(deviceName);
}

Result SvcMapDeviceAddressSpace(size_t* pMapSize, nn::svc::Handle addressSpace, nn::svc::Handle process,
        uint64_t processAddress, size_t size, uint64_t deviceAddress, nn::svc::MemoryPermission devicePermission, bool breakAllocate = true)
{
    if ((processAddress % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((deviceAddress % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(processAddress < processAddress + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (!(deviceAddress < deviceAddress + size))
    {
        return nn::svc::ResultInvalidRegion();
    }
    if (processAddress != static_cast<uintptr_t>(processAddress))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    switch (devicePermission)
    {
    case nn::svc::MemoryPermission_Read:
    case nn::svc::MemoryPermission_Write:
    case nn::svc::MemoryPermission_ReadWrite:
        break;
    default:
        return nn::svc::ResultInvalidNewMemoryPermission();
    }

    KDeviceAddressSpace* pDeviceAddressSpace = GetCurrentProcess().GetHandleTable().GetObject<KDeviceAddressSpace>(addressSpace);

    if (!pDeviceAddressSpace)
    {
        return nn::svc::ResultInvalidHandle();
    }
    KScopedAutoObject<KDeviceAddressSpace> autoCloser(pDeviceAddressSpace);

    KProcess* pProcess = GetObjectFromRealOrPseudoHandle<KProcess>(GetCurrentProcess().GetHandleTable(), process);

    if (!pProcess)
    {
        return nn::svc::ResultInvalidHandle();
    }
    KScopedAutoObject<KProcess> autoProcessCloser(pProcess);

    KProcessPageTable& pageTable = pProcess->GetPageTable();
    if (!pageTable.IsInRange(processAddress, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    return pDeviceAddressSpace->Map(pMapSize, &pageTable, KProcessAddress(processAddress), size, deviceAddress, devicePermission, breakAllocate);
}

Result SvcMapDeviceAddressSpaceByForce(nn::svc::Handle addressSpace, nn::svc::Handle process,
        uint64_t processAddress, size_t size, uint64_t deviceAddress, nn::svc::MemoryPermission devicePermission)
{
    size_t mapSize = 0;
    return SvcMapDeviceAddressSpace(&mapSize, addressSpace, process, processAddress, size, deviceAddress, devicePermission, false);
}

Result SvcMapDeviceAddressSpaceAligned(nn::svc::Handle addressSpace, nn::svc::Handle process,
        uint64_t processAddress, size_t size, uint64_t deviceAddress,
        nn::svc::MemoryPermission devicePermission)
{
    if ((processAddress % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((deviceAddress % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((processAddress & DeviceAddressSpaceAlignMask) != (deviceAddress & DeviceAddressSpaceAlignMask))
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(processAddress < processAddress + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (!(deviceAddress < deviceAddress + size))
    {
        return nn::svc::ResultInvalidRegion();
    }
    if (processAddress != static_cast<uintptr_t>(processAddress))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    switch (devicePermission)
    {
    case nn::svc::MemoryPermission_Read:
    case nn::svc::MemoryPermission_Write:
    case nn::svc::MemoryPermission_ReadWrite:
        break;
    default:
        return nn::svc::ResultInvalidNewMemoryPermission();
    }

    KDeviceAddressSpace* pDeviceAddressSpace = GetCurrentProcess().GetHandleTable().GetObject<KDeviceAddressSpace>(addressSpace);

    if (!pDeviceAddressSpace)
    {
        return nn::svc::ResultInvalidHandle();
    }
    KScopedAutoObject<KDeviceAddressSpace> autoCloser(pDeviceAddressSpace);

    KProcess* pProcess = GetObjectFromRealOrPseudoHandle<KProcess>(GetCurrentProcess().GetHandleTable(), process);

    if (!pProcess)
    {
        return nn::svc::ResultInvalidHandle();
    }
    KScopedAutoObject<KProcess> autoProcessCloser(pProcess);

    KProcessPageTable& pageTable = pProcess->GetPageTable();
    if (!pageTable.IsInRange(processAddress, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    return pDeviceAddressSpace->MapAligned(&pageTable, KProcessAddress(processAddress), size, deviceAddress, devicePermission);
}

Result SvcUnmapDeviceAddressSpace(nn::svc::Handle addressSpace, nn::svc::Handle process,
        uint64_t processAddress, size_t size, uint64_t deviceAddress)
{
    if ((processAddress % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((deviceAddress % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(processAddress < processAddress + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (!(deviceAddress < deviceAddress + size))
    {
        return nn::svc::ResultInvalidRegion();
    }
    if (processAddress != static_cast<uintptr_t>(processAddress))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KDeviceAddressSpace* pDeviceAddressSpace = GetCurrentProcess().GetHandleTable().GetObject<KDeviceAddressSpace>(addressSpace);

    if (!pDeviceAddressSpace)
    {
        return nn::svc::ResultInvalidHandle();
    }
    KScopedAutoObject<KDeviceAddressSpace> autoCloser(pDeviceAddressSpace);

    KProcess* pProcess = GetObjectFromRealOrPseudoHandle<KProcess>(GetCurrentProcess().GetHandleTable(), process);

    if (!pProcess)
    {
        return nn::svc::ResultInvalidHandle();
    }
    KScopedAutoObject<KProcess> autoProcessCloser(pProcess);

    KProcessPageTable& pageTable = pProcess->GetPageTable();
    if (!pageTable.IsInRange(processAddress, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    return pDeviceAddressSpace->Unmap(&pageTable, KProcessAddress(processAddress), size, deviceAddress);
}

Result SvcReadWriteRegister(Bit32* pOut, nn::svc::PhysicalAddress address, Bit32 mask, Bit32 value)
{
    *pOut = 0;
    return KSystemControl::ReadWriteRegister(pOut, address, mask, value);
}

}

#if defined(NN_BUILD_CONFIG_CPU_SVC_32)
Result SvcCreateInterruptEvent32(nn::svc::Handle* pReadHandle, int32_t name, nn::svc::InterruptType type)
{
    Result result = SvcCreateInterruptEvent(pReadHandle, name, type);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcCreateDeviceAddressSpace32(nn::svc::Handle* pOut, uint64_t spaceAddress, uint64_t spaceSize)
{
    Result result = SvcCreateDeviceAddressSpace(pOut, spaceAddress, spaceSize);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcAttachDeviceAddressSpace32(nn::svc::DeviceName deviceName, nn::svc::Handle handle)
{
    Result result = SvcAttachDeviceAddressSpace(deviceName, handle);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcDetachDeviceAddressSpace32(nn::svc::DeviceName deviceName, nn::svc::Handle handle)
{
    Result result = SvcDetachDeviceAddressSpace(deviceName, handle);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcMapDeviceAddressSpaceByForce32(nn::svc::Handle addressSpace, nn::svc::Handle process,
        uint64_t processAddress, size_t size, uint64_t deviceAddress,
        nn::svc::MemoryPermission devicePermission)
{
    Result result = SvcMapDeviceAddressSpaceByForce(addressSpace, process, processAddress, size, deviceAddress, devicePermission);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcMapDeviceAddressSpaceAligned32(nn::svc::Handle addressSpace, nn::svc::Handle process,
        uint64_t processAddress, size_t size, uint64_t deviceAddress,
        nn::svc::MemoryPermission devicePermission)
{
    Result result = SvcMapDeviceAddressSpaceAligned(addressSpace, process, processAddress, size, deviceAddress, devicePermission);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcMapDeviceAddressSpace32(size_t* pMappedSize, nn::svc::Handle addressSpace, nn::svc::Handle process,
        uint64_t processAddress, size_t size, uint64_t deviceAddress,
        nn::svc::MemoryPermission devicePermission)
{
    Result result = SvcMapDeviceAddressSpace(pMappedSize, addressSpace, process, processAddress, size, deviceAddress, devicePermission);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcUnmapDeviceAddressSpace32(nn::svc::Handle addressSpace, nn::svc::Handle process,
        uint64_t processAddress, size_t size, uint64_t deviceAddress)
{
    Result result = SvcUnmapDeviceAddressSpace(addressSpace, process, processAddress, size, deviceAddress);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcReadWriteRegister32(Bit32* pOut, nn::svc::PhysicalAddress address, Bit32 mask, Bit32 value)
{
    Result result = SvcReadWriteRegister(pOut, address, mask, value);
    ClearSvcOutRegistersReturnResult();
    return result;
}

#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64)
Result SvcCreateInterruptEvent64(nn::svc::Handle* pReadHandle, int32_t name, nn::svc::InterruptType type)
{
    Result result = SvcCreateInterruptEvent(pReadHandle, name, type);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcCreateDeviceAddressSpace64(nn::svc::Handle* pOut, uint64_t spaceAddress, uint64_t spaceSize)
{
    Result result = SvcCreateDeviceAddressSpace(pOut, spaceAddress, spaceSize);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcAttachDeviceAddressSpace64(nn::svc::DeviceName deviceName, nn::svc::Handle handle)
{
    Result result = SvcAttachDeviceAddressSpace(deviceName, handle);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcDetachDeviceAddressSpace64(nn::svc::DeviceName deviceName, nn::svc::Handle handle)
{
    Result result = SvcDetachDeviceAddressSpace(deviceName, handle);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcMapDeviceAddressSpaceByForce64(nn::svc::Handle addressSpace, nn::svc::Handle process,
        uint64_t processAddress, size_t size, uint64_t deviceAddress,
        nn::svc::MemoryPermission devicePermission)
{
    Result result = SvcMapDeviceAddressSpaceByForce(addressSpace, process, processAddress, size, deviceAddress, devicePermission);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcMapDeviceAddressSpaceAligned64(nn::svc::Handle addressSpace, nn::svc::Handle process,
        uint64_t processAddress, size_t size, uint64_t deviceAddress,
        nn::svc::MemoryPermission devicePermission)
{
    Result result = SvcMapDeviceAddressSpaceAligned(addressSpace, process, processAddress, size, deviceAddress, devicePermission);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcMapDeviceAddressSpace64(size_t* pMappedSize, nn::svc::Handle addressSpace, nn::svc::Handle process,
        uint64_t processAddress, size_t size, uint64_t deviceAddress,
        nn::svc::MemoryPermission devicePermission)
{
    Result result = SvcMapDeviceAddressSpace(pMappedSize, addressSpace, process, processAddress, size, deviceAddress, devicePermission);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcUnmapDeviceAddressSpace64(nn::svc::Handle addressSpace, nn::svc::Handle process,
        uint64_t processAddress, size_t size, uint64_t deviceAddress)
{
    Result result = SvcUnmapDeviceAddressSpace(addressSpace, process, processAddress, size, deviceAddress);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcReadWriteRegister64(Bit32* pOut, nn::svc::PhysicalAddress address, Bit32 mask, Bit32 value)
{
    Result result = SvcReadWriteRegister(pOut, address, mask, value);
    ClearSvcOutRegistersReturnResult();
    return result;
}

#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
Result
SvcCreateInterruptEvent64From32(nn::svc::Handle* pReadHandle, int32_t name, nn::svc::InterruptType type)
{
    Result result = SvcCreateInterruptEvent(pReadHandle, name, type);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcCreateDeviceAddressSpace64From32(nn::svc::Handle* pOut, uint64_t spaceAddress, uint64_t spaceSize)
{
    Result result = SvcCreateDeviceAddressSpace(pOut, spaceAddress, spaceSize);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcAttachDeviceAddressSpace64From32(nn::svc::DeviceName deviceName, nn::svc::Handle handle)
{
    Result result = SvcAttachDeviceAddressSpace(deviceName, handle);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcDetachDeviceAddressSpace64From32(nn::svc::DeviceName deviceName, nn::svc::Handle handle)
{
    Result result = SvcDetachDeviceAddressSpace(deviceName, handle);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcMapDeviceAddressSpaceByForce64From32(nn::svc::Handle addressSpace, nn::svc::Handle process,
        uint64_t processAddress, size_t size, uint64_t deviceAddress,
        nn::svc::MemoryPermission devicePermission)
{
    Result result = SvcMapDeviceAddressSpaceByForce(addressSpace, process, processAddress, size, deviceAddress, devicePermission);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcMapDeviceAddressSpaceAligned64From32(nn::svc::Handle addressSpace, nn::svc::Handle process,
        uint64_t processAddress, size_t size, uint64_t deviceAddress,
        nn::svc::MemoryPermission devicePermission)
{
    Result result = SvcMapDeviceAddressSpaceAligned(addressSpace, process, processAddress, size, deviceAddress, devicePermission);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcMapDeviceAddressSpace64From32(size_t* pMappedSize, nn::svc::Handle addressSpace, nn::svc::Handle process,
        uint64_t processAddress, size_t size, uint64_t deviceAddress,
        nn::svc::MemoryPermission devicePermission)
{
    Result result = SvcMapDeviceAddressSpace(pMappedSize, addressSpace, process, processAddress, size, deviceAddress, devicePermission);
    ClearSvcOutRegistersReturnResult();
    return result;
}


Result SvcUnmapDeviceAddressSpace64From32(nn::svc::Handle addressSpace, nn::svc::Handle process,
        uint64_t processAddress, size_t size, uint64_t deviceAddress)
{
    Result result = SvcUnmapDeviceAddressSpace(addressSpace, process, processAddress, size, deviceAddress);
    ClearSvcOutRegistersReturnResult();
    return result;
}

Result SvcReadWriteRegister64From32(Bit32* pOut, nn::svc::PhysicalAddress address, Bit32 mask, Bit32 value)
{
    Result result = SvcReadWriteRegister(pOut, address, mask, value);
    ClearSvcOutRegistersReturnResult();
    return result;
}

#endif // #if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
}}}
