﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include "../kern_Platform.h"
#include "../kern_DebugSelect.h"
#include "../kern_Kernel.h"
#include "../kern_KAutoObject.h"
#include "../kern_Panic.h"
#include "../kern_Utility.h"
#include "kern_MemoryMap.h"
#include "../kern_DebugString.h"
#include "../kern_HardwareBreakPointSelect.h"
#include "../kern_InterruptManagerSelect.h"
#include "../kern_MemoryCopySelect.h"
#include "../kern_KScopedSchedulingLock.h"
#include "kern_SystemControl.h"
#include "../kern_KTrace.h"
#include "kern_SvcHandlers.h"
#include "kern_SvcValueCheck.h"
#include <cstring>

namespace nn { namespace kern { namespace svc {

namespace {
const int NumDebuggableThreads = 96;

#define CHECK_KDEBUG(pDebug)                if (!pDebug) { return nn::svc::ResultInvalidHandle(); }

#ifdef NN_KERN_FOR_DEVELOPMENT
void PrintBreak(nn::svc::BreakReason reason)
{
    NN_KERN_RELEASE_LOG("nn::svc::Break(%d) called at pid=%lld(%s) tid=%lld\n",
            reason,
            GetCurrentProcess().GetId(),
            GetCurrentProcess().GetName(),
            GetCurrentThread().GetId());
    KDebug::PrintRegister();
    KDebug::PrintBacktrace();
}
#endif

Result SvcDebugActiveProcess(nn::svc::Handle* pOut, Bit64 targetProcessId)
{
    KProcess* pTargetProcess = KProcess::GetProcessFromId(targetProcessId);

    // IDチェック
    if (!pTargetProcess)
    {
        return nn::svc::ResultInvalidProcessId();
    }
    KScopedAutoObject<KProcess> autoCloser(pTargetProcess);

    // 対象プロセスがデバッグ禁止
    // 且つ、自プロセスが強制デバッグ不可なら失敗
    if (!pTargetProcess->IsPermittedDebug())
    {
        if (!GetCurrentProcess().CanForceDebug())
        {
            return nn::svc::ResultInvalidState();
        }
    }

    if (pTargetProcess == &GetCurrentProcess())
    {
        return nn::svc::ResultInvalidState();
    }

    // KHandleTableに登録
    // カレントプロセスのKHandleTable取得
    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();

    KDebug* pDebug = KDebug::Create();
    if (!pDebug)
    {
        return nn::svc::ResultOutOfResource();
    }

    KScopedAutoObject<KDebug> autoCloserDebug(pDebug);

    pDebug->Initialize();
    Result result = KDebug::Register(pDebug);
    if (result.IsFailure())
    {
        return result;
    }
    result = pDebug->Attach(pTargetProcess);
    if (result.IsFailure())
    {
        return result;
    }

    result = handleTable.Add(pOut, pDebug);

    return result;
}

Result SvcBreakDebugProcess(nn::svc::Handle debug)
{
    if (!KTargetSystem::IsDevelopmentHardware())
    {
        return nn::svc::ResultNotImplemented();
    }

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KDebug* pDebug = handleTable.GetObject<KDebug>(debug);

    CHECK_KDEBUG(pDebug);
    KScopedAutoObject<KDebug> autoCloser(pDebug);

    return pDebug->BreakProcess();
}

Result SvcContinueDebugEvent(nn::svc::Handle debug, Bit32 flags, KUserPointer<const Bit64*> pThreadIds, int32_t arraySize)
{
    Bit64 threadIds[NumDebuggableThreads];
    if (!KTargetSystem::IsDevelopmentHardware())
    {
        return nn::svc::ResultNotImplemented();
    }

    Bit32 allFlags = nn::svc::ContinueFlag_ExceptionHandled | nn::svc::ContinueFlag_EnableExceptionEvent | nn::svc::ContinueFlag_ContinueAll | nn::svc::ContinueFlag_ContinueOthers;
    if ((flags | allFlags) != allFlags)
    {
        return nn::svc::ResultInvalidEnum();
    }

    if ((flags & (nn::svc::ContinueFlag_ContinueAll | nn::svc::ContinueFlag_ContinueOthers)) ==
            (nn::svc::ContinueFlag_ContinueAll | nn::svc::ContinueFlag_ContinueOthers))
    {
        return nn::svc::ResultInvalidEnum();
    }

    if (!(0 <= arraySize && arraySize <= NumDebuggableThreads))
    {
        return nn::svc::ResultOutOfRange();
    }

    if (arraySize > 0)
    {
        if (pThreadIds.CopyArrayTo(threadIds, arraySize).IsFailure())
        {
            return nn::svc::ResultInvalidPointer();
        }
    }

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KDebug* pDebug = handleTable.GetObject<KDebug>(debug);

    CHECK_KDEBUG(pDebug);
    KScopedAutoObject<KDebug> autoCloser(pDebug);

    return pDebug->ContinueDebug(flags, threadIds, arraySize);
}

Result SvcGetDebugEvent(KUserPointer<nn::svc::DebugEventInfo*> pInfo, nn::svc::Handle debug )
{
    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KDebug* pDebug = handleTable.GetObject<KDebug>(debug);

    CHECK_KDEBUG(pDebug);
    KScopedAutoObject<KDebug> autoCloser(pDebug);

    nn::svc::DebugEventInfo info;
    std::memset(&info, 0, sizeof(info));
    Result result = pDebug->GetDebugEventInfo(&info);

    if (result.IsSuccess())
    {
        result = pInfo.CopyFrom(&info);
    }

    return result;
}

#if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
Result SvcGetDebugEvent(KUserPointer<nn::svc::ilp32::DebugEventInfo*> pInfo, nn::svc::Handle debug)
{
    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KDebug* pDebug = handleTable.GetObject<KDebug>(debug);

    CHECK_KDEBUG(pDebug);
    KScopedAutoObject<KDebug> autoCloser(pDebug);

    nn::svc::ilp32::DebugEventInfo info;
    std::memset(&info, 0, sizeof(info));
    Result result = pDebug->GetDebugEventInfo(&info);

    if (result.IsSuccess())
    {
        result = pInfo.CopyFrom(&info);
    }

    return result;
}
#endif

Result SvcQueryDebugProcessMemory(nn::svc::MemoryInfo* pBlockInfo, nn::svc::PageInfo* pPageInfo, nn::svc::Handle debug, uintptr_t addr)
{
    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KDebug* pDebug = handleTable.GetObject<KDebug>(debug);

    CHECK_KDEBUG(pDebug);
    KScopedAutoObject<KDebug> autoCloser(pDebug);

    return pDebug->QueryInfo(pBlockInfo, pPageInfo, addr);
}

Result SvcReadDebugProcessMemory(uintptr_t bufAddr, nn::svc::Handle debug, uintptr_t addr, size_t size)
{
    if (size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(addr < addr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (!(bufAddr < bufAddr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KDebug* pDebug = handleTable.GetObject<KDebug>(debug);

    CHECK_KDEBUG(pDebug);
    KScopedAutoObject<KDebug> autoCloser(pDebug);

    return pDebug->ReadMemory(reinterpret_cast<void*>(bufAddr), addr, size);
}

Result SvcWriteDebugProcessMemory(nn::svc::Handle debug, uintptr_t bufAddr, uintptr_t addr, size_t size)
{
    if (!KTargetSystem::IsDevelopmentHardware())
    {
        return nn::svc::ResultNotImplemented();
    }

    if (size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(addr < addr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (!(bufAddr < bufAddr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KDebug* pDebug = handleTable.GetObject<KDebug>(debug);

    CHECK_KDEBUG(pDebug);
    KScopedAutoObject<KDebug> autoCloser(pDebug);

    return pDebug->WriteMemory( reinterpret_cast<const void*>(bufAddr), addr, size);
}

Result SvcGetProcessList(int32_t* pNumProcesses, KUserPointer<Bit64*> pProcessIds, int32_t arraySize)
{
    if (!(0 <= arraySize && arraySize <= static_cast<int32_t>(INT32_MAX / sizeof(Bit64))))
    {
        return nn::svc::ResultOutOfRange();
    }
    if (arraySize > 0)
    {
        if (!GetCurrentProcess().GetPageTable().IsInRange(KProcessAddress(GetUnsafePointer(pProcessIds)), arraySize * sizeof(Bit64)))
        {
            return nn::svc::ResultInvalidCurrentMemory();
        }
    }

    return KProcess::GetProcessList(pNumProcesses, pProcessIds, arraySize);
}

Result SvcGetThreadList(int32_t* pNumThreads, KUserPointer<Bit64*> pThreadIds, int32_t arraySize, nn::svc::Handle debug)
{
    if (!(0 <= arraySize && arraySize <= static_cast<int32_t>(INT32_MAX / sizeof(Bit64))))
    {
        return nn::svc::ResultOutOfRange();
    }
    if (arraySize > 0)
    {
        if (!GetCurrentProcess().GetPageTable().IsInRange(KProcessAddress(GetUnsafePointer(pThreadIds)), arraySize * sizeof(Bit64)))
        {
            return nn::svc::ResultInvalidCurrentMemory();
        }
    }

    if (debug == nn::svc::Handle(0))
    {
        return KThread::GetThreadList(pNumThreads, pThreadIds, arraySize);
    }
    else
    {
        KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
        KDebug* pDebug = handleTable.GetObject<KDebug>(debug);
        if (pDebug)
        {
            KScopedAutoObject<KDebug> autoCloser(pDebug);
            KProcess *pProcess = pDebug->GetProcess();
            if (pProcess)
            {
                KScopedAutoObject<KProcess> autoProcessCloser(pProcess);
                return pProcess->GetThreadList(pNumThreads, pThreadIds, arraySize);
            }
            else
            {
                return nn::svc::ResultProcessTerminated();
            }
        }
        else
        {
            KProcess* pProcess = handleTable.GetObject<KProcess>(debug);
            if (pProcess)
            {
                KScopedAutoObject<KProcess> autoCloser(pProcess);
                return pProcess->GetThreadList(pNumThreads, pThreadIds, arraySize);
            }
            else
            {
                return nn::svc::ResultInvalidHandle();
            }
        }
    }
}

Result SvcOutputDebugString(KUserPointer<const char*> textUser, size_t byteSize)
{
    if (byteSize == 0)
    {
        return ResultSuccess();
    }
    if (!GetCurrentProcess().GetPageTable().IsInRange(KProcessAddress(GetUnsafePointer(textUser)), byteSize))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    return KOutputUserString(textUser, byteSize);
}


void SvcBreak(nn::svc::BreakReason reason, uintptr_t pData, size_t byteSize)
{
    if (!(reason & nn::svc::BreakReason_NotificationOnlyFlag))
    {
#ifdef NN_KERN_FOR_DEVELOPMENT
        PrintBreak(reason);
#endif
    }

    if (GetCurrentProcess().IsDebuggerAttached())
    {
        Result result = KDebug::BreakWhenAttached(reason, pData, byteSize);
        if (result.IsSuccess())
        {
            KDebug::SetPreviousPc();
            return;
        }
    }

    if (reason & nn::svc::BreakReason_NotificationOnlyFlag)
    {
        return;
    }

    NN_KERN_FATAL_LOG("Break() called. %016lx\n", GetCurrentProcess().GetProgramId());

    NN_KERN_KTRACE_STOP();

    if (GetCurrentProcess().EnterJitDebug(nn::svc::DebugEvent_Exception, nn::svc::DebugException_UserBreak, KDebug::RetrivePc(GetCurrentThread()), reason, pData, byteSize))
    {
        KDebug::SetPreviousPc();
        NN_KERN_KTRACE_DELAYED_DUMP_WITH_ADDITIONAL_INFO();
        return;
    }

    NN_KERN_KTRACE_DUMP_WITH_ADDITIONAL_INFO();
    GetCurrentProcess().Exit();
}

Result SvcGetDebugThreadContext( KUserPointer<nn::svc::ThreadContext*> pContext, nn::svc::Handle debug, Bit64 threadId, Bit32 controlFlags )
{
    Bit32 allFlags = nn::svc::ContextFlag_General | nn::svc::ContextFlag_Control | nn::svc::ContextFlag_Fpu | nn::svc::ContextFlag_FpuControl;
    if ((controlFlags | allFlags) != allFlags)
    {
        return nn::svc::ResultInvalidEnum();
    }

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KDebug* pDebug = handleTable.GetObject<KDebug>(debug);

    CHECK_KDEBUG(pDebug);
    KScopedAutoObject<KDebug> autoCloser(pDebug);

    nn::svc::ThreadContext context = {};
    Result result = pDebug->GetThreadContext(&context, threadId, controlFlags);
    if (result.IsFailure())
    {
        return result;
    }

    result = pContext.CopyFrom(&context);
    if (result.IsFailure())
    {
        return nn::svc::ResultInvalidPointer();
    }

    return ResultSuccess();
}

Result SvcSetDebugThreadContext( nn::svc::Handle debug, Bit64 threadId, KUserPointer<const nn::svc::ThreadContext*> userContext, Bit32 controlFlags )
{
    if (!KTargetSystem::IsDevelopmentHardware())
    {
        return nn::svc::ResultNotImplemented();
    }

    Bit32 allFlags = nn::svc::ContextFlag_General | nn::svc::ContextFlag_Control | nn::svc::ContextFlag_Fpu | nn::svc::ContextFlag_FpuControl;
    if ((controlFlags | allFlags) != allFlags)
    {
        return nn::svc::ResultInvalidEnum();
    }

    nn::svc::ThreadContext context;
    Result result = userContext.CopyTo(&context);
    if (result.IsFailure())
    {
        return nn::svc::ResultInvalidPointer();
    }

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KDebug* pDebug = handleTable.GetObject<KDebug>(debug);

    CHECK_KDEBUG(pDebug);
    KScopedAutoObject<KDebug> autoCloser(pDebug);

    result = pDebug->SetThreadContext(context, threadId, controlFlags);

    return result;
}

Result SvcTerminateDebugProcess(nn::svc::Handle debug)
{
    if (!KTargetSystem::IsDevelopmentHardware())
    {
        return nn::svc::ResultNotImplemented();
    }

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KDebug* pDebug = handleTable.GetObject<KDebug>(debug);

    CHECK_KDEBUG(pDebug);
    KScopedAutoObject<KDebug> autoCloser(pDebug);

    return pDebug->TerminateProcess();
}

Result SvcSetHardwareBreakPoint(nn::svc::HardwareBreakPointRegisterName regNo, Bit64 control, Bit64 value)
{
    if (!KTargetSystem::IsDevelopmentHardware())
    {
        return nn::svc::ResultNotImplemented();
    }

    return KDebug::SetHardwareBreakPoint(regNo, control, value);
}

Result SvcGetDebugThreadParam(Bit64* pOut1, Bit32* pOut2, nn::svc::Handle debug, Bit64 threadId, nn::svc::DebugThreadParam param)
{
    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KDebug* pDebug = handleTable.GetObject<KDebug>(debug);

    CHECK_KDEBUG(pDebug);
    KScopedAutoObject<KDebug> autoCloser(pDebug);

    KThread* pThread = KThread::GetThreadById(threadId);
    if (!pThread)
    {
        return nn::svc::ResultInvalidThreadId();
    }
    KScopedAutoObject<KThread> autoCloserDebug(pThread);

    KProcess *pProcess = pDebug->GetProcess();
    if (!pProcess)
    {
        return nn::svc::ResultProcessTerminated();
    }
    KScopedAutoObject<KProcess> autoProcessCloser(pProcess);

    if (pProcess != &pThread->GetParent())
    {
        return nn::svc::ResultInvalidThreadId();
    }

    switch (param)
    {
    case nn::svc::DebugThreadParam_Priority:
        {
            *pOut2 = pThread->GetPriority();
        }
        break;
    case nn::svc::DebugThreadParam_State:
        {
            KThread::ThreadState state;
            bool isSuspendedUser;
            bool isSuspendedDebug;
            {
                KScopedSchedulingLock locker;
                isSuspendedUser = pThread->IsSuspendRequested(KThread::SuspendType_Thread);
                isSuspendedDebug = pThread->IsSuspendRequested(KThread::SuspendType_Debug);
                state = pThread->GetState();
            }
            *pOut1 = 0;
            if (isSuspendedUser)
            {
                *pOut1 |= nn::svc::ThreadSuspend_User;
            }
            if (isSuspendedDebug)
            {
                *pOut1 |= nn::svc::ThreadSuspend_Debug;
            }
            switch (state)
            {
            case KThread::ThreadState::STATE_INITIALIZED:
                {
                    *pOut2 = nn::svc::ThreadState::ThreadState_Initializing;
                }
                break;
            case KThread::ThreadState::STATE_RUNNABLE:
                {
                    *pOut2 = nn::svc::ThreadState::ThreadState_Runnable;
                }
                break;
            case KThread::ThreadState::STATE_WAIT:
                {
                    *pOut2 = nn::svc::ThreadState::ThreadState_Wait;
                }
                break;
            case KThread::ThreadState::STATE_TERMINATED:
                {
                    *pOut2 = nn::svc::ThreadState::ThreadState_Terminated;
                }
                break;
            default:
                {
                    return nn::svc::ResultInvalidState();
                }
                break;
            }
        }
        break;
    case nn::svc::DebugThreadParam_IdealProcessor:
        {
            *pOut2 = pThread->GetIdealProcessor();
        }
        break;
    case nn::svc::DebugThreadParam_CurrentCore:
        {
            *pOut2 = pThread->GetRunningProcessor();
        }
        break;
    case nn::svc::DebugThreadParam_AffinityMask:
        {
            *pOut1 = pThread->GetAffinityMask().GetAffinityMask();
        }
        break;
    default:
        return nn::svc::ResultInvalidEnum();
    }

    return ResultSuccess();
}

Result SvcGetDebugFutureThreadInfo(nn::svc::LastThreadContext* pContext, nn::Bit64* pThreadId, nn::svc::Handle debug, int64_t ns)
{
    Result result;
#ifdef NN_KERN_ENABLE_NX_PROFILER
    if (!KTargetSystem::IsDevelopmentHardware())
    {
        return nn::svc::ResultNoThread();
    }

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KDebug* pDebug = handleTable.GetObject<KDebug>(debug);

    CHECK_KDEBUG(pDebug);
    KScopedAutoObject<KDebug> autoCloser(pDebug);

    KWaitObject* pWaitObject = GetCurrentProcess().GetWaitObjectPointer();

    int64_t timeoutTick;
    if (ns > 0)
    {
        // 途中でオーバーフローして負になった場合、永久待ちになる。
        uint64_t tick = KHardwareTimer::GetTick();
        tick += nn::svc::Tick(TimeSpan::FromNanoSeconds(ns));
        tick += 2;
        timeoutTick = tick;
    }
    else
    {
        timeoutTick = ns;
    }

    result = pWaitObject->Synchronize(timeoutTick);
    if (result.IsSuccess())
    {
        result = pDebug->GetRunningThreadInfo(pContext, pThreadId);
    }
#else
    NN_UNUSED(pContext);
    NN_UNUSED(pThreadId);
    NN_UNUSED(debug);
    NN_UNUSED(ns);
    result = nn::svc::ResultNotImplemented();
#endif

    return result;
}
}

#if defined(NN_BUILD_CONFIG_CPU_SVC_32)
Result SvcDebugActiveProcess32( nn::svc::Handle* pOut, Bit64 targetProcessId )
{
    Result result = SvcDebugActiveProcess(pOut, targetProcessId);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcBreakDebugProcess32( nn::svc::Handle debug )
{
    Result result = SvcBreakDebugProcess(debug);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcContinueDebugEvent32( nn::svc::Handle debug, Bit32 flags, KUserPointer<const Bit64*> pThreadIds, int32_t arraySize)
{
    Result result = SvcContinueDebugEvent(debug, flags, pThreadIds, arraySize);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetDebugEvent32( KUserPointer<nn::svc::ilp32::DebugEventInfo*> pInfo, nn::svc::Handle debug )
{
    Result result = SvcGetDebugEvent(pInfo, debug);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcQueryDebugProcessMemory32( KUserPointer<nn::svc::ilp32::MemoryInfo*> pBlockInfo, nn::svc::PageInfo* pPageInfo, nn::svc::Handle process, uintptr_t addr )
{
    nn::svc::MemoryInfo blockInfo = {};

    Result result = SvcQueryDebugProcessMemory(&blockInfo, pPageInfo, process, addr);
    if (result.IsSuccess())
    {
        result = pBlockInfo.CopyFrom(&blockInfo);
    }

    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcReadDebugProcessMemory32( uintptr_t bufAddr, nn::svc::Handle debug, uintptr_t addr, size_t size )
{
    Result result = SvcReadDebugProcessMemory(bufAddr, debug, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcWriteDebugProcessMemory32( nn::svc::Handle debug, uintptr_t bufAddr, uintptr_t addr, size_t size )
{
    Result result = SvcWriteDebugProcessMemory(debug, bufAddr, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetProcessList32( int32_t* pNumProcesses, KUserPointer<Bit64*> pProcessIds, int32_t arraySize )
{
    Result result = SvcGetProcessList(pNumProcesses, pProcessIds, arraySize);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetThreadList32( int32_t* pNumThreads, KUserPointer<Bit64*> pThreadIds, int32_t arraySize, nn::svc::Handle debug )
{
    Result result = SvcGetThreadList(pNumThreads, pThreadIds, arraySize, debug);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcOutputDebugString32( KUserPointer<const char*> textAddr, size_t byteSize )
{
    Result result = SvcOutputDebugString(textAddr, byteSize);
    ClearSvcOutRegistersReturnResult();
    return result;
}
void SvcBreak32(nn::svc::BreakReason reason, uintptr_t pData, size_t byteSize )
{
    SvcBreak(reason, pData, byteSize);
    ClearSvcOutRegisters();
}
Result SvcGetDebugThreadContext32( KUserPointer<nn::svc::ThreadContext*> pContext, nn::svc::Handle debug, Bit64 threadId, Bit32 controlFlags )
{
    Result result = SvcGetDebugThreadContext(pContext, debug, threadId, controlFlags);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetDebugThreadContext32( nn::svc::Handle debug, Bit64 threadId, KUserPointer<const nn::svc::ThreadContext*> userContext, Bit32 controlFlags )
{
    Result result = SvcSetDebugThreadContext(debug, threadId, userContext, controlFlags);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcTerminateDebugProcess32( nn::svc::Handle debug )
{
    Result result = SvcTerminateDebugProcess(debug);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetHardwareBreakPoint32(nn::svc::HardwareBreakPointRegisterName regNo, Bit64 control, Bit64 value)
{
    Result result = SvcSetHardwareBreakPoint(regNo, control, value);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetDebugThreadParam32(Bit64* pOut1, Bit32* pOut2, nn::svc::Handle debug, Bit64 threadId, nn::svc::DebugThreadParam select)
{
    Result result = SvcGetDebugThreadParam(pOut1, pOut2, debug, threadId, select);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetDebugFutureThreadInfo32(nn::svc::ilp32::LastThreadContext* pContext, nn::Bit64* pThreadId, nn::svc::Handle debug, int64_t ns)
{
    Result result = SvcGetDebugFutureThreadInfo(pContext, pThreadId, debug, ns);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif // #if defined(NN_BUILD_CONFIG_CPU_SVC_32)

#if defined(NN_BUILD_CONFIG_CPU_SVC_64)
Result SvcDebugActiveProcess64( nn::svc::Handle* pOut, Bit64 targetProcessId )
{
    Result result = SvcDebugActiveProcess(pOut, targetProcessId);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcBreakDebugProcess64( nn::svc::Handle debug )
{
    Result result = SvcBreakDebugProcess(debug);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcContinueDebugEvent64( nn::svc::Handle debug, Bit32 flags, KUserPointer<const Bit64*> pThreadIds, int32_t arraySize)
{
    Result result = SvcContinueDebugEvent(debug, flags, pThreadIds, arraySize);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetDebugEvent64( KUserPointer<nn::svc::lp64::DebugEventInfo*> pInfo, nn::svc::Handle debug )
{
    Result result = SvcGetDebugEvent(pInfo, debug);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcQueryDebugProcessMemory64( KUserPointer<nn::svc::lp64::MemoryInfo*> pBlockInfo, nn::svc::PageInfo* pPageInfo, nn::svc::Handle process, uintptr_t addr )
{
    nn::svc::MemoryInfo blockInfo = {};

    Result result = SvcQueryDebugProcessMemory(&blockInfo, pPageInfo, process, addr);
    if (result.IsSuccess())
    {
        result = pBlockInfo.CopyFrom(&blockInfo);
    }

    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcReadDebugProcessMemory64( uintptr_t bufAddr, nn::svc::Handle debug, uintptr_t addr, size_t size )
{
    Result result = SvcReadDebugProcessMemory(bufAddr, debug, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcWriteDebugProcessMemory64( nn::svc::Handle debug, uintptr_t bufAddr, uintptr_t addr, size_t size )
{
    Result result = SvcWriteDebugProcessMemory(debug, bufAddr, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetProcessList64( int32_t* pNumProcesses, KUserPointer<Bit64*> pProcessIds, int32_t arraySize )
{
    Result result = SvcGetProcessList(pNumProcesses, pProcessIds, arraySize);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetThreadList64( int32_t* pNumThreads, KUserPointer<Bit64*> pThreadIds, int32_t arraySize, nn::svc::Handle debug )
{
    Result result = SvcGetThreadList(pNumThreads, pThreadIds, arraySize, debug);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcOutputDebugString64( KUserPointer<const char*> textAddr, size_t byteSize )
{
    Result result = SvcOutputDebugString(textAddr, byteSize);
    ClearSvcOutRegistersReturnResult();
    return result;
}
void SvcBreak64(nn::svc::BreakReason reason, uintptr_t pData, size_t byteSize )
{
    SvcBreak(reason, pData, byteSize);
    ClearSvcOutRegisters();
}
Result SvcGetDebugThreadContext64( KUserPointer<nn::svc::ThreadContext*> pContext, nn::svc::Handle debug, Bit64 threadId, Bit32 controlFlags )
{
    Result result = SvcGetDebugThreadContext(pContext, debug, threadId, controlFlags);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetDebugThreadContext64( nn::svc::Handle debug, Bit64 threadId, KUserPointer<const nn::svc::ThreadContext*> userContext, Bit32 controlFlags )
{
    Result result = SvcSetDebugThreadContext(debug, threadId, userContext, controlFlags);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcTerminateDebugProcess64( nn::svc::Handle debug )
{
    Result result = SvcTerminateDebugProcess(debug);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetHardwareBreakPoint64(nn::svc::HardwareBreakPointRegisterName regNo, Bit64 control, Bit64 value)
{
    Result result = SvcSetHardwareBreakPoint(regNo, control, value);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetDebugThreadParam64(Bit64* pOut1, Bit32* pOut2, nn::svc::Handle debug, Bit64 threadId, nn::svc::DebugThreadParam select)
{
    Result result = SvcGetDebugThreadParam(pOut1, pOut2, debug, threadId, select);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetDebugFutureThreadInfo64(nn::svc::lp64::LastThreadContext* pContext, nn::Bit64* pThreadId, nn::svc::Handle debug, int64_t ns)
{
    Result result = SvcGetDebugFutureThreadInfo(pContext, pThreadId, debug, ns);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif // #if defined(NN_BUILD_CONFIG_CPU_SVC_64)


#if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
Result SvcDebugActiveProcess64From32( nn::svc::Handle* pOut, Bit64 targetProcessId )
{
    Result result = SvcDebugActiveProcess(pOut, targetProcessId);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcBreakDebugProcess64From32( nn::svc::Handle debug )
{
    Result result = SvcBreakDebugProcess(debug);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcContinueDebugEvent64From32( nn::svc::Handle debug, Bit32 flags, KUserPointer<const Bit64*> pThreadIds, int32_t arraySize)
{
    Result result = SvcContinueDebugEvent(debug, flags, pThreadIds, arraySize);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetDebugEvent64From32( KUserPointer<nn::svc::ilp32::DebugEventInfo*> pInfo, nn::svc::Handle debug )
{
    Result result = SvcGetDebugEvent(pInfo, debug);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcQueryDebugProcessMemory64From32( KUserPointer<nn::svc::ilp32::MemoryInfo*> pBlockInfo, nn::svc::PageInfo* pPageInfo, nn::svc::Handle process, uintptr_t addr )
{
    nn::svc::MemoryInfo blockInfo = {};
    nn::svc::PageInfo   pageInfo = {};

    Result result = SvcQueryDebugProcessMemory(&blockInfo, &pageInfo, process, addr);
    if (result.IsSuccess())
    {
        nn::svc::ilp32::MemoryInfo blockInfo32 = {};
        blockInfo32.baseAddress = blockInfo.baseAddress;
        blockInfo32.size        = blockInfo.size;
        blockInfo32.permission  = blockInfo.permission;
        blockInfo32.attribute   = blockInfo.attribute;
        blockInfo32.state       = blockInfo.state;
        blockInfo32.ipcCount    = blockInfo.ipcCount;
        blockInfo32.deviceCount = blockInfo.deviceCount;

        result = pBlockInfo.CopyFrom(&blockInfo32);
    }

    pPageInfo->flags        = pageInfo.flags;

    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcReadDebugProcessMemory64From32( uintptr_t bufAddr, nn::svc::Handle debug, uintptr_t addr, size_t size )
{
    Result result = SvcReadDebugProcessMemory(bufAddr, debug, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcWriteDebugProcessMemory64From32( nn::svc::Handle debug, uintptr_t bufAddr, uintptr_t addr, size_t size )
{
    Result result = SvcWriteDebugProcessMemory(debug, bufAddr, addr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetProcessList64From32( int32_t* pNumProcesses, KUserPointer<Bit64*> pProcessIds, int32_t arraySize )
{
    Result result = SvcGetProcessList(pNumProcesses, pProcessIds, arraySize);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetThreadList64From32( int32_t* pNumThreads, KUserPointer<Bit64*> pThreadIds, int32_t arraySize, nn::svc::Handle debug )
{
    Result result = SvcGetThreadList(pNumThreads, pThreadIds, arraySize, debug);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcOutputDebugString64From32( KUserPointer<const char*> textAddr, size_t byteSize )
{
    Result result = SvcOutputDebugString(textAddr, byteSize);
    ClearSvcOutRegistersReturnResult();
    return result;
}
void SvcBreak64From32(nn::svc::BreakReason reason, uintptr_t pData, size_t byteSize )
{
    SvcBreak(reason, pData, byteSize);
    ClearSvcOutRegisters();
}
Result SvcGetDebugThreadContext64From32( KUserPointer<nn::svc::ThreadContext*> pContext, nn::svc::Handle debug, Bit64 threadId, Bit32 controlFlags )
{
    Result result = SvcGetDebugThreadContext(pContext, debug, threadId, controlFlags);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetDebugThreadContext64From32( nn::svc::Handle debug, Bit64 threadId, KUserPointer<const nn::svc::ThreadContext*> userContext, Bit32 controlFlags )
{
    Result result = SvcSetDebugThreadContext(debug, threadId, userContext, controlFlags);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcTerminateDebugProcess64From32( nn::svc::Handle debug )
{
    Result result = SvcTerminateDebugProcess(debug);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcSetHardwareBreakPoint64From32(nn::svc::HardwareBreakPointRegisterName regNo, Bit64 control, Bit64 value)
{
    Result result = SvcSetHardwareBreakPoint(regNo, control, value);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetDebugThreadParam64From32(Bit64* pOut1, Bit32* pOut2, nn::svc::Handle debug, Bit64 threadId, nn::svc::DebugThreadParam select)
{
    Result result = SvcGetDebugThreadParam(pOut1, pOut2, debug, threadId, select);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcGetDebugFutureThreadInfo64From32(nn::svc::ilp32::LastThreadContext* pContext, nn::Bit64* pThreadId, nn::svc::Handle debug, int64_t ns)
{
    nn::svc::LastThreadContext context = {};
    Result result = SvcGetDebugFutureThreadInfo(&context, pThreadId, debug, ns);
    pContext->fp = context.fp;
    pContext->sp = context.sp;
    pContext->lr = context.lr;
    pContext->pc = context.pc;
    ClearSvcOutRegistersReturnResult();
    return result;
}

#endif // #if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
}}}
