﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include "../kern_Platform.h"
#include "../kern_KProcess.h"
#include "../kern_KHandleTable.h"
#include "../kern_KPageTableBase.h"
#include "kern_SvcHandlers.h"
#include "kern_SvcValueCheck.h"

namespace nn { namespace kern { namespace svc {
namespace
{
Result SvcMapProcessMemory(uintptr_t toAddr, nn::svc::Handle process, uint64_t fromAddr, size_t size)
{
    if (fromAddr != static_cast<uintptr_t>(fromAddr))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if ((toAddr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((fromAddr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(toAddr < toAddr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (!(fromAddr < fromAddr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KProcess* pCallerProcess = GetCurrentProcessPointer();
    KHandleTable& handleTable = pCallerProcess->GetHandleTable();
    KProcess* pTargetProcess = handleTable.GetObject<KProcess>(process);

    if (!pTargetProcess)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KProcess> autoCloser(pTargetProcess);

    KProcessPageTable& callerPageTable = pCallerProcess->GetPageTable();
    KProcessPageTable& targetPageTable  = pTargetProcess->GetPageTable();

    if (!targetPageTable.IsInRange(fromAddr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    if (!callerPageTable.IsInRange(toAddr, size, KMemoryState_SharedCode))
    {
        return nn::svc::ResultInvalidRegion();
    }

    KScopedPageGroup pg(callerPageTable.GetBlockInfoManager());

    Result result = targetPageTable.MakePageGroupAndOpen(&pg.GetPageGroup(), fromAddr, size / NN_KERN_FINEST_PAGE_SIZE,
            KMemoryState_FlagsMapProcess, KMemoryState_FlagsMapProcess,
            0, 0,
            KMemoryAttribute_All, KMemoryAttribute_None);
    if (result.IsFailure())
    {
        return result;
    }

    result = callerPageTable.MapPageGroup(toAddr, pg.GetPageGroup(), KMemoryState_SharedCode, KMemoryPermission_UserReadWrite);

    pg.GetPageGroup().Close();

    return result;
}

Result SvcUnmapProcessMemory(uintptr_t toAddr, nn::svc::Handle process, uint64_t fromAddr, size_t size)
{
    if (fromAddr != static_cast<uintptr_t>(fromAddr))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if ((toAddr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((fromAddr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(toAddr < toAddr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (!(fromAddr < fromAddr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KProcess* pCallerProcess = GetCurrentProcessPointer();
    KHandleTable& handleTable = pCallerProcess->GetHandleTable();
    KProcess* pTargetProcess = handleTable.GetObject<KProcess>(process);

    if (!pTargetProcess)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KProcess> autoCloser(pTargetProcess);

    KProcessPageTable& callerPageTable = pCallerProcess->GetPageTable();
    KProcessPageTable& targetPageTable  = pTargetProcess->GetPageTable();

    if (!targetPageTable.IsInRange(fromAddr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    if (!callerPageTable.IsInRange(toAddr, size, KMemoryState_SharedCode))
    {
        return nn::svc::ResultInvalidRegion();
    }

    KScopedPageGroup pg(callerPageTable.GetBlockInfoManager());

    Result result = targetPageTable.MakePageGroupAndOpen(&pg.GetPageGroup(), fromAddr, size / NN_KERN_FINEST_PAGE_SIZE, KMemoryState_FlagsMapProcess, KMemoryState_FlagsMapProcess, 0, 0, KMemoryAttribute_All, 0);
    if (result.IsSuccess())
    {
        result = callerPageTable.UnmapPageGroup(toAddr, pg.GetPageGroup(), KMemoryState_SharedCode);

        pg.GetPageGroup().Close();
    }

    return result;
}

Result SvcMapProcessCodeMemory(nn::svc::Handle process, uint64_t toAddr, uint64_t fromAddr, uint64_t size)
{
    if (toAddr != static_cast<uintptr_t>(toAddr))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (fromAddr != static_cast<uintptr_t>(fromAddr))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (size != static_cast<size_t>(size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if ((toAddr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((fromAddr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(toAddr < toAddr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (!(fromAddr < fromAddr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KProcess* pProcess = handleTable.GetObject<KProcess>(process);

    if (!pProcess)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KProcess> autoCloser(pProcess);

    KProcessPageTable& pageTable  = pProcess->GetPageTable();

    if (!pageTable.IsInRange(fromAddr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    if (!pageTable.IsInRange(toAddr, size, KMemoryState_AliasCode))
    {
        return nn::svc::ResultInvalidRegion();
    }

    return pageTable.MapCodeMemory(toAddr, fromAddr, size);
}

Result SvcUnmapProcessCodeMemory(nn::svc::Handle process, uint64_t toAddr, uint64_t fromAddr, uint64_t size)
{
    if (toAddr != static_cast<uintptr_t>(toAddr))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (fromAddr != static_cast<uintptr_t>(fromAddr))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (size != static_cast<size_t>(size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if ((toAddr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((fromAddr % NN_KERN_FINEST_PAGE_SIZE) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((size % NN_KERN_FINEST_PAGE_SIZE) != 0 || size == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (!(toAddr < toAddr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (!(fromAddr < fromAddr + size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    KHandleTable& handleTable = GetCurrentProcess().GetHandleTable();
    KProcess* pProcess = handleTable.GetObject<KProcess>(process);

    if (!pProcess)
    {
        return nn::svc::ResultInvalidHandle();
    }

    KScopedAutoObject<KProcess> autoCloser(pProcess);

    KProcessPageTable& pageTable  = pProcess->GetPageTable();

    if (!pageTable.IsInRange(fromAddr, size))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }

    if (!pageTable.IsInRange(toAddr, size, KMemoryState_AliasCode))
    {
        return nn::svc::ResultInvalidRegion();
    }

    return pageTable.UnmapCodeMemory(toAddr, fromAddr, size);
}
}

#if defined(NN_BUILD_CONFIG_CPU_SVC_32)
Result SvcMapProcessMemory32(uintptr_t toAddr, nn::svc::Handle process, uint64_t fromAddr, size_t size)
{
    Result result = SvcMapProcessMemory(toAddr, process, fromAddr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcUnmapProcessMemory32(uintptr_t toAddr, nn::svc::Handle process, uint64_t fromAddr, size_t size)
{
    Result result = SvcUnmapProcessMemory(toAddr, process, fromAddr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcMapProcessCodeMemory32(nn::svc::Handle process, uint64_t toAddr, uint64_t fromAddr, uint64_t size)
{
    Result result = SvcMapProcessCodeMemory(process, toAddr, fromAddr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcUnmapProcessCodeMemory32(nn::svc::Handle process, uint64_t toAddr, uint64_t fromAddr, uint64_t size)
{
    Result result = SvcUnmapProcessCodeMemory(process, toAddr, fromAddr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64)
Result SvcMapProcessMemory64(uintptr_t toAddr, nn::svc::Handle process, uint64_t fromAddr, size_t size)
{
    Result result = SvcMapProcessMemory(toAddr, process, fromAddr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcUnmapProcessMemory64(uintptr_t toAddr, nn::svc::Handle process, uint64_t fromAddr, size_t size)
{
    Result result = SvcUnmapProcessMemory(toAddr, process, fromAddr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcMapProcessCodeMemory64(nn::svc::Handle process, uint64_t toAddr, uint64_t fromAddr, uint64_t size)
{
    Result result = SvcMapProcessCodeMemory(process, toAddr, fromAddr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcUnmapProcessCodeMemory64(nn::svc::Handle process, uint64_t toAddr, uint64_t fromAddr, uint64_t size)
{
    Result result = SvcUnmapProcessCodeMemory(process, toAddr, fromAddr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
Result SvcMapProcessMemory64From32(uintptr_t toAddr, nn::svc::Handle process, uint64_t fromAddr, size_t size)
{
    Result result = SvcMapProcessMemory(toAddr, process, fromAddr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcUnmapProcessMemory64From32(uintptr_t toAddr, nn::svc::Handle process, uint64_t fromAddr, size_t size)
{
    Result result = SvcUnmapProcessMemory(toAddr, process, fromAddr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcMapProcessCodeMemory64From32(nn::svc::Handle process, uint64_t toAddr, uint64_t fromAddr, uint64_t size)
{
    Result result = SvcMapProcessCodeMemory(process, toAddr, fromAddr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
Result SvcUnmapProcessCodeMemory64From32(nn::svc::Handle process, uint64_t toAddr, uint64_t fromAddr, uint64_t size)
{
    Result result = SvcUnmapProcessCodeMemory(process, toAddr, fromAddr, size);
    ClearSvcOutRegistersReturnResult();
    return result;
}
#endif // #if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)

}}}
