﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include "../kern_Platform.h"
#include "../kern_Kernel.h"
#include "../kern_KProcess.h"
#include "kern_MemoryMap.h"
#include "kern_SvcHandlers.h"

namespace nn { namespace kern { namespace svc {
namespace {
bool IsKernelAddress(uintptr_t addr)
{
    return (NN_KERN_V_ADDR_KERNEL <= addr && addr < NN_KERN_V_ADDR_KERNEL_END);
}

Result SvcSignalToAddress(uintptr_t addr, nn::svc::SignalType type, int32_t value, int32_t num)
{
    if (NN_UNLIKELY(IsKernelAddress(addr)))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (addr % sizeof(int32_t) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    switch (type)
    {
    case nn::svc::SignalType_SignalOnly:
    case nn::svc::SignalType_IncrementIfEqual:
    case nn::svc::SignalType_UpdateByCountIfEqual:
        break;
    default:
        return nn::svc::ResultInvalidEnum();
    }

    return GetCurrentProcess().SignalAddressArbiter(addr, type, value, num);
}

Result SvcWaitForAddress(uintptr_t addr, nn::svc::ArbitrationType type, int32_t value, int64_t ns)
{
    if (NN_UNLIKELY(IsKernelAddress(addr)))
    {
        return nn::svc::ResultInvalidCurrentMemory();
    }
    if (addr % sizeof(int32_t) != 0)
    {
        return nn::svc::ResultInvalidAddress();
    }
    switch (type)
    {
    case nn::svc::ArbitrationType_WaitIfLessThan:
    case nn::svc::ArbitrationType_DecrementAndWaitIfLessThan:
    case nn::svc::ArbitrationType_WaitIfEqual:
        break;
    default:
        return nn::svc::ResultInvalidEnum();
    }

    int64_t timeout;
    if (ns > 0)
    {
        nn::svc::Tick offset(TimeSpan::FromNanoSeconds(ns));
        if (NN_LIKELY(offset > 0))
        {
            timeout = KHardwareTimer::GetTick() + offset + 2;
            if (NN_UNLIKELY(timeout <= 0))
            {
                timeout = INT64_MAX;
            }
        }
        else
        {
            timeout = INT64_MAX;
        }
    }
    else
    {
        timeout = ns;
    }

    return GetCurrentProcess().WaitAddressArbiter(addr, type, value, timeout);
}
}

#if defined(NN_BUILD_CONFIG_CPU_SVC_32)
Result SvcSignalToAddress32(uintptr_t addr, nn::svc::SignalType type, int32_t value, int32_t num)
{
    return SvcSignalToAddress(addr, type, value, num);
}
Result SvcWaitForAddress32(uintptr_t addr, nn::svc::ArbitrationType type, int32_t value, int64_t ns)
{
    return SvcWaitForAddress(addr, type, value, ns);
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64)
Result SvcSignalToAddress64(uintptr_t addr, nn::svc::SignalType type, int32_t value, int32_t num)
{
    return SvcSignalToAddress(addr, type, value, num);
}
Result SvcWaitForAddress64(uintptr_t addr, nn::svc::ArbitrationType type, int32_t value, int64_t ns)
{
    return SvcWaitForAddress(addr, type, value, ns);
}
#endif

#if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
Result SvcSignalToAddress64From32(uintptr_t addr, nn::svc::SignalType type, int32_t value, int32_t num)
{
    return SvcSignalToAddress(addr, type, value, num);
}
Result SvcWaitForAddress64From32(uintptr_t addr, nn::svc::ArbitrationType type, int32_t value, int64_t ns)
{
    return SvcWaitForAddress(addr, type, value, ns);
}
#endif // #if defined(NN_BUILD_CONFIG_CPU_SVC_64FROM32)
}}}
