﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include "kern_Assert.h"
#include "kern_Result.h"
#include "kern_KTimerTask.h"
#include "kern_KThread.h"

namespace nn { namespace kern {

class KWaitObject :
    public KTimerTask
{
private:
    KThread::QueueEntry m_Root;
    bool                m_TimerUsed;

public:
    //! コンストラクタ
    KWaitObject(): m_TimerUsed(false)
    {
        m_Root.pPrev = nullptr;
        m_Root.pNext = nullptr;
    }
    Result Synchronize(int64_t timeoutTick);
    void OnTimer();

private:
    void Enqueue(KThread* pAdd)
    {
        // Rootノードが未使用(両方NULL)か、使用中(両方NULLではない)であることを確認します
        NN_KERN_ASSERT( ((m_Root.pPrev == NULL) ^ (m_Root.pNext == NULL)) == 0 );

        // このオブジェクトはまだどのキューにも所属していないことを確認します
        NN_KERN_EQUAL_ASSERT( pAdd->m_QueueEntry.pPrev, NULL );
        NN_KERN_EQUAL_ASSERT( pAdd->m_QueueEntry.pNext, NULL );
        // キューに一つだけの場合は上のチェックをパスしてしまうので
        // 専用にチェックする
        NN_KERN_NOT_EQUAL_ASSERT( m_Root.pPrev, pAdd );
        NN_KERN_NOT_EQUAL_ASSERT( m_Root.pNext, pAdd );

        KThread* pLast = m_Root.pPrev;
        KThread::QueueEntry& lastE = (pLast != NULL) ? pLast->m_QueueEntry: m_Root;

        // キューの最後尾に対象のスレッドを足します
        pAdd->m_QueueEntry.pPrev = pLast;
        pAdd->m_QueueEntry.pNext = NULL;

        // 今回追加したスレッドを末尾のオブジェクトとします。
        lastE .pNext    = pAdd;
        m_Root.pPrev    = pAdd;

        // Rootノードが未使用(両方NULL)か、使用中(両方NULLではない)であることを確認します
        NN_KERN_ASSERT( ((m_Root.pPrev == NULL) ^ (m_Root.pNext == NULL)) == 0 );

#ifndef NN_SWITCH_DISABLE_ASSERT_WARNING
        KThread::QueueEntry* pe = &m_Root;
        // リンクを前方に辿ったら pAdd にたどり着くことを確認
        while( (pe->pNext != NULL) && (pe->pNext != pAdd) )
        {
            pe = &pe->pNext->m_QueueEntry;
        }
        NN_KERN_EQUAL_ASSERT( pe->pNext, pAdd );
        // さらにリンクを最後まで辿ったら、最後のエントリにたどり着くことを確認
        while( (pe->pNext != NULL) )
        {
            pe = &pe->pNext->m_QueueEntry;
        }
        NN_KERN_EQUAL_ASSERT( pe, &m_Root.pPrev->m_QueueEntry );

        // リンクを後方に辿ったら pAdd にたどり着くことを確認
        pe = &m_Root;
        while( (pe->pPrev != NULL) && (pe->pPrev != pAdd) )
        {
            pe = &pe->pPrev->m_QueueEntry;
        }
        NN_KERN_EQUAL_ASSERT( pe->pPrev, pAdd );
        // さらにリンクを最後まで辿ったら、最初のエントリにたどり着くことを確認
        while( (pe->pPrev != NULL) )
        {
            pe = &pe->pPrev->m_QueueEntry;
        }
        NN_KERN_EQUAL_ASSERT( pe, &m_Root.pNext->m_QueueEntry );
#endif
    }

    void Remove(KThread* pRemove)
    {
        NN_KERN_ASSERT( ((m_Root.pPrev == NULL) ^ (m_Root.pNext == NULL)) == 0 );
        // 安全のためスレッドキューに対象スレッドが入っているかどうかを確認します
#ifndef NN_SWITCH_DISABLE_ASSERT_WARNING
        KThread::QueueEntry* pe = &m_Root;
        // リンクを前方に辿ったら pRemove にたどり着くことを確認
        while( (pe->pNext != NULL) && (pe->pNext != pRemove) )
        {
            pe = &pe->pNext->m_QueueEntry;
        }
        NN_KERN_EQUAL_ASSERT( pe->pNext, pRemove );
        // さらにリンクを最後まで辿ったら、最後のエントリにたどり着くことを確認
        while( (pe->pNext != NULL) )
        {
            pe = &pe->pNext->m_QueueEntry;
        }
        NN_KERN_EQUAL_ASSERT( pe, &m_Root.pPrev->m_QueueEntry );

        // リンクを後方に辿ったら pRemove にたどり着くことを確認
        pe = &m_Root;
        while( (pe->pPrev != NULL) && (pe->pPrev != pRemove) )
        {
            pe = &pe->pPrev->m_QueueEntry;
        }
        NN_KERN_EQUAL_ASSERT( pe->pPrev, pRemove );
        // さらにリンクを最後まで辿ったら、最初のエントリにたどり着くことを確認
        while( (pe->pPrev != NULL) )
        {
            pe = &pe->pPrev->m_QueueEntry;
        }
        NN_KERN_EQUAL_ASSERT( pe, &m_Root.pNext->m_QueueEntry );
#endif
        // pRemove対象のオブジェクト前後を繋ぎなおします
        KThread* pPrev = pRemove->m_QueueEntry.pPrev;
        KThread* pNext = pRemove->m_QueueEntry.pNext;
        KThread::QueueEntry& prevE = (pPrev != NULL) ? pPrev->m_QueueEntry: m_Root;
        KThread::QueueEntry& nextE = (pNext != NULL) ? pNext->m_QueueEntry: m_Root;

        prevE.pNext = pNext;
        nextE.pPrev = pPrev;
#ifndef NN_SWITCH_DISABLE_ASSERT_WARNING
        pRemove->m_QueueEntry.pPrev = NULL;
        pRemove->m_QueueEntry.pNext = NULL;
#endif
        // Rootノードが未使用(両方NULL)か、使用中(両方NULLではない)であることを確認します
        NN_KERN_ASSERT( ((m_Root.pPrev == NULL) ^ (m_Root.pNext == NULL)) == 0 );
    }
};

}}

