﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Macro.h>
#include <nn/nn_BitTypes.h>
#include "kern_Assert.h"
#include "kern_KSimpleLock.h"
#include "kern_Platform.h"

#if defined NN_KERN_ENABLE_KTRACE
// #define NN_KERN_ENABLE_KTRACE_DUMP
// #define NN_KERN_KTRACE_ENABLE_DUMP_TRACE_COUNT_LIMIT
// #define NN_KERN_START_KTRACE_ON_STARTUP
#endif

namespace nn { namespace kern {

class KTrace
{
public:
    enum KTraceType
    {
        KTraceType_Sched = 1,
        KTraceType_ThreadStateChanged,
        KTraceType_SvcEnter0,
        KTraceType_SvcEnter1,
        KTraceType_SvcLeave0,
        KTraceType_SvcLeave1,
        KTraceType_Irq,
        KTraceType_IpcSend,
        KTraceType_IpcReceive,
        KTraceType_IpcReply,
        KTraceType_SchedChanged,
        KTraceType_Backtrace0,
        KTraceType_Backtrace1,
        KTraceType_CoreMigration,
        KTraceType_User0 = 48,
    };

    static void Initialize(KVirtualAddress addr, size_t size);
    static void BindStopInterruptHandler(int32_t coreNo);
    static void Start();
    static void Stop();
    static void Record(Bit8 type, Bit64 arg0, Bit64 arg1, Bit64 arg2, Bit64 arg3, Bit64 arg4, Bit64 arg5);
    static bool IsEnabled() { return g_Enable; }
    static void SetTypeFilter(Bit64 mask);
    static void SetProcessFilter(uint32_t index, Bit64 mask);
    static void Info();
    static void GetCurrentBacktrace(uintptr_t* pOut, size_t num);
    static void DumpOnce();
    static void DumpOnceWithAdditionalInfo();
    static void DelayedDumpOnceWithAdditionalInfo();

private:
    static void Dump();

    static bool g_Enable;
    static KSimpleLock g_Lock;
    static KVirtualAddress g_BufferAddress;
    static size_t g_Size;
    static Bit64 g_TypeMask;
    static Bit64 g_ProcessMask[2];
};

#if defined NN_KERN_ENABLE_KTRACE

#define NN_KERN_KTRACE_START()  nn::kern::KTrace::Start()
#define NN_KERN_KTRACE_STOP()   nn::kern::KTrace::Stop()

#define NN_KERN_KTRACE_RECORD(type, arg0, arg1, arg2, arg3, arg4, arg5)                         \
    do                                                                                          \
    {                                                                                           \
        if (nn::kern::KTrace::IsEnabled())                                                      \
        {                                                                                       \
            nn::kern::KTrace::Record((type), (arg0), (arg1), (arg2), (arg3), (arg4), (arg5));   \
        }                                                                                       \
    } while (NN_STATIC_CONDITION(0))

#define NN_KERN_KTRACE_SCHED(pNextThread) \
        NN_KERN_KTRACE_RECORD(nn::kern::KTrace::KTraceType_Sched, (pNextThread)->GetId(), 0, 0, 0, 0, 0)

#define NN_KERN_KTRACE_SVC_ENTER(svcNo, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7)                                         \
    do                                                                                                                          \
    {                                                                                                                           \
        if (nn::kern::KTrace::IsEnabled())                                                                                      \
        {                                                                                                                       \
            nn::kern::KTrace::Record(nn::kern::KTrace::KTraceType_SvcEnter0, (svcNo), (arg0), (arg1), (arg2), (arg3), (arg4));  \
            nn::kern::KTrace::Record(nn::kern::KTrace::KTraceType_SvcEnter1, (arg5), (arg6), (arg7), 0, 0, 0);                  \
        }                                                                                                                       \
    } while (NN_STATIC_CONDITION(0))

#define NN_KERN_KTRACE_SVC_LEAVE(svcNo, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7)                                         \
    do                                                                                                                          \
    {                                                                                                                           \
        if (nn::kern::KTrace::IsEnabled())                                                                                      \
        {                                                                                                                       \
            nn::kern::KTrace::Record(nn::kern::KTrace::KTraceType_SvcLeave0, (svcNo), (arg0), (arg1), (arg2), (arg3), (arg4));  \
            nn::kern::KTrace::Record(nn::kern::KTrace::KTraceType_SvcLeave1, (arg5), (arg6), (arg7), 0, 0, 0);                  \
        }                                                                                                                       \
    } while (NN_STATIC_CONDITION(0))

#define NN_KERN_KTRACE_IRQ(irq) \
        NN_KERN_KTRACE_RECORD(nn::kern::KTrace::KTraceType_Irq, (irq), 0, 0, 0, 0, 0)

#define NN_KERN_KTRACE_THREAD_STATE_CHANGED(pThread, oldState, newState) \
        NN_KERN_KTRACE_RECORD(nn::kern::KTrace::KTraceType_ThreadStateChanged, (pThread)->GetId(), (oldState), (newState), 0, 0, 0)

#define NN_KERN_KTRACE_IPC_SEND(sessionIndex, requestInex) \
        NN_KERN_KTRACE_RECORD(nn::kern::KTrace::KTraceType_IpcSend, (sessionIndex), (requestInex), 0, 0, 0, 0)

#define NN_KERN_KTRACE_IPC_REPLY(sessionIndex, requestInex, header0, header1) \
        NN_KERN_KTRACE_RECORD(nn::kern::KTrace::KTraceType_IpcReply, (sessionIndex), (requestInex), (header0), (header1), 0, 0)

#define NN_KERN_KTRACE_IPC_RECEIVE(sessionIndex, requestInex, header0, header1) \
        NN_KERN_KTRACE_RECORD(nn::kern::KTrace::KTraceType_IpcReceive, (sessionIndex), (requestInex), (header0), (header1), 0, 0)

#define NN_KERN_KTRACE_SCHE_CHANGED(coreid, prev, next) \
        NN_KERN_KTRACE_RECORD(nn::kern::KTrace::KTraceType_SchedChanged, coreid, (prev)->GetId(), (next)->GetId(),  0, 0, 0)

#define NN_KERN_KTRACE_CORE_MIGRATION(tid, prevCore, nextCore, type) \
        NN_KERN_KTRACE_RECORD(nn::kern::KTrace::KTraceType_CoreMigration, (tid), (prevCore), (nextCore), (type), 0, 0)

// バックトレースは重いので普段はとらない
#if 0
#define NN_KERN_KTRACE_BACKTRACE()                                                                                              \
    do                                                                                                                          \
    {                                                                                                                           \
        if (nn::kern::KTrace::IsEnabled())                                                                                      \
        {                                                                                                                       \
            const size_t num = 12;                                                                                              \
            uintptr_t buf[num];                                                                                                 \
            nn::kern::KTrace::GetCurrentBacktrace(buf, num);                                                                    \
            nn::kern::KTrace::Record(nn::kern::KTrace::KTraceType_Backtrace0, buf[0], buf[1], buf[2], buf[3], buf[4], buf[5]);  \
            nn::kern::KTrace::Record(nn::kern::KTrace::KTraceType_Backtrace1, buf[6], buf[7], buf[8], buf[9], buf[10], buf[11]);\
        }                                                                                                                       \
    } while (NN_STATIC_CONDITION(0))
#else
#define NN_KERN_KTRACE_BACKTRACE()
#endif

#else

#define NN_KERN_KTRACE_START()
#define NN_KERN_KTRACE_STOP()

#define NN_KERN_KTRACE_SCHED(nextThread)
#define NN_KERN_KTRACE_SVC_ENTER(svcNo, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7)
#define NN_KERN_KTRACE_SVC_LEAVE(svcNo, arg0, arg1, arg2, arg3, arg4, arg5, arg6, arg7)
#define NN_KERN_KTRACE_IRQ(irq)
#define NN_KERN_KTRACE_THREAD_STATE_CHANGED(pThread, oldState, newState)
#define NN_KERN_KTRACE_IPC_SEND(sessionIndex, requestInex)
#define NN_KERN_KTRACE_IPC_REPLY(sessionIndex, requestInex, header0, header1)
#define NN_KERN_KTRACE_IPC_RECEIVE(sessionIndex, requestInex, header0, header1)
#define NN_KERN_KTRACE_SCHE_CHANGED(coreid, prev, next)
#define NN_KERN_KTRACE_BACKTRACE()
#define NN_KERN_KTRACE_CORE_MIGRATION(tid, prevCore, nextCore, type)

#endif

#if defined(NN_KERN_ENABLE_KTRACE) && defined(NN_KERN_ENABLE_KTRACE_DUMP)
#define NN_KERN_KTRACE_DUMP() nn::kern::KTrace::DumpOnce()
#define NN_KERN_KTRACE_DUMP_WITH_ADDITIONAL_INFO() nn::kern::KTrace::DumpOnceWithAdditionalInfo()
#define NN_KERN_KTRACE_DELAYED_DUMP_WITH_ADDITIONAL_INFO() nn::kern::KTrace::DelayedDumpOnceWithAdditionalInfo()
#else
#define NN_KERN_KTRACE_DUMP()
#define NN_KERN_KTRACE_DUMP_WITH_ADDITIONAL_INFO()
#define NN_KERN_KTRACE_DELAYED_DUMP_WITH_ADDITIONAL_INFO()
#endif

}}

