﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "kern_KThread.h"
#include "kern_KSynchronizationObject.h"

namespace nn { namespace kern {

/*! @file

    @brief      KSynchronization クラスの定義です。

    このクラスは同期待ち処理のユーティリティクラスです。
    派生もメンバー変数も持たないため、インスタンスを持つ意味合いはないのですが
    Kernel::GetSynchronization() 経由でインスタンスを取得できるようになっています。

*/

class KSynchronization
{
    friend class KSynchronizationObject;
    friend class KTimeOutTask;

public:
    explicit KSynchronization();
    ~KSynchronization();

    void Initialize(){};

    /*!
        @brief      一つ以上の同期オブジェクトに対してシグナル待ちをします。

        オブジェクトがシグナル状態になるか、指定時間経過するまでブロッキングします。
        blockAllHandles パラメーターにより異なる意味合いを持ちます。
        スレッド pIndex は実行可能状態である必要があります。

        @param[out] pIndex          何番目の同期オブジェクトがシグナル状態になったかを取得
        @param[in]  pObjs           同期待ちオブジェクトポインタが並んだ配列
        @param[in]  num             同期待ちオブジェクトポインタ数
        @param[in]  blockAllHandles すべてのオブジェクトがシグナル状態になるまで待つ場合は true
        @param[in]  timeout         timeoutTick 起床時間(tick) 0以下の場合は起床しない

        @return     成功、タイムアウト、ハンドルが閉じられた等の終了要因を返します。

    */
    Result Wait(
            int32_t*                pIndex,
            KSynchronizationObject* objs[],
            const int32_t           num,
            int64_t                 timeout);

private:
    /*!
        @brief      同期オブジェクトのの待機状態が変化した際の通知

        同期オブジェクトがシグナル状態になるなど、
        状態が変化した際に呼ばれます。

        @param[in]      pObj    シグナル状態になった同期オブジェクト

    */
    void  OnAvailable(KSynchronizationObject *pobj);

    /*!
        @brief      同期オブジェクトが閉じられる際に呼ばれます。

        @param[in]      pObj    閉じられる対象の同期オブジェクト
        @param[in]      reason  閉じた理由

    */
    void  OnAbort    (KSynchronizationObject *pobj, Result reason);
};

}}

