﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_Kernel.h"
#include "kern_KPort.h"
#include "kern_KSynchronization.h"
#include "kern_KSession.h"
#include "kern_KLightSession.h"
#include "kern_KScheduler.h"
#include "kern_KScopedSchedulingLock.h"

namespace nn { namespace kern {

NN_AUTOOBJECT_DEFINE_TYPE_NAME(KServerPort);


/* ------------------------------------------------------------------------
        初期化/破棄
   ------------------------------------------------------------------------ */

void KServerPort::Initialize(KPort* p)
{
    NN_KERN_THIS_ASSERT();
    m_pParent = p;
}

void KServerPort::Cleanup()
{
    // サーバーセッションキューのクリーンナップ処理
    NN_KERN_ASSERT(IsLight() || m_LightSessionList.empty());
    NN_KERN_ASSERT(!IsLight() || m_SessionList.empty());

    for (;;)
    {
        KServerSession* pSession = nullptr;

        {
            KScopedSchedulingLock locker;
            while (!m_SessionList.empty())
            {
                pSession = &m_SessionList.front();
                m_SessionList.pop_front();
            }
        }
        if (pSession)
        {
            pSession->Close();
        }
        else
        {
            break;
        }
    }

    for (;;)
    {
        KLightServerSession* pSession = nullptr;

        {
            KScopedSchedulingLock locker;
            while (!m_LightSessionList.empty())
            {
                pSession = &m_LightSessionList.front();
                m_LightSessionList.pop_front();
            }
        }
        if (pSession)
        {
            pSession->Close();
        }
        else
        {
            break;
        }
    }
}

void KServerPort::Destroy()
{
    NN_KERN_THIS_ASSERT();
    m_pParent->OnServerClosed();

    Cleanup();

    m_pParent->Close();
}



/* ------------------------------------------------------------------------
        KSynchronizationObject
   ------------------------------------------------------------------------ */

bool KServerPort::IsSignaled() const
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());

    // サーバーセッションキューに空きはあるか?
    if (IsLight())
    {
        return !m_LightSessionList.empty();
    }
    else
    {
        return !m_SessionList.empty();
    }
}

/* ------------------------------------------------------------------------
        xxxx
   ------------------------------------------------------------------------ */

void KServerPort::EnqueueSession(KServerSession* pSession)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(!IsLight());
    KScopedSchedulingLock locker;

    // サーバーセッションキューに登録する
    m_SessionList.push_back(*pSession);

    if (m_SessionList.size() == 1)
    {
        NotifyAvailable();
    }
}

void KServerPort::EnqueueSession(KLightServerSession* pSession)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(IsLight());
    KScopedSchedulingLock locker;

    // サーバーセッションキューに登録する
    m_LightSessionList.push_back(*pSession);

    if (m_LightSessionList.size() == 1)
    {
        NotifyAvailable();
    }
}


KServerSession* KServerPort::AcceptSession()
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(!IsLight());
    KScopedSchedulingLock locker;

    if (m_SessionList.empty())
    {
        return nullptr;
    }

    KServerSession* pSession = &m_SessionList.front();
    m_SessionList.pop_front();
    return pSession;
}

KLightServerSession* KServerPort::AcceptLightSession()
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ASSERT(IsLight());
    KScopedSchedulingLock locker;

    if (m_LightSessionList.empty())
    {
        return nullptr;
    }

    KLightServerSession* pSession = &m_LightSessionList.front();
    m_LightSessionList.pop_front();
    return pSession;
}

bool KServerPort::IsLight() const
{
    return m_pParent->IsLight();
}

void KServerPort::OnClientClosed()
{
    NN_KERN_THIS_ASSERT();
}


}}
