﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include "kern_Assert.h"
#include <iterator>
#include <sys/tree.h>

namespace nn { namespace kern {

struct IntrusiveRbTreeNode
{
private:
    NN_DISALLOW_COPY(IntrusiveRbTreeNode);
public:
    IntrusiveRbTreeNode() {}
private:
    template <class, class, class> friend class IntrusiveRbTree;
    RB_ENTRY(IntrusiveRbTreeNode) entry;
};

/*
 * Compareについて
 * 以下のようなclassを用意する
 * == 0となるのは、find()のときのみ使用される。
 * insert()、remove()時に == 0となるのは許容しない。
 * struct CompareClass
 * {
 *     int operator() (const T& a, const T& b)
 *     {
 *         if (a < b) return -1;
 *         if (a > b) return 1;
 *         if (a == b) return 0;
 *     }
 * };
 */

template<class T, class NodeTraits, class Compare>
class IntrusiveRbTree
{
    NN_DISALLOW_COPY(IntrusiveRbTree);
public:
    class const_iterator;
    class iterator;

    typedef T value_type;
    typedef value_type* pointer;
    typedef const value_type* const_pointer;
    typedef value_type& reference;
    typedef const value_type& const_reference;
    typedef std::reverse_iterator<iterator> reverse_iterator;
    typedef std::reverse_iterator<const_iterator> const_reverse_iterator;
    typedef int size_type;
    typedef int difference_type;


    class const_iterator
    {
    public:
        typedef const IntrusiveRbTree::value_type value_type;
        typedef IntrusiveRbTree::difference_type difference_type;
        typedef value_type* pointer;
        typedef value_type& reference;
        typedef std::bidirectional_iterator_tag iterator_category;
        template <class, class, class> friend class IntrusiveRbTree;

    public:
        explicit const_iterator(pointer node)
            : m_Node(node)
        {
        }

        reference operator*() const
        {
            return *m_Node;
        }

        pointer operator->() const
        {
            return m_Node;
        }

        const_iterator& operator++()
        {
            m_Node = NodeTraits::GetItemPtr(Next(NodeTraits::GetNodePtr(*m_Node)));
            return *this;
        }

        const_iterator operator++(int)
        {
            const_iterator temporary(*this);
            ++(*this);
            return temporary;
        }

        const_iterator& operator--()
        {
            m_Node = NodeTraits::GetItemPtr(Prev(NodeTraits::GetNodePtr(*m_Node)));
            return *this;
        }

        const_iterator operator--(int)
        {
            const_iterator temporary(*this);
            --(*this);
            return temporary;
        }

        bool operator==(const const_iterator& target) const
        {
            return m_Node == target.m_Node;
        }

        bool operator!=(const const_iterator& target) const
        {
            return m_Node != target.m_Node;
        }

    private:
        pointer m_Node;
    };


    class iterator
    {
    public:
        typedef IntrusiveRbTree::value_type value_type;
        typedef IntrusiveRbTree::difference_type difference_type;
        typedef value_type* pointer;
        typedef value_type& reference;
        typedef std::bidirectional_iterator_tag iterator_category;
        template <class, class, class> friend class IntrusiveRbTree;

    public:
        explicit iterator(pointer node)
            : m_Node(node)
        {
        }

        NN_IMPLICIT operator const_iterator() const
        {
            return const_iterator(m_Node);
        }

        reference operator*() const
        {
            return *m_Node;
        }

        pointer operator->() const
        {
            return m_Node;
        }

        iterator& operator++()
        {
            m_Node = NodeTraits::GetItemPtr(Next(NodeTraits::GetNodePtr(*m_Node)));
            return *this;
        }

        iterator operator++(int)
        {
            iterator temporary(*this);
            ++(*this);
            return temporary;
        }

        iterator& operator--()
        {
            m_Node = NodeTraits::GetItemPtr(Prev(NodeTraits::GetNodePtr(*m_Node)));
            return *this;
        }

        iterator operator--(int)
        {
            iterator temporary(*this);
            --(*this);
            return temporary;
        }

        bool operator==(const iterator& target) const
        {
            return m_Node == target.m_Node;
        }

        bool operator!=(const iterator& target) const
        {
            return m_Node != target.m_Node;
        }

    private:
        pointer m_Node;
    };


public:
    IntrusiveRbTree() { InitializeTree(); }

    bool empty() const
    {
        return IsEmpty();
    }

    reference back()
    {
        return *NodeTraits::GetItemPtr(GetMax());
    }

    const_reference back() const
    {
        return *NodeTraits::GetItemPtr(GetMax());
    }

    reference front()
    {
        return *NodeTraits::GetItemPtr(GetMin());
    }

    const_reference front() const
    {
        return *NodeTraits::GetItemPtr(GetMin());
    }

    iterator begin()
    {
        return iterator(NodeTraits::GetItemPtr(GetMin()));
    }

    const_iterator begin() const
    {
        return const_iterator(NodeTraits::GetItemPtr(GetMin()));
    }

    iterator end()
    {
        return iterator(NodeTraits::GetItemPtr(static_cast<IntrusiveRbTreeNode*>(nullptr)));
    }

    const_iterator end() const
    {
        return const_iterator(NodeTraits::GetItemPtr(static_cast<const IntrusiveRbTreeNode*>(nullptr)));
    }

    iterator iterator_to(reference value)
    {
        return iterator(&value);
    }

    const_iterator iterator_to(const_reference value) const
    {
        return const_iterator(&value);
    }

    iterator insert(reference ref)
    {
        IntrusiveRbTreeNode* tmp = Insert(NodeTraits::GetNodePtr(ref));
        NN_UNUSED(tmp);
        NN_KERN_ASSERT(tmp == nullptr);
        return iterator(&ref);
    }

    iterator erase(iterator it)
    {
        IntrusiveRbTreeNode* tmp = NodeTraits::GetNodePtr(*it);
        IntrusiveRbTreeNode* next = Next(tmp);
        Remove(tmp);
        return iterator(NodeTraits::GetItemPtr(next));
    }

    iterator find(const_reference ref) const
    {
        return iterator(NodeTraits::GetItemPtr(Find(NodeTraits::GetNodePtr(ref))));
    }

    iterator nfind(const_reference ref) const
    {
        return iterator(NodeTraits::GetItemPtr(NFind(NodeTraits::GetNodePtr(ref))));
    }

private:
    RB_HEAD(IntrusiveRbTreeInternal, IntrusiveRbTreeNode);

    void InitializeTree()
    {
        RB_INIT(&m_Tree);
    }

    bool IsEmpty() const
    {
        return RB_EMPTY(&m_Tree);
    }

    static IntrusiveRbTreeNode* Next(IntrusiveRbTreeNode* elm)
    {
        return RB_NEXT(IntrusiveRbTreeInternal, nullptr, elm);
    }

    static IntrusiveRbTreeNode* Prev(IntrusiveRbTreeNode* elm)
    {
        return RB_PREV(IntrusiveRbTreeInternal, nullptr, elm);
    }

    static const IntrusiveRbTreeNode* Next(const IntrusiveRbTreeNode* elm)
    {
        return const_cast<const IntrusiveRbTreeNode*>(RB_NEXT(IntrusiveRbTreeInternal, nullptr, const_cast<IntrusiveRbTreeNode*>(elm)));
    }

    static const IntrusiveRbTreeNode* Prev(const IntrusiveRbTreeNode* elm)
    {
        return const_cast<const IntrusiveRbTreeNode*>(RB_PREV(IntrusiveRbTreeInternal, nullptr, const_cast<IntrusiveRbTreeNode*>(elm)));
    }

    IntrusiveRbTreeNode* Remove(IntrusiveRbTreeNode* elm)
    {
        return RB_REMOVE(IntrusiveRbTreeInternal, &m_Tree, elm);
    }

    IntrusiveRbTreeNode* Insert(IntrusiveRbTreeNode* elm)
    {
        return RB_INSERT(IntrusiveRbTreeInternal, &m_Tree, elm);
    }

    IntrusiveRbTreeNode* Find(const IntrusiveRbTreeNode* elm) const
    {
        return RB_FIND(IntrusiveRbTreeInternal, const_cast<IntrusiveRbTreeInternal*>(&m_Tree), const_cast<IntrusiveRbTreeNode*>(elm));
    }

    IntrusiveRbTreeNode* NFind(const IntrusiveRbTreeNode* elm) const
    {
        return RB_NFIND(IntrusiveRbTreeInternal, const_cast<IntrusiveRbTreeInternal*>(&m_Tree), const_cast<IntrusiveRbTreeNode*>(elm));
    }

    IntrusiveRbTreeNode* GetMin() const
    {
        return RB_MIN(IntrusiveRbTreeInternal, const_cast<IntrusiveRbTreeInternal*>(&m_Tree));
    }

    IntrusiveRbTreeNode* GetMax() const
    {
        return RB_MAX(IntrusiveRbTreeInternal, const_cast<IntrusiveRbTreeInternal*>(&m_Tree));
    }

    static int IntrusiveRbTreeCompate(const IntrusiveRbTreeNode* a, const IntrusiveRbTreeNode* b)
    {
        return Compare()(*NodeTraits::GetItemPtr(a), *NodeTraits::GetItemPtr(b));
    }

    RB_GENERATE_STATIC(IntrusiveRbTreeInternal, IntrusiveRbTreeNode, entry, IntrusiveRbTreeCompate);
    IntrusiveRbTreeInternal m_Tree;
};

template<class HolderT, IntrusiveRbTreeNode HolderT::*Member, class T = HolderT>
class IntrusiveRbTreeMemberNodeTraits
{
private:
    template <class, class, class> friend class IntrusiveRbTree;

    static IntrusiveRbTreeNode* GetNodePtr(T& ref) NN_NOEXCEPT
    {
        return &(ref.*Member);
    }

    static const IntrusiveRbTreeNode* GetNodePtr(const T& ref) NN_NOEXCEPT
    {
        return &(ref.*Member);
    }

    static T* GetItemPtr(IntrusiveRbTreeNode* node) NN_NOEXCEPT
    {
        return reinterpret_cast<T*>(reinterpret_cast<char*>(node) - GetOffset());
    }

    static const T* GetItemPtr(const IntrusiveRbTreeNode* node) NN_NOEXCEPT
    {
        return reinterpret_cast<const T*>(reinterpret_cast<const char*>(node) - GetOffset());
    }

    static uintptr_t GetOffset() NN_NOEXCEPT
    {
        return reinterpret_cast<uintptr_t>(&((reinterpret_cast<T*>(0))->*Member));
    }
};

template <class T, class Tag = T>
class IntrusiveRbTreeBaseNode : public IntrusiveRbTreeNode
{
};

template <class T, class Tag = T>
class IntrusiveRbTreeBaseNodeTraits
{
private:
    template <class, class, class> friend class IntrusiveRbTree;

    typedef IntrusiveRbTreeBaseNode<T, Tag> BaseNodeType;

    static IntrusiveRbTreeNode* GetNodePtr(T& ref)
    {
        return &static_cast<BaseNodeType&>(ref);
    }

    static const IntrusiveRbTreeNode* GetNodePtr(const T& ref)
    {
        return &static_cast<const BaseNodeType&>(ref);
    }

    static T* GetItemPtr(IntrusiveRbTreeNode* node)
    {
        return static_cast<T*>(static_cast<BaseNodeType*>(node));
    }

    static const T* GetItemPtr(const IntrusiveRbTreeNode* node)
    {
        return static_cast<const T*>(static_cast<const BaseNodeType*>(node));
    }

};

}}
