﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

#include "kern_Result.h"
#include "kern_Assert.h"
#include "kern_KMemoryBlock.h"
#include "kern_KPageGroup.h"
#include "kern_KTaggedAddress.h"

namespace nn { namespace kern {
namespace detail {
class PageTable
{
private:
    enum {
        PageTableSize = NN_KERN_FINEST_PAGE_SIZE,
    };
    char m_Buffer[PageTableSize] NN_IS_UNUSED_MEMBER;
};

class PageBuffer
{
private:
    char    m_Buffer[NN_KERN_FINEST_PAGE_SIZE] NN_IS_UNUSED_MEMBER;
};

template <class T>
class KResourceManager: protected KSlabAllocatorImpl
{
public:
    KResourceManager() :
        m_pNextAllocator(nullptr),
        m_Used(0),
        m_Peak(0),
        m_Num(0),
        m_Size(0)
    {}
    void Initialize(KResourceManager<PageBuffer>* pNextAllocator)
    {
        m_pNextAllocator = pNextAllocator;
        m_Base = pNextAllocator->GetBase();
        m_Size = pNextAllocator->GetSize();
    }
    void Initialize(KVirtualAddress buffer, size_t bufferSize)
    {
        m_Base = buffer;
        m_Num = bufferSize / sizeof(T);
        m_Size = m_Num * sizeof(T);

        uint8_t* p = GetTypedPointer<uint8_t>(m_Base + m_Size);

        for (size_t i = 0; i < m_Num; ++i)
        {
            p -= sizeof(T);
            KSlabAllocatorImpl::FreeToSlab(p);
        }
    }

    T* Allocate()
    {
        T *pBlock = reinterpret_cast<T*>(KSlabAllocatorImpl::AllocateFromSlab());

        if (NN_UNLIKELY(pBlock == nullptr))
        {
            if (m_pNextAllocator != nullptr)
            {
                pBlock = reinterpret_cast<T*>(m_pNextAllocator->Allocate());
                if (pBlock != nullptr)
                {
                    for (size_t i = 1; i < sizeof(PageBuffer) / sizeof(T); i++)
                    {
                        KSlabAllocatorImpl::FreeToSlab(pBlock + i);
                    }
                    m_Num += sizeof(PageBuffer) / sizeof(T);
                }
            }
        }
        if (NN_LIKELY(pBlock != nullptr))
        {
            size_t used = ++m_Used;
            size_t peak = m_Peak;
            while (peak < used)
            {
                size_t old = m_Peak.CompareAndSwap(peak, used);
                if (NN_LIKELY(old == peak))
                {
                    break;
                }
                peak = old;
            }
            new (reinterpret_cast<void*>(pBlock)) T();
        }
        return pBlock;
    }
    void Free(T* pBlock)
    {
        KSlabAllocatorImpl::FreeToSlab(pBlock);
        --m_Used;
    }

    bool InRange(KVirtualAddress addr) const
    {
        return m_Base <= addr && addr <= (m_Base + m_Size - 1);
    }
    KVirtualAddress GetBase() const
    {
        return m_Base;
    }
    size_t GetSize()  const
    {
        return m_Size;
    }
    size_t GetUsed()  const { return m_Used; }
    size_t GetPeak()  const { return m_Peak; }
    size_t GetNum()   const { return m_Num; }

private:
    KResourceManager<PageBuffer>* m_pNextAllocator;
    InterlockedVariable<size_t> m_Used;
    InterlockedVariable<size_t> m_Peak;
    InterlockedVariable<size_t> m_Num;
    KVirtualAddress             m_Base;
    size_t                      m_Size;

};
}

class KUnusedPageManager : public detail::KResourceManager<detail::PageBuffer> {};

class KMemoryBlockResourceManager : public detail::KResourceManager<KMemoryBlock> {};

class KBlockInfoManager : public detail::KResourceManager<KBlockInfo> {};

class KPageTableManager : public detail::KResourceManager<detail::PageTable>
{
public:
    static const size_t PageTableSize = sizeof(detail::PageTable);

public:
    KPageTableManager() : m_pRefCount(nullptr) {}

    static size_t CalcReferenceCounterBufferSize(size_t bufferSize)
    {
        return (bufferSize / sizeof(detail::PageTable)) * sizeof(uint16_t);
    }

    void Initialize(KUnusedPageManager* pNextAllocator, uint16_t* pRefCounter)
    {
        detail::KResourceManager<detail::PageTable>::Initialize(pNextAllocator);
        Initialize(pRefCounter);
    }

    void Initialize(KVirtualAddress buffer, size_t bufferSize, uint16_t* pRefCounter)
    {
        detail::KResourceManager<detail::PageTable>::Initialize(buffer, bufferSize);
        Initialize(pRefCounter);
    }

    void Open(KVirtualAddress addr, int refCount)
    {
        NN_KERN_ASSERT(InPageTableHeap(addr));

        *GetPageTableRefCounter(addr) += refCount;

        NN_KERN_ABORT_UNLESS(GetRefCount(addr) > 0);
    }

    bool Close(KVirtualAddress addr, int refCount)
    {
        NN_KERN_ASSERT(InPageTableHeap(addr));

        NN_KERN_ABORT_UNLESS(GetRefCount(addr) >= refCount);

        *GetPageTableRefCounter(addr) -= refCount;

        if (GetRefCount(addr) == 0)
        {
            return true;
        }
        else
        {
            return false;
        }
    }

    uint16_t GetRefCount(KVirtualAddress addr) const
    {
        NN_KERN_ASSERT(InPageTableHeap(addr));
        return *GetPageTableRefCounter(addr);
    }

    KVirtualAddress Allocate()
    {
        return KVirtualAddress(detail::KResourceManager<detail::PageTable>::Allocate());
    }

    void Free(KVirtualAddress addr)
    {
        detail::KResourceManager<detail::PageTable>::Free(GetTypedPointer<detail::PageTable>(addr));
    }

    bool InPageTableHeap(KVirtualAddress addr) const { return InRange(addr); }

private:
    uint16_t* GetPageTableRefCounter(KVirtualAddress addr) const
    {
        return &m_pRefCount[((addr - GetBase()) / sizeof(detail::PageTable))];
    }

    void Initialize(uint16_t* pRefCounter)
    {
        m_pRefCount = pRefCounter;
        for (size_t i = 0; i < GetSize() / sizeof(detail::PageTable); i++)
        {
            m_pRefCount[i] = 0;
        }
    }

private:
    uint16_t* m_pRefCount;
};

}}

