﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include "kern_Platform.h"
#include "kern_KObjectName.h"
#include "kern_KAutoObject.h"
#include <cstring>

namespace nn { namespace kern {

    KObjectName::NameList   KObjectName::s_List;
    KLightMutex             KObjectName::s_Mutex;

    /*!
        @brief      KObjectName に対して pObj と名前を対応付けます。

        @param[in]  pObj    登録するオブジェクト
        @param[in]  name    オブジェクトに対応する名前

    */
    void KObjectName::Initialize(KAutoObject* pObj, const char* name)
    {
        // CHECK: 名前の長さが規定以下か、診断することを推奨します。
        m_pObject = pObj;
        std::strncpy(m_Name, name, sizeof(m_Name));
        m_Name[sizeof(m_Name) - 1] = '\0';

        pObj->Open();
    }

    /*!
        @brief      名前が等しいかどうか判定します。

        CHECK: 高速化のため最初の 1 バイト比較 (name[0] == m_Name[0]) をご検討ください。

        @param[in]  name    名前

        @return     名前が等しければ true を返します。

    */
    bool KObjectName::Is(const char* name) const
    {
        return std::strncmp(m_Name, name, sizeof(m_Name)) == 0;
    }

    /*!
        @brief      名前と Object の対応をリストに登録します。

        この関数は this を持たない static 関数です。

        @param[in]  pObj    登録するオブジェクト
        @param[in]  name    オブジェクトに対応する名前

    */
    Result KObjectName::Name(KAutoObject* pObj, const char* name)
    {
        // スラブアローケーターから KObjectName を取得します
        KObjectName* pName = Allocate();
        if( pName == NULL )
        {
            return nn::svc::ResultOutOfResource();
        }

        // KObjectName に対して名前とオブジェクトを関連付けます。
        pName->Initialize(pObj, name);

        // 名前リストに追加します
        KAutoObject* pObject;
        {
            KScopedLightLock lock(&s_Mutex);

            pObject = FindImpl(name);
            if (!pObject)
            {
                s_List.push_back(*pName);
            }
        }

        if (pObject)
        {
            pObject->Close();
            pObj->Close();
            Free(pName);
            return nn::svc::ResultInvalidState();
        }

        return ResultSuccess();
    }

    /*!
        @brief      名前に対応する Object を取得します。

        この関数は this を持たない static 関数です。

        @param[in]  name    オブジェクトに対応する名前

        @return     名前に対応する Object のポインタを返します。

    */
    KAutoObject* KObjectName::Find(const char* name)
    {
        KScopedLightLock lock(&s_Mutex);
        return FindImpl(name);
    }
    KAutoObject* KObjectName::FindImpl(const char* name)
    {
        // 名前リストから名前引きします
        {
            NameList::const_iterator end = s_List.end();

            for(NameList::const_iterator i = s_List.begin(); i != end; ++i )
            {
                // 名前が等しい場合
                if( i->Is(name) )
                {
                    // 参照カウンタをインクリメントします
                    KAutoObject* pObj = i->GetObject();
                    pObj->Open();
                    return pObj;
                }
            }
        }

        return NULL;
    }

    Result KObjectName::Delete(KAutoObject* pObj, const char* name)
    {
        {
            KScopedLightLock lock(&s_Mutex);
            const NameList::iterator end = s_List.end();

            for(NameList::iterator i = s_List.begin(); i != end; ++i )
            {
                // 名前が等しい場合
                if( i->Is(name) && pObj == i->GetObject())
                {
                    pObj->Close();

                    KObjectName* pName = &*i;
                    s_List.erase(i);
                    Free(pName);
                    return ResultSuccess();
                }
            }
        }

        return nn::svc::ResultNotFound();
    }

}}


