﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include "kern_Assert.h"
#include "kern_KAutoObject.h"
#include "kern_KObjectContainer.h"
#include "kern_KSlabAllocator.h"
#include "kern_KLightMutex.h"

// TODO: ベースの Allocator の Traits を加えるべき。
// TODO: 第二テンプレート引数は AutoObject の継承クラスに限る、というロジックを入れる。

namespace nn {
    namespace kern {

class KProcess;

template <typename T, class U>
class KObjectAdaptor : public U
{
public:
    class ListAccessor
        : public KObjectContainer::ListAccessor
    {
    public:
        ListAccessor()
            : KObjectContainer::ListAccessor(&s_Container)
        {}
    };

public:
    virtual ~KObjectAdaptor(){}

    static T* Create()
    {
        T* pObj = KObjectAdaptor<T, U>::Allocate();
        if( pObj != NULL )
        {
            KAutoObject::Create(pObj);
        }
        return pObj;
    }

    static Result InitializeAllocator(void* pMem, size_t memSize)
    {
        s_SlabAllocator.Initialize(pMem, memSize);
        s_Container.Initialize();

        return ResultSuccess();
    }

    static size_t GetOwnedNum(KProcess* pOwner) { return s_Container.GetOwnedNum(pOwner); }

    static size_t GetObjSize()              { return s_SlabAllocator.GetObjSize(); }
    static size_t GetSlabSize()             { return s_SlabAllocator.GetSlabSize(); }
    static size_t GetNumRemain()            { return s_SlabAllocator.GetNumRemain(); }
    static size_t GetPeakNum()              { return s_SlabAllocator.GetPeakNum(); }
    static uintptr_t GetSlabAddr()          { return s_SlabAllocator.GetSlabAddr(); }

    static Result Register(T* pObj)
    {
        return s_Container.Register(pObj);
    }

    int32_t GetIndex() const { return s_SlabAllocator.GetIndex(static_cast<const T*>(this)); }

    virtual bool IsInitialized() const { return true; }

    virtual uintptr_t GetPostFinalizeArgument() const { return 0; }

private:
    // 上位クラスでオーバライドされないように private でシールド。
    virtual void Destroy()
    {
        bool isInitialized = IsInitialized();
        uintptr_t arg;
        if (isInitialized)
        {
            s_Container.Unregister(this);
            arg = this->GetPostFinalizeArgument();
            this->Finalize();
        }
        KObjectAdaptor<T, U>::Free(static_cast<T*>(this));
        if (isInitialized)
        {
            T::PostFinalize(arg);
        }
    }

private:
    static T* Allocate()
    {
        return s_SlabAllocator.AllocateFromSlab();
    }

    static void Free(T* pObj)
    {
        s_SlabAllocator.FreeToSlab(pObj);
    }

private:
    static KSlabAllocator<T>  s_SlabAllocator;
    static KObjectContainer   s_Container;
};


}}

