﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <cstring>
#include "kern_Platform.h"
#include "kern_Kernel.h"
#include "kern_Assert.h"
#include "kern_KHeapArrange.h"
#include "kern_KMemoryManager.h"
#include "kern_KResourceLimit.h"
#include "kern_SystemControl.h"
#include "kern_Utility.h"
#include <nn/svc/svc_Kernel.h>

namespace nn { namespace kern {
namespace {
#if   defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1)
// 4K, 8K, 4M
const size_t blockPageShift[]   = { 12, 13, 22 };
#elif defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK2) || \
defined(NN_BUILD_CONFIG_HARDWARE_NX)
// 4K, 64K, 2M, 4M, 32M, 512M, 1G
const size_t blockPageShift[]   = { 12, 16, 21, 22, 25, 29, 30 };
#elif defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
// 4K, 8K
const size_t blockPageShift[]   = { 12, 13 };
#elif defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
// 4K, 64K, 2M, 32M
const size_t blockPageShift[]   = { 12, 16, 21, 25 };
#endif

}

void KMemoryManager::KMemoryManagerImpl::Initialize(KVirtualAddress heapStart, size_t heapSize, KVirtualAddress heapBackStart, KVirtualAddress managementStart, size_t managementSize)
{
    m_pRefCount = GetTypedPointer<uint16_t>(managementStart);
    size_t refCountSize = RoundUp((heapSize / KMemoryManager::PageSize) * sizeof(uint16_t), KMemoryManager::PageSize);

    NN_KERN_ABORT_UNLESS((managementStart % KMemoryManager::PageSize) == 0);
    NN_KERN_ABORT_UNLESS((managementSize % KMemoryManager::PageSize) == 0);
    NN_KERN_ABORT_UNLESS(managementSize >= refCountSize);

    m_Heap.Initialize(
            heapStart, heapSize, heapBackStart,
            managementStart + refCountSize, managementSize - refCountSize,
            blockPageShift, NN_ARRAY_SIZE(blockPageShift));
}

size_t KMemoryManager::KMemoryManagerImpl::CalcManagementAreaSize(size_t heapSize)
{
    return RoundUp((heapSize / KMemoryManager::PageSize) * sizeof(uint16_t), KMemoryManager::PageSize) +
        KPageHeap::CalcManagementAreaSize(heapSize, blockPageShift, NN_ARRAY_SIZE(blockPageShift));
}

void KMemoryManager::Initialize(const KHeapArrange& arrange)
{
    NN_KERN_THIS_ASSERT();
    NN_KERN_ABORT_UNLESS((arrange.managementAddr % PageSize) == 0);
    NN_KERN_ABORT_UNLESS((arrange.managementSize % PageSize) == 0);
    NN_KERN_ABORT_UNLESS((arrange.secureSystemAddr % PageSize) == 0);
    NN_KERN_ABORT_UNLESS((arrange.secureSystemSize % PageSize) == 0);
    NN_KERN_ABORT_UNLESS((arrange.nonSecureSystemAddr % PageSize) == 0);
    NN_KERN_ABORT_UNLESS((arrange.nonSecureSystemSize % PageSize) == 0);
    NN_KERN_ABORT_UNLESS((arrange.appletAddr % PageSize) == 0);
    NN_KERN_ABORT_UNLESS((arrange.appletSize % PageSize) == 0);
    NN_KERN_ABORT_UNLESS((arrange.applicationAddr % PageSize) == 0);
    NN_KERN_ABORT_UNLESS((arrange.applicationSize % PageSize) == 0);

    std::memset(GetUntypedPointer(arrange.managementAddr), 0, arrange.managementSize);
    NN_KERN_ABORT_UNLESS(CalcManagementAreaSize(arrange.secureSystemSize)
            + CalcManagementAreaSize(arrange.nonSecureSystemSize)
            + CalcManagementAreaSize(arrange.appletSize)
            + CalcManagementAreaSize(arrange.applicationSize)
            <= arrange.managementSize);

    KVirtualAddress managementAddr = arrange.managementAddr;
    KVirtualAddress managementAddrEnd = arrange.managementAddr + arrange.managementSize;

    NN_KERN_ABORT_UNLESS(Kernel::GetSystemResourceLimit().TestLimit(nn::svc::LimitableResource_PhysicalMemoryMax, arrange.managementSize));

    if (arrange.secureSystemSize)
    {
        size_t managementSize = KMemoryManagerImpl::CalcManagementAreaSize(arrange.secureSystemSize);
        NN_KERN_ABORT_UNLESS(managementAddr + managementSize <= managementAddrEnd);

        m_MemoryManager[Region_SecureSystem].Initialize(arrange.secureSystemAddr, arrange.secureSystemSize, arrange.secureSystemBackAddr, managementAddr, managementSize);
        managementAddr += managementSize;
    }
    if (arrange.nonSecureSystemSize)
    {
        size_t managementSize = KMemoryManagerImpl::CalcManagementAreaSize(arrange.nonSecureSystemSize);
        NN_KERN_ABORT_UNLESS(managementAddr + managementSize <= managementAddrEnd);

        m_MemoryManager[Region_NonSecureSystem].Initialize(arrange.nonSecureSystemAddr, arrange.nonSecureSystemSize, arrange.nonSecureSystemBackAddr, managementAddr, managementSize);
        managementAddr += managementSize;
    }
    if (arrange.appletSize)
    {
        size_t managementSize = KMemoryManagerImpl::CalcManagementAreaSize(arrange.appletSize);
        NN_KERN_ABORT_UNLESS(managementAddr + managementSize <= managementAddrEnd);

        m_MemoryManager[Region_Applet].Initialize(arrange.appletAddr, arrange.appletSize, arrange.appletBackAddr, managementAddr, managementSize);
        managementAddr += managementSize;
    }
    if (arrange.applicationSize)
    {
        size_t managementSize = KMemoryManagerImpl::CalcManagementAreaSize(arrange.applicationSize);
        NN_KERN_ABORT_UNLESS(managementAddr + managementSize <= managementAddrEnd);

        m_MemoryManager[Region_Application].Initialize(arrange.applicationAddr, arrange.applicationSize, arrange.applicationBackAddr, managementAddr, managementSize);
        managementAddr += managementSize;
    }
}


}}
