﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include "kern_Platform.h"
#include "kern_InterlockedSelect.h"
#include "kern_KLockWithPriorityInheritance.h"
#include "kern_KScheduler.h"
#include "kern_MemoryCopySelect.h"
#include "kern_KScopedSchedulingLock.h"
#include "kern_KScopedSchedulingLockAndSleep.h"
#include "kern_Utility.h"
#include "kern_KProcess.h"
#include "kern_CPUSelect.h"
#include "kern_KLinkedList.h"

namespace nn { namespace kern {
namespace
{

inline bool ReadUserSpace(Bit32* p, KProcessAddress addr)
{
    return CopyMemoryFromUserOnly4(p, GetUntypedPointer(addr));
}

inline bool WriteUserSpace(KProcessAddress addr, const Bit32* p)
{
    return CopyMemoryToUserOnly4(GetUntypedPointer(addr), p);
}
inline bool AccessOk(KProcessAddress addr)
{
    return KCPU::CanAtomicAccess(addr);
}

struct UpdateLock
{
    Bit32 val;
    Bit32 old;
    bool operator()(Bit32* x)
    {
        this->old = *x;
        if (*x == 0)
        {
            *x = this->val;
        }
        else
        {
            *x |= nn::svc::Handle::WaitMask;
        }
        return true;
    }
};


KThread s_Comp;
}

Result KLockWithPriorityInheritance::WaitForAddress(nn::svc::Handle handle, KProcessAddress addr, Bit32 ownValue)
{
    Result ret;

    KThread* pCurrentThread = &GetCurrentThread();

    KThread* pToCloseThread = nullptr;
    {
        KScopedSchedulingLock locker;
        pCurrentThread->SetSyncedObject(nullptr, ResultSuccess());

        if (pCurrentThread->IsTerminateRequested())
        {
            return nn::svc::ResultTerminateRequested();
        }

        {
            Bit32 test;
            if (!ReadUserSpace(&test, addr))
            {
                return nn::svc::ResultInvalidCurrentMemory();
            }

            if (test != (static_cast<nnHandle>(handle).value | nn::svc::Handle::WaitMask))
            {
                return ResultSuccess();
            }

            // test == handle | かつ 無効なハンドルなら異常系
            KThread* pOwnerThread = GetCurrentProcess().GetHandleTable().GetObject<KThread>(handle);
            if (!pOwnerThread)
            {
                return nn::svc::ResultInvalidHandle();
            }
            pToCloseThread = pOwnerThread;

            // ロック獲得待ちにする
            pCurrentThread->SetAddressKey(addr, ownValue);
            pOwnerThread->AddWaiter(pCurrentThread);

            pCurrentThread->SetState(KThread::STATE_WAIT);
        }
    }
    NN_KERN_ASSERT(pToCloseThread != nullptr);
    pToCloseThread->Close();

    {
        // プロセス終了時などで、Signalされていないのに待ちが解除されることがある。
        KScopedSchedulingLock locker;
        KThread* pOwnerThread = pCurrentThread->GetLockOwner();
        if (pOwnerThread)
        {
            pOwnerThread->RemoveWaiter(pCurrentThread);
        }
    }

    // Signal側でアクセス違反の返却がある
    KSynchronizationObject* dummy;
    ret = pCurrentThread->GetWaitResult(&dummy);

    return ret;
}

Result KLockWithPriorityInheritance::SignalToAddress(KProcessAddress addr)
{
    KThread* pOwnerThread = &GetCurrentThread();

    {
        KScopedSchedulingLock locker;
        int32_t numWaiter;
        KThread* pNextOwnerThread = pOwnerThread->RemoveWaiterByKey(&numWaiter, addr);

        Bit32 nextValue = 0;
        if (pNextOwnerThread)
        {
            nextValue = pNextOwnerThread->GetAddressKeyValue();
            if (numWaiter > 1)
            {
                nextValue |= nn::svc::Handle::WaitMask;
            }

            pNextOwnerThread->SetSyncedObject(nullptr, ResultSuccess());
            pNextOwnerThread->WakeUp();
        }

        if (!WriteUserSpace(addr, &nextValue))
        {
            if (pNextOwnerThread)
            {
                pNextOwnerThread->SetSyncedObject(nullptr, nn::svc::ResultInvalidCurrentMemory());
            }

            return nn::svc::ResultInvalidCurrentMemory();
        }
    }

    return ResultSuccess();
}

Result KLockWithPriorityInheritance::WaitConditionVariable(KProcessAddress addr, uintptr_t cvKey, Bit32 ownValue, int64_t timeout)
{
    // WaitConditionVariable が呼ばれるのは lock 獲得中.
    KThread* pOwnerThread = &GetCurrentThread();
    KHardwareTimer* pTimer;
    {
        KScopedSchedulingLockAndSleep sleep(&pTimer, pOwnerThread, timeout);
        pOwnerThread->SetSyncedObject(nullptr, nn::svc::ResultTimeout());
        if (pOwnerThread->IsTerminateRequested())
        {
            sleep.Cancel();
            return nn::svc::ResultTerminateRequested();
        }

        {
            {
                // アンロック処理
                int32_t numWaiter;
                KThread* pNextOwnerThread = pOwnerThread->RemoveWaiterByKey(&numWaiter, addr);

                Bit32 nextValue = 0;
                if (pNextOwnerThread)
                {
                    nextValue = pNextOwnerThread->GetAddressKeyValue();
                    if (numWaiter > 1)
                    {
                        nextValue |= nn::svc::Handle::WaitMask;
                    }

                    pNextOwnerThread->SetSyncedObject(nullptr, ResultSuccess());
                    pNextOwnerThread->WakeUp();
                }

                if (!WriteUserSpace(addr, &nextValue))
                {
                    sleep.Cancel();
                    return nn::svc::ResultInvalidCurrentMemory();
                }
            }

            {
                // 条件変数待ち
                pOwnerThread->SetConditionVariable(this, addr, cvKey, ownValue);
                m_CvList.insert(*pOwnerThread);
            }
            if (timeout != 0)
            {
                pOwnerThread->SetState(KThread::STATE_WAIT);
            }
        }
    }

    // タイムアウトで待ち解除されても、ここでSignalされることがある。
    if (pTimer != nullptr)
    {
        // タイマーのキャンセル
        pTimer->CancelTask(pOwnerThread);
    }

    {
        // タイムアウト、プロセス終了時などで、Signalされていないのに待ちが解除されることがある。
        KScopedSchedulingLock locker;
        KThread* pOwner = pOwnerThread->GetLockOwner();
        if (pOwner)
        {
            pOwner->RemoveWaiter(pOwnerThread);
        }
        if (pOwnerThread->InWaitingForConditionVariable())
        {
            m_CvList.erase(m_CvList.iterator_to(*pOwnerThread));
            pOwnerThread->ClearConditionVariable();
        }
    }

    KSynchronizationObject* dummy = nullptr;
    Result ret = pOwnerThread->GetWaitResult(&dummy);

    return ret;
}

KThread* KLockWithPriorityInheritance::SignalConditionVariableImpl(KThread* pThread)
{
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());

    KProcessAddress addr = pThread->GetAddressKey();
    Bit32 ownValue = pThread->GetAddressKeyValue();
    KThread* pCloseThread = nullptr;
    Bit32 value;
    bool bAccess;

    {
        KDisableInterrupt di;
        bAccess = AccessOk(addr);

        if (NN_LIKELY(bAccess))
        {
            UpdateLock f;
            f.val = ownValue;
            GetTypedPointer<InterlockedVariable<Bit32>>(addr)->AtomicUpdateConditional(&f);
            value = f.old;
        }
    }

    if (NN_LIKELY(bAccess))
    {
        if (value == 0)
        {
            // 誰もロックしていなかった場合, pThread がロックして返る
            pThread->SetSyncedObject(nullptr, ResultSuccess());
            pThread->WakeUp();
        }
        else
        {
            // 誰かがロックしている場合
            KThread* pOwnerThread = GetCurrentProcess().GetHandleTable().GetObject<KThread>(nn::svc::Handle(value & ~nn::svc::Handle::WaitMask));
            if (NN_LIKELY(pOwnerThread != nullptr))
            {
                // ここを抜けると、pOwnerThread は Close() される。
                // pThread が待ち解除されるより先に pOwnerThread が終了する場合は
                // pOwnerThread の Finalize() でリストから外される。
                pCloseThread = pOwnerThread;
                // ロック獲得待ちにする
                // pThread の AddressKey, AddressKeyValue は設定済み
                pOwnerThread->AddWaiter(pThread);
            }
            else
            {
                pThread->SetSyncedObject(nullptr, nn::svc::ResultInvalidState());
                pThread->WakeUp();
            }
        }
    }
    else
    {
        pThread->SetSyncedObject(nullptr, nn::svc::ResultInvalidCurrentMemory());
        pThread->WakeUp();
    }
    return pCloseThread;
}

void KLockWithPriorityInheritance::SignalConditionVariable(uintptr_t cvKey, int32_t num)
{
    KLinkedList<KThread> threadList;

    // Closeを最適化する配列
    const int maxClose = 16;
    KThread* threadArray[maxClose];
    int numToClose = 0;

    int numWaiter = 0;
    {
        KScopedSchedulingLock locker;
        s_Comp.SetupForConditionVariableCompare(cvKey, -1);
        ConditionVariableList::iterator it = m_CvList.nfind(s_Comp);

        while ((it != m_CvList.end()) && (num <= 0 || numWaiter < num) && (it->GetConditionVariableKeyValue() == cvKey))
        {
            KThread* pTargetThread = &*it;
            KThread* pThread = SignalConditionVariableImpl(pTargetThread);
            if (pThread)
            {
                if (numToClose < maxClose)
                {
                    threadArray[numToClose++] = pThread;
                }
                else
                {
                    threadList.PushBack(pThread);
                }
            }
            it = m_CvList.erase(it);
            pTargetThread->ClearConditionVariable();
            ++numWaiter;
        }
    }

    for (int i = 0; i < numToClose; i++)
    {
        threadArray[i]->Close();
    }

    if (numWaiter > maxClose)
    {
        KLinkedList<KThread>::Iterator itr = threadList.GetBeginIter();
        while (itr != threadList.GetEndIter())
        {
            KThread* pThread = &*itr;
            pThread->Close();
            itr = threadList.Erase(itr);
        }
    }
}

}}
