﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include "kern_Platform.h"
#include "kern_InterlockedSelect.h"
#include "kern_Assert.h"
#include "kern_Kernel.h"
#include "kern_KLightMutex.h"
#include "kern_KThread.h"
#include "kern_KScheduler.h"
#include "kern_InterruptManagerSelect.h"
#include "kern_KScopedSchedulingLock.h"

namespace nn { namespace kern {

//! オブジェクトの診断
void KLightMutex::LockAssert()
{
    NN_KERN_THIS_ASSERT();

    NN_KERN_ASSERT( Kernel::GetState() != Kernel::STATE_NORMAL || KScheduler::CanSchedule() );

    // 自スレッドをこのミューテックスをロックしていないことを検証する
    NN_KERN_ASSERT( ! IsLockedByMe() );
}

//! ロックしているスレッドが存在する場合の追加処理
void KLightMutex::Lock2(uintptr_t ownerValue, uintptr_t currentValue)
{
    KThread* pCurrentThread = reinterpret_cast<KThread*>(currentValue);

    {
        KScopedSchedulingLock locker;

        if (NN_UNLIKELY(m_pHeldThread.Read() != ownerValue))
        {
            return;
        }
        if (NN_UNLIKELY(!GetCoreScheduler().IsActive()))
        {
            return;
        }

        // ロック獲得待ちにする
        KThread* pOwnerThread = reinterpret_cast<KThread*>(ownerValue & ~0x1ul);
        pCurrentThread->SetAddressKey(reinterpret_cast<uintptr_t>(&m_pHeldThread));
        pOwnerThread->AddWaiter(pCurrentThread);
        if (NN_LIKELY(pCurrentThread->GetState() == KThread::STATE_RUNNABLE))
        {
            pCurrentThread->SetState(KThread::STATE_WAIT);
        }
        if (pOwnerThread->IsSuspended())
        {
            pOwnerThread->Continue();
        }
    }

    {
        KScopedSchedulingLock locker;
        KThread* pOwnerThread = pCurrentThread->GetLockOwner();
        if (NN_UNLIKELY(pOwnerThread))
        {
            pOwnerThread->RemoveWaiter(pCurrentThread);
        }
    }
}

void KLightMutex::Unlock2(uintptr_t currentValue)
{
    KThread* pOwnerThread = reinterpret_cast<KThread*>(currentValue);

    {
        KScopedSchedulingLock locker;
        int32_t numWaiter;
        KThread* pNextOwnerThread = pOwnerThread->RemoveWaiterByKey(&numWaiter, reinterpret_cast<uintptr_t>(&m_pHeldThread));

        uintptr_t nextValue = 0;
        if (pNextOwnerThread)
        {
            nextValue = reinterpret_cast<uintptr_t>(pNextOwnerThread);
            if (numWaiter > 1)
            {
                nextValue |= 0x1;
            }

            // プロセス終了時などで、すでに起床済みの場合がある。
            if (NN_LIKELY(pNextOwnerThread->GetState() == KThread::STATE_WAIT))
            {
                pNextOwnerThread->SetState(KThread::STATE_RUNNABLE);
            }
        }

        if (pOwnerThread->IsSuspended())
        {
            pOwnerThread->TrySuspend();
        }

        m_pHeldThread.WriteNotAtomic(nextValue);
    }
}
}}

