﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include "kern_Result.h"
#include "kern_KInterruptEvent.h"
#include "kern_KInterruptHandler.h"

namespace nn { namespace kern {

/*!
    @brief      割り込みマネージャの基底クラスです。

*/
class KInterruptManagerBase
{
public:
    KInterruptManagerBase& operator=(const KInterruptManagerBase&) = delete;
    KInterruptManagerBase(const KInterruptManagerBase&) = delete;
    KInterruptManagerBase() = default;

    /*!
        @brief 初期化します。
    */
    void Initialize();

    /*!
        @brief     指定したコア、 IRQ 番号にイベントを割り当てます。

        @param[in] pEvent 割り込みハンドラ
        @param[in] irqNo IRQ 番号
        @param[in] coreNo コア
        @param[in] priority 優先度
        @param[in] isManualClear 割り込みフラグを手動でクリアするならば true
        @param[in] isLevel レベルトリガに設定するなら true,
                           エッジトリガに設定するなら false

        @return    関数の実行結果を返します。

    */
    Result  BindEvent     (KInterruptEvent* pEvent, int32_t irqNo, int32_t coreNo, int32_t priority, bool isManualClear, bool isLevel);

    /*!
        @brief     指定した IRQ 番号へのイベント割り当てを解除します。

        @param[in] pEvent 割り込みハンドラ
        @param[in] irqNo IRQ 番号

        @return    関数の実行結果を返します。

    */
    Result  UnbindEvent   (KInterruptEvent* pEvent, int32_t irqNo);

    /*!
        @brief     割り込みをクリアします。

        @param[in] pEvent 割り込みハンドラ
        @param[in] irqNo IRQ 番号

        @return    関数の実行結果を返します。

    */
    Result  ClearInterrupt(int32_t irqNo, int32_t coreNo);

    /*!
        @brief IRQ 番号の割り込みフラグをクリアします。

        @param[in] irqNo IRQ 番号

        @return    関数の実行結果を返します。

    */
    Result  ClearInterrupt(int32_t irqNo);

    /*!
        @brief     割り込みハンドラを割り当てます。

        @return    関数の実行結果を返します。

    */
    Result  BindHandler   (KInterruptHandler* pHandler, int32_t irqNo, int32_t coreNo, int32_t priority, bool isManualClear, bool isLevel);

    /*!
        @brief     指定したコア、IRQ 番号に割り当てられているハンドラを解除します。

        @param[in] irqNo IRQ 番号
        @param[in] coreNo コア

        @return    関数の実行結果を返します。

    */
    Result  UnbindHandler (int32_t irqNo, int32_t coreNo);
};

}}

