﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include <nn/nn_BitTypes.h>
#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_Kernel.h"
#include "kern_KCodeMemory.h"
#include "kern_KProcess.h"
#include "kern_PageTableSelect.h"
#include "kern_KMemoryManager.h"

namespace nn { namespace kern {

NN_AUTOOBJECT_DEFINE_TYPE_NAME(KCodeMemory);

KCodeMemory::KCodeMemory()
: m_pOwner(nullptr)
, m_Addr(0)
, m_IsInitialized(false)
, m_IsOwnerMapped(false)
, m_IsMapped(false)
{
    NN_KERN_THIS_ASSERT();
}

KCodeMemory::~KCodeMemory()
{
    NN_KERN_THIS_ASSERT();
}

Result KCodeMemory::Initialize(KProcessAddress addr, size_t size)
{
    NN_KERN_THIS_ASSERT();

    Result result;
    m_pOwner = GetCurrentProcessPointer();
    KProcessPageTable& pageTable = m_pOwner->GetPageTable();

    m_Memory.Initialize(pageTable.GetBlockInfoManager());
    result = pageTable.LockForCodeMemory(&m_Memory, addr, size);
    if (result.IsFailure())
    {
        m_Memory.Finalize();
        return result;
    }

    // fill & cache clean
    //  aarch32: 0xffffffff: UNDEFINED
    //  aarch64: 0xffffffff: undefined
    for (KPageGroup::BlockInfoList::const_iterator it = m_Memory.GetBlockBeginIter(); it != m_Memory.GetBlockEndIter(); it++)
    {
        ::std::memset(GetUntypedPointer(it->GetBlockAddr()), 0xff, it->GetNumPages() * NN_KERN_FINEST_PAGE_SIZE);
        KCPU::StoreDataCache(GetUntypedPointer(it->GetBlockAddr()), it->GetNumPages() * NN_KERN_FINEST_PAGE_SIZE);
    }

    m_pOwner->Open();
    m_Addr = addr;
    m_IsInitialized = true;
    m_IsOwnerMapped = false;
    m_IsMapped = false;

    return ResultSuccess();
}

void KCodeMemory::Finalize()
{
    NN_KERN_THIS_ASSERT();

    size_t size = m_Memory.GetTotalNumPages() * NN_KERN_FINEST_PAGE_SIZE;

    if (!m_IsMapped && !m_IsOwnerMapped)
    {
        KProcessPageTable& pageTable = m_pOwner->GetPageTable();
        Result result = pageTable.UnlockForCodeMemory(m_Addr, size, m_Memory);
        NN_KERN_ABORT_IF_FAILED(result);
    }

    m_Memory.Close();
    m_Memory.Finalize();
    m_pOwner->Close();

    KObjectAdaptor<KCodeMemory, KAutoObjectWithList>::Finalize();
}

Result KCodeMemory::Map(KProcessAddress addr, size_t size)
{
    NN_KERN_THIS_ASSERT();

    if (m_Memory.GetTotalNumPages() != KMemoryManager::SizeToPage(size))
    {
        return nn::svc::ResultInvalidSize();
    }

    KScopedLightLock locker(&m_Lock);

    if (m_IsMapped)
    {
        return nn::svc::ResultInvalidState();
    }

    Result result = GetCurrentProcess().GetPageTable().MapPageGroup(addr, m_Memory, KMemoryState_CodeOut, KMemoryPermission_UserReadWrite);
    if (result.IsSuccess())
    {
        m_IsMapped = true;
    }
    return result;
}

Result KCodeMemory::Unmap(KProcessAddress addr, size_t size)
{
    NN_KERN_THIS_ASSERT();

    if (m_Memory.GetTotalNumPages() != KMemoryManager::SizeToPage(size))
    {
        return nn::svc::ResultInvalidSize();
    }

    KScopedLightLock locker(&m_Lock);

    Result result = GetCurrentProcess().GetPageTable().UnmapPageGroup(addr, m_Memory, KMemoryState_CodeOut);
    if (result.IsSuccess())
    {
        NN_KERN_ASSERT(m_IsMapped);
        m_IsMapped = false;
    }
    return result;
}

Result KCodeMemory::MapToOwner(KProcessAddress addr, size_t size, nn::svc::MemoryPermission permission)
{
    NN_KERN_THIS_ASSERT();

    if (m_Memory.GetTotalNumPages() != KMemoryManager::SizeToPage(size))
    {
        return nn::svc::ResultInvalidSize();
    }

    KScopedLightLock locker(&m_Lock);

    if (m_IsOwnerMapped)
    {
        return nn::svc::ResultInvalidState();
    }

    KMemoryPermission kpermission;
    switch (permission)
    {
    case nn::svc::MemoryPermission_Read:
        {
            kpermission = KMemoryPermission_UserRead;
        }
        break;
    case nn::svc::MemoryPermission_ReadExecute:
        {
            kpermission = KMemoryPermission_UserReadExecute;
        }
        break;
    default:
        {
            NN_KERN_ABORT();
        }
        break;
    }

    Result result = m_pOwner->GetPageTable().MapPageGroup(addr, m_Memory, KMemoryState_GeneratedCode, kpermission);
    if (result.IsSuccess())
    {
        m_IsOwnerMapped = true;
    }
    return result;
}

Result KCodeMemory::UnmapFromOwner(KProcessAddress addr, size_t size)
{
    NN_KERN_THIS_ASSERT();

    if (m_Memory.GetTotalNumPages() != KMemoryManager::SizeToPage(size))
    {
        return nn::svc::ResultInvalidSize();
    }

    KScopedLightLock locker(&m_Lock);

    Result result = m_pOwner->GetPageTable().UnmapPageGroup(addr, m_Memory, KMemoryState_GeneratedCode);
    if (result.IsSuccess())
    {
        NN_KERN_ASSERT(m_IsOwnerMapped);
        m_IsOwnerMapped = false;
    }
    return result;
}


}}

