﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_Kernel.h"
#include "kern_KClientSession.h"
#include "kern_KThread.h"
#include "kern_KProcess.h"
#include "kern_KScheduler.h"
#include "kern_KSession.h"
#include "kern_KSessionRequest.h"
#include "kern_KScopedSchedulingLock.h"
#include <nn/svc/ipc/svc_SessionMessage.h>

namespace nn { namespace kern {

NN_AUTOOBJECT_DEFINE_TYPE_NAME(KClientSession);

void KClientSession::Destroy()
{
    NN_KERN_THIS_ASSERT();
    //! クライアントセッションが閉じたことをセッションに通知します
    m_pParent->OnClientClosed();

    m_pParent->Close();
    // このタイミングで既に this が破棄されている可能性があるので
    // 以降、this にアクセスしてはいけない。
}

//! ペアとなるサーバーセッションが閉じたことの通知
void KClientSession::OnServerClosed()
{
    NN_KERN_THIS_ASSERT();
}

//! サーバーセッションに対して同期リクエストする
nn::Result KClientSession::SendSyncRequest(uintptr_t addr, size_t size)
{
    NN_KERN_THIS_ASSERT();
    Result result;
    KSessionRequest* pRequest = KSessionRequest::Create();
    if (!pRequest)
    {
        return nn::svc::ResultOutOfResource();
    }

    KScopedAutoObject<KSessionRequest> autoCloser(pRequest);

    pRequest->Initialize(nullptr, addr, size);

    {
        KScopedSchedulingLock locker;

        // 待ち行列に加える前の処理
        GetCurrentThread().SetSyncedObject(NULL, ResultSuccess());

        // セッションにリクエストを通知することで、
        // サーバセッションにリクエストを登録してもらう
        result = m_pParent->OnRequest(pRequest);
        // 寝る
    }

    if ( result.IsFailure() )
    {
        return result;
    }

    KSynchronizationObject* pSyncedObj;
    result = GetCurrentThread().GetWaitResult(&pSyncedObj);

    return result;
}

nn::Result KClientSession::SendAsyncRequest(KWritableEvent* pEvent, uintptr_t addr, size_t size)
{
    NN_KERN_THIS_ASSERT();
    Result result;
    KSessionRequest* pRequest = KSessionRequest::Create();
    if (!pRequest)
    {
        return nn::svc::ResultOutOfResource();
    }

    KScopedAutoObject<KSessionRequest> autoCloser(pRequest);
    pRequest->Initialize(pEvent, addr, size);

    {
        KScopedSchedulingLock locker;

        result = m_pParent->OnRequest(pRequest);
    }

    return result;
}
}}
