﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>

#include "kern_Platform.h"
#include "kern_Assert.h"
#include "kern_KCapability.h"
#include "kern_KMemoryManager.h"
#include "kern_KPageTableBase.h"
#include "kern_Kernel.h"

// CHECK: 4, 128等、値がハードコーディングされた箇所の定数化を推奨します。

namespace nn { namespace kern {

/*!
    @brief     ケイパビリティクラスを初期化します
    @param[in]    flags     ケイパビリティリスト
    @param[in]    numFlags       ケイパビリティリストの要素数
    @param[in]    pageTable          当該プロセスのページテーブル
    @return       ステータスを返します。

*/
// 初期プログラム
Result KCapability::Initialize(const Bit32 flags[], int32_t num, KProcessPageTable* pPageTable)
{
    m_OtherCapability.storage= 0;
    m_IntentedKernelVersion.storage= 0;
    m_HandleTableSize = 0;
    std::memset(m_SvcCapability, 0, sizeof(m_SvcCapability));
    std::memset(m_InterruptCapability, 0, sizeof(m_InterruptCapability));
    m_ProgramType = 0;

    m_CoreMask = ((1ull << KCPU::NUM_CORE) - 1);
    m_PriorityMask = 0xFFFFFFFFFFFFFFFFull;
    m_OtherCapability.Set<OtherCapability::PermitDebug>(false);
    m_OtherCapability.Set<OtherCapability::DebugProcess>(false);
    m_IntentedKernelVersion.Set<VersionFlags::Major>(NN_SVC_VERSION_MAJOR);
    m_IntentedKernelVersion.Set<VersionFlags::Minor>(NN_SVC_VERSION_MINOR);

    return SetCapabilities(flags, num, pPageTable);
}

// ローダから起動されたプログラム
Result KCapability::Initialize(svc::KUserPointer<const Bit32*> flags, int32_t num, KProcessPageTable* pPageTable)
{
    m_OtherCapability.storage= 0;
    m_IntentedKernelVersion.storage= 0;
    m_HandleTableSize = 0;
    std::memset(m_SvcCapability, 0, sizeof(m_SvcCapability));
    std::memset(m_InterruptCapability, 0, sizeof(m_InterruptCapability));
    m_ProgramType = 0;

    m_CoreMask = 0;
    m_PriorityMask = 0;

    return SetCapabilities(flags, num, pPageTable);
}


/*!
    @brief     与えられたケイパビリティリストをもとにケイパビリティを更新します
    @param[in]    flags     ケイパビリティリスト
    @param[in]    numFlags       ケイパビリティリストの要素数
    @param[in]    pPageTable      当該プロセスのページテーブル
    @return       ステータスを返します。

*/
Result KCapability::SetCapability(Bit32* pSetFlags, Bit32* pSetSvcIndex,
            const nn::util::BitPack32 flag, KProcessPageTable* pPageTable)
{
    Result result = ResultSuccess();

    Bit32 flagType = GetFlagType(flag);

    if (flagType == 0)
    {
        return nn::svc::ResultInvalidArgument();
    }
    if (flagType == PaddingType)
    {
        return ResultSuccess();
    }

    int flagTypeLength = 32 - __builtin_clz(flagType);
    if ((*pSetFlags & InitializeOnceFlags) & (1u << flagTypeLength))
    {
        return nn::svc::ResultInvalidCombination();
    }
    *pSetFlags |= (1u << flagTypeLength);

    switch (flagType)
    {
    case PriorityCoreNoType:
        {
            result = SetPriorityCoreNoCapability(flag);
        }
        break;
    case VersionType:
        {
            result = SetVersionCapability(flag);
        }
        break;
    case MiscType:
        {
            result = SetMiscCapability(flag);
        }
        break;
    case InterruptType:
        {
            result = SetInterruptCapability(flag);
        }
        break;
    case SvcType:
        {
            result = SetSvcCapability(pSetSvcIndex, flag);
        }
        break;
    case HandleType:
        {
            if (flag.Get<HandleFlags::Reserved>() != 0)
            {
                result = nn::svc::ResultReservedIsUsed();
            }
            else
            {
                m_HandleTableSize = flag.Get<HandleFlags::MaxHandles>();
            }
        }
        break;
    case OtherType:
        {
            result = SetOtherCapability(flag);
        }
        break;
    case MapIoType:
        {
            result = MapIoPages(flag, pPageTable);
        }
        break;
    default:
        {
            result = nn::svc::ResultInvalidArgument();
        }
        break;
    }

    return result;
}

Result KCapability::SetCapabilities(const Bit32 flags[], int32_t num, KProcessPageTable* pPageTable)
{
    Bit32 setFlags = 0;
    Bit32 setSvcIndex = 0;

    for (int i = 0; i < num; i++)
    {
        Result result = ResultSuccess();
        nn::util::BitPack32 flag = { flags[i] };
        if (GetFlagType(flag) == MapPhysicalType)
        {
            i++;
            if (i < num)
            {
                nn::util::BitPack32 size = { flags[i] };
                if (GetFlagType(size) == MapPhysicalType)
                {
                    result = MapPages(flag, size, pPageTable);
                }
                else
                {
                    result = nn::svc::ResultInvalidCombination();
                }
            }
            else
            {
                result = nn::svc::ResultInvalidCombination();
            }
        }
        else
        {
            result = SetCapability(&setFlags, &setSvcIndex, flag, pPageTable);
        }
        if (result.IsFailure())
        {
            return result;
        }
    }

    return ResultSuccess();
}

Result KCapability::SetCapabilities(svc::KUserPointer<const Bit32*> flags, int32_t num, KProcessPageTable* pPageTable)
{
    Bit32 setFlags = 0;
    Bit32 setSvcIndex = 0;

    for (int i = 0; i < num; i++)
    {
        Result result = ResultSuccess();

        Bit32 x0;
        result = flags.CopyArrayEntryTo(&x0, i);
        if (result.IsFailure())
        {
            return result;
        }
        nn::util::BitPack32 flag = { x0 };

        if (GetFlagType(flag) == MapPhysicalType)
        {
            i++;
            if (i < num)
            {
                Bit32 x1;
                result = flags.CopyArrayEntryTo(&x1, i);
                if (result.IsFailure())
                {
                    return result;
                }
                nn::util::BitPack32 size = { x1 };

                if (GetFlagType(size) == MapPhysicalType)
                {
                    result = MapPages(flag, size, pPageTable);
                }
                else
                {
                    result = nn::svc::ResultInvalidCombination();
                }
            }
            else
            {
                result = nn::svc::ResultInvalidCombination();
            }
        }
        else
        {
            result = SetCapability(&setFlags, &setSvcIndex, flag, pPageTable);
        }
        if (result.IsFailure())
        {
            return result;
        }
    }

    return ResultSuccess();
}

Result KCapability::SetVersionCapability(nn::util::BitPack32 flag)
{
    if (m_IntentedKernelVersion.Get<VersionFlags::Major>() != 0)
    {
        // resultは暫定
        return nn::svc::ResultInvalidArgument();
    }

    m_IntentedKernelVersion = flag;

    if (m_IntentedKernelVersion.Get<VersionFlags::Major>() == 0)
    {
        // resultは暫定
        return nn::svc::ResultInvalidArgument();
    }

    return ResultSuccess();
}

Result KCapability::SetPriorityCoreNoCapability(nn::util::BitPack32 flag)
{
    if (m_PriorityMask != 0 || m_CoreMask != 0)
    {
        // resultは暫定
        return nn::svc::ResultInvalidArgument();
    }

    uint32_t coreNoMin = flag.Get<PriorityCoreNoFlags::CoreNoMin>();
    uint32_t coreNoMax = flag.Get<PriorityCoreNoFlags::CoreNoMax>();
    uint32_t priorityMin = flag.Get<PriorityCoreNoFlags::PriorityHighest>();
    uint32_t priorityMax = flag.Get<PriorityCoreNoFlags::PriorityLowest>();

    if (coreNoMin > coreNoMax || priorityMin > priorityMax)
    {
        return nn::svc::ResultInvalidCombination();
    }
    if (KCPU::NUM_CORE <= coreNoMax)
    {
        return nn::svc::ResultInvalidCoreNumber();
    }

    NN_KERN_ASSERT(coreNoMax < 64);
    NN_KERN_ASSERT(priorityMax < 64);

    for (uint32_t i = coreNoMin; i <= coreNoMax; i++)
    {
        m_CoreMask |= (1ull << i);
    }
    for (uint32_t i = priorityMin; i <= priorityMax; i++)
    {
        m_PriorityMask |= (1ull << i);
    }

    NN_KERN_ASSERT((m_CoreMask | ((1ull << KCPU::NUM_CORE) - 1)) == ((1ull << KCPU::NUM_CORE) - 1));
    NN_KERN_ASSERT((m_PriorityMask | 0xFFFFFFFFFFFFFFFFull) == 0xFFFFFFFFFFFFFFFFull);

    if (m_PriorityMask == 0 || m_CoreMask == 0)
    {
        // resultは暫定
        return nn::svc::ResultInvalidArgument();
    }

    return ResultSuccess();
}

Result KCapability::SetMiscCapability(nn::util::BitPack32 flag)
{
    if (flag.Get<MiscFlags::Reserved>() != 0)
    {
        return nn::svc::ResultReservedIsUsed();
    }
    m_ProgramType = flag.Get<MiscFlags::ProgramType>();

    return ResultSuccess();
}

Result KCapability::SetInterruptCapability(nn::util::BitPack32 flag)
{
    uint32_t intr[2] =
    {
        flag.Get<InterruptFlags::Intr0>(),
        flag.Get<InterruptFlags::Intr1>()
    };

    for (size_t i = 0; i < sizeof(intr) / sizeof(*intr); i++)
    {
        if (intr[i] != PaddingInterruptNumber)
        {
            if (!Kernel::GetInterruptManager().IsDefinedInterrupt(intr[i]) || !PermitInterrupt(intr[i]))
            {
                return nn::svc::ResultOutOfRange();
            }
        }
    }

    return ResultSuccess();
}

// システムコールの許可フラグの設定
Result KCapability::SetSvcCapability(Bit32* pSetSvcIndex, nn::util::BitPack32 flag)
{
    uint32_t flags = flag.Get<SvcFlags::Flags>();
    uint32_t index = flag.Get<SvcFlags::Index>();

    if (*pSetSvcIndex & (1u << index))
    {
        return nn::svc::ResultInvalidCombination();
    }
    *pSetSvcIndex |= (1u << index);

    for (int i = 0; i < 24; i++)
    {
        uint32_t svcNo = index * 24 + i;
        if (flags & (1u << i))
        {
            if (!PermitSvc(svcNo))
            {
                return nn::svc::ResultOutOfRange();
            }
        }
    }
    return ResultSuccess();
}

// その他のケイパビリティの設定
Result KCapability::SetOtherCapability(nn::util::BitPack32 flag)
{
    if (flag.Get<OtherFlags::Reserved>() != 0)
    {
        return nn::svc::ResultReservedIsUsed();
    }

    m_OtherCapability.Set<OtherCapability::PermitDebug>(flag.Get<OtherFlags::PermitDebug>());
    m_OtherCapability.Set<OtherCapability::DebugProcess>(flag.Get<OtherFlags::DebugProcess>());
    return ResultSuccess();
}

// static マップ
Result KCapability::MapPages(nn::util::BitPack32 start, nn::util::BitPack32 size, KProcessPageTable* pPageTable)
{
    if (size.Get<MapPhysicalSizeFlags::Reserved>() != 0)
    {
        return nn::svc::ResultOutOfRange();
    }

    uint64_t physAddr = (static_cast<uint64_t>(start.Get<MapPhysicalStartFlags::Pfn>()) << 12);
    if (physAddr != GetAsInteger(KPhysicalAddress(physAddr)))
    {
        return nn::svc::ResultInvalidAddress();
    }

    size_t numPages  = size.Get<MapPhysicalSizeFlags::Pages>();
    KMemoryPermission permission = (start.Get<MapPhysicalStartFlags::ReadOnly>() ? KMemoryPermission_UserRead: KMemoryPermission_UserReadWrite);
    bool isIo = !size.Get<MapPhysicalSizeFlags::Normal>();
    if (numPages == 0)
    {
        return nn::svc::ResultInvalidSize();
    }
    if (physAddr >= physAddr + numPages * NN_KERN_FINEST_PAGE_SIZE)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((physAddr + numPages * NN_KERN_FINEST_PAGE_SIZE - 1) & ~PhysicalMapMask)
    {
        return nn::svc::ResultInvalidAddress();
    }

    if (isIo)
    {
        return pPageTable->MapIo(physAddr, numPages * NN_KERN_FINEST_PAGE_SIZE, permission);
    }
    else
    {
        return pPageTable->MapStatic(physAddr, numPages * NN_KERN_FINEST_PAGE_SIZE, permission);
    }
}

// IO ページのマップ
Result KCapability::MapIoPages(nn::util::BitPack32 flag, KProcessPageTable* pPageTable)
{
    uint64_t physAddr = (static_cast<uint64_t>(flag.Get<MapIoFlags::Pfn>()) << 12);
    if (physAddr != GetAsInteger(KPhysicalAddress(physAddr)))
    {
        return nn::svc::ResultInvalidAddress();
    }
    size_t numPages = 1;

    if (physAddr >= physAddr + numPages * NN_KERN_FINEST_PAGE_SIZE)
    {
        return nn::svc::ResultInvalidAddress();
    }
    if ((physAddr + numPages * NN_KERN_FINEST_PAGE_SIZE - 1) & ~PhysicalMapMask)
    {
        return nn::svc::ResultInvalidAddress();
    }
    return pPageTable->MapIo(KPhysicalAddress(physAddr), numPages * NN_KERN_FINEST_PAGE_SIZE, KMemoryPermission_UserReadWrite);
}

}}  // namespace nn::kern
