﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include "kern_InterlockedSelect.h"
#include "kern_Platform.h"
#include "kern_KRbTree.h"


/*! @file

    @brief      KAutoObject クラス定義です。

    AutoObject は以下の機能を提供します。

    ・スマートポインタ

    参照されるたびにインクリメントするカウンターを持ちます。
    生成したオブジェクトがすべての箇所から参照されなくなったら ( Close() されたら) 消滅します。

    ・型のキャスト

    KAutoObjectを派生する型に対して、型診断つきのキャストを提供します。

*/

namespace nn { namespace kern {

//! 型取得のための関数定義
#define NN_AUTOOBJECT_DEFINE_TYPE_FUNC                                                  \
    virtual TypeObj     GetTypeObj() const  { return TypeObj( TYPE_NAME, TYPE_ID ); }   \
    virtual const char* GetTypeName() const { return GetTypeObj().GetTypeName(); }      \
    static TypeObj      GetTypeObjStatic()  { return TypeObj( TYPE_NAME, TYPE_ID ); }   \
    static const char*  GetTypeNameStatic() { return GetTypeObjStatic().GetTypeName(); }

//! 型名宣言/定義
#define NN_AUTOOBJECT_DECLARE_TYPE_NAME(className) \
    static const char* const TYPE_NAME;
#define NN_AUTOOBJECT_DECLARE_TYPE_ID(className)   \
    static const TypeObjId TYPE_ID = className ## TypeId ;
#ifdef NN_KERN_FOR_DEVELOPMENT
#define NN_AUTOOBJECT_DEFINE_TYPE_NAME(className)  \
    const char* const className::TYPE_NAME = #className
#else
#define NN_AUTOOBJECT_DEFINE_TYPE_NAME(className)  \
    const char* const className::TYPE_NAME = ""
#endif

//! セット
//! KAutoObjectのプリセット関数セット定義です。KAutoObjectを継承するクラスの末尾に記述する必要があります(public/private)
#define NN_AUTOOBJECT_DERIVED_FUNCSET(className, baseClassName) \
    public:                                                     \
        NN_AUTOOBJECT_DEFINE_TYPE_FUNC                          \
    private:                                                    \
        NN_AUTOOBJECT_DECLARE_TYPE_NAME(className)              \
        NN_AUTOOBJECT_DECLARE_TYPE_ID(className)


class KProcess;

class KAutoObject
{
protected:
    enum TypeObjId
    {
        KAutoObjectTypeId            = 0,
        KSynchronizationObjectTypeId = ( 1 << 0) | KAutoObjectTypeId,
        KReadableEventTypeId         = ( 1 << 1) | KSynchronizationObjectTypeId,
        KInterruptEventTypeId        = (((1 << 2) | (1 << 1) | (1 << 0)) << 8) | KReadableEventTypeId,
        KDebugTypeId                 = (((1 << 3) | (1 << 1) | (1 << 0)) << 8) | KSynchronizationObjectTypeId,
        KThreadTypeId                = (((1 << 4) | (1 << 1) | (1 << 0)) << 8) | KSynchronizationObjectTypeId,
        KServerPortTypeId            = (((1 << 5) | (1 << 1) | (1 << 0)) << 8) | KSynchronizationObjectTypeId,
        KServerSessionTypeId         = (((1 << 6) | (1 << 1) | (1 << 0)) << 8) | KSynchronizationObjectTypeId,
        KClientPortTypeId            = (((1 << 7) | (1 << 1) | (1 << 0)) << 8) | KSynchronizationObjectTypeId,

        KClientSessionTypeId         = (((1 << 3) | (1 << 2) | (1 << 0)) << 8) | KAutoObjectTypeId,
        KProcessTypeId               = (((1 << 4) | (1 << 2) | (1 << 0)) << 8) | KSynchronizationObjectTypeId,
        KResourceLimitTypeId         = (((1 << 5) | (1 << 2) | (1 << 0)) << 8) | KAutoObjectTypeId,
        KLightSessionTypeId          = (((1 << 6) | (1 << 2) | (1 << 0)) << 8) | KAutoObjectTypeId,
        KPortTypeId                  = (((1 << 7) | (1 << 2) | (1 << 0)) << 8) | KAutoObjectTypeId,

        KSessionTypeId               = (((1 << 4) | (1 << 3) | (1 << 0)) << 8) | KAutoObjectTypeId,
        KSharedMemoryTypeId          = (((1 << 5) | (1 << 3) | (1 << 0)) << 8) | KAutoObjectTypeId,
        KEventTypeId                 = (((1 << 6) | (1 << 3) | (1 << 0)) << 8) | KAutoObjectTypeId,
        KWritableEventTypeId         = (((1 << 7) | (1 << 3) | (1 << 0)) << 8) | KAutoObjectTypeId,

        KLightClientSessionTypeId    = (((1 << 5) | (1 << 4) | (1 << 0)) << 8) | KAutoObjectTypeId,
        KLightServerSessionTypeId    = (((1 << 6) | (1 << 4) | (1 << 0)) << 8) | KAutoObjectTypeId,
        KTransferMemoryTypeId        = (((1 << 7) | (1 << 4) | (1 << 0)) << 8) | KAutoObjectTypeId,

        KDeviceAddressSpaceTypeId    = (((1 << 6) | (1 << 5) | (1 << 0)) << 8) | KAutoObjectTypeId,
        KSessionRequestTypeId        = (((1 << 7) | (1 << 5) | (1 << 0)) << 8) | KAutoObjectTypeId,
        KCodeMemoryTypeId            = (((1 << 7) | (1 << 6) | (1 << 0)) << 8) | KAutoObjectTypeId,

        //                             (((1 << 3) | (1 << 2) | (1 << 1)) << 8)
        //                             (((1 << 4) | (1 << 2) | (1 << 1)) << 8)
        //                             (((1 << 5) | (1 << 2) | (1 << 1)) << 8)
        //                             (((1 << 6) | (1 << 2) | (1 << 1)) << 8)
        //                             (((1 << 7) | (1 << 2) | (1 << 1)) << 8)
        //                             (((1 << 4) | (1 << 3) | (1 << 1)) << 8)
        //                             (((1 << 5) | (1 << 3) | (1 << 1)) << 8)
        //                             (((1 << 6) | (1 << 3) | (1 << 1)) << 8)
        //                             (((1 << 7) | (1 << 3) | (1 << 1)) << 8)
        //                             (((1 << 5) | (1 << 4) | (1 << 1)) << 8)
        //                             (((1 << 6) | (1 << 4) | (1 << 1)) << 8)
        //                             (((1 << 7) | (1 << 4) | (1 << 1)) << 8)
        //                             (((1 << 6) | (1 << 5) | (1 << 1)) << 8)
        //                             (((1 << 7) | (1 << 5) | (1 << 1)) << 8)
        //                             (((1 << 7) | (1 << 6) | (1 << 1)) << 8)
        //                             (((1 << 4) | (1 << 3) | (1 << 2)) << 8)
        //                             (((1 << 5) | (1 << 3) | (1 << 2)) << 8)
        //                             (((1 << 6) | (1 << 3) | (1 << 2)) << 8)
        //                             (((1 << 7) | (1 << 3) | (1 << 2)) << 8)
        //                             (((1 << 5) | (1 << 4) | (1 << 2)) << 8)
        //                             (((1 << 6) | (1 << 4) | (1 << 2)) << 8)
        //                             (((1 << 7) | (1 << 4) | (1 << 2)) << 8)
        //                             (((1 << 6) | (1 << 5) | (1 << 2)) << 8)
        //                             (((1 << 7) | (1 << 5) | (1 << 2)) << 8)
        //                             (((1 << 7) | (1 << 6) | (1 << 2)) << 8)
        //                             (((1 << 5) | (1 << 4) | (1 << 3)) << 8)
        //                             (((1 << 6) | (1 << 4) | (1 << 3)) << 8)
        //                             (((1 << 7) | (1 << 4) | (1 << 3)) << 8)
        //                             (((1 << 6) | (1 << 5) | (1 << 3)) << 8)
        //                             (((1 << 7) | (1 << 5) | (1 << 3)) << 8)
        //                             (((1 << 7) | (1 << 6) | (1 << 3)) << 8)
        //                             (((1 << 6) | (1 << 5) | (1 << 4)) << 8)
        //                             (((1 << 7) | (1 << 5) | (1 << 4)) << 8)
        //                             (((1 << 7) | (1 << 6) | (1 << 4)) << 8)
        //                             (((1 << 7) | (1 << 6) | (1 << 5)) << 8)
    };

    // 型情報を提供する
    class TypeObj
    {
    public:
        // コンストラクタ
        explicit TypeObj(const char* name, TypeObjId id)
            : m_pName(name), m_Id(id)
        {
        }

        // 型名を取得
        const char* GetTypeName() const { return m_pName; }

        // 型の比較
        bool operator ==(const TypeObj& rhs) const { return this->GetTypeID() == rhs.GetTypeID(); }
        bool operator !=(const TypeObj& rhs) const { return this->GetTypeID() != rhs.GetTypeID(); }

        // タイプIDを取得
        Bit16 GetTypeID() const { return m_Id; }

    private:
        const char* m_pName;      // 型名リソース
        TypeObjId m_Id;
        NN_PADDING2;
    };

private:
    InterlockedVariable<uint32_t> m_ReferenceCount;
//    Bit32   m_CreationTimeLo;
//    Bit32   m_CreationTimeHi;
#if NN_KERN_ENABLE_OBJECT_INFO
      Bit64   m_ParentId;
#endif

      struct IncrementIfPositive
      {
          bool operator()(uint32_t* x)
          {
              uint32_t v = *x;
              if (NN_LIKELY(v > 0))
              {
                  NN_KERN_ABORT_UNLESS(v + 1 > v);
                  *x = v + 1;
                  return true;
              }
              else
              {
                  return false;
              }
          }
      };

      struct DecrementIfPositive
      {
          uint32_t newVal;
          bool operator()(uint32_t* x)
          {
              uint32_t v = *x;
              NN_KERN_ABORT_UNLESS(v > 0);
              newVal = *x = v - 1;
              return true;
          }
      };

protected:
    explicit KAutoObject() :m_ReferenceCount(0) { NN_KERN_THIS_ASSERT(); }
    virtual  ~KAutoObject() { NN_KERN_THIS_ASSERT(); }
    KAutoObject& operator=(const KAutoObject&) = delete;
    KAutoObject(const KAutoObject&) = delete;

    /*
        @brief      AutoObject の終了

        参照カウントが 0 になったら呼ばれます。

    */
    virtual void Destroy() { NN_KERN_THIS_ASSERT(); }
    virtual void Finalize() {}

public:
    /*
        @brief      AutoObject の初期化処理

        @return     参照カウントが 1 になった状態のオブジェクトを返します。

    */
    static KAutoObject* Create(KAutoObject* obj);

    /*
        @brief      AutoObject の参照

        参照カウントを増やします。通常、GetObject等で暗黙的に呼ばれます。

    */
    NN_FORCEINLINE bool Open()
    {
        NN_KERN_THIS_ASSERT();
        IncrementIfPositive updater;
        return m_ReferenceCount.AtomicUpdateConditional(&updater);
    }

    /*
        @brief      AutoObject の参照終了

        参照カウントを減らします。GetObject等で得たオブジェクトの開放に使用します。

    */
    NN_FORCEINLINE void Close()
    {
        NN_KERN_THIS_ASSERT();

        //PrintOpenCloseEvent(false, this, count);
        NN_KERN_ASSERT(CanClose());

        DecrementIfPositive updater;
        m_ReferenceCount.AtomicUpdateConditional(&updater);

        // 減算した後に count == 0 であったら、実際の削除処理をする。
        if (updater.newVal == 0)
        {
            Destroy();
        }
    }

    /*
        @brief      参照カウンターを取得します

        @return     参照カウントを返します。

    */
    uint32_t     GetReferenceCount() const { return m_ReferenceCount; }
#if NN_KERN_ENABLE_OBJECT_INFO
    Bit64   GetParentId()       const { return m_ParentId; }
//    int64_t     GetCreationTime()   const { return (static_cast<int64_t>(m_CreationTimeHi) << 32) | m_CreationTimeLo; }
#endif

    /*
        @brief      内部状態を表示します。

    */
    void Show();

    virtual KProcess* GetOwner() const { return nullptr; }


//---- 以下、動的キャスト実現のためのコード
private:
    template<typename T> struct depointer
    {
        typedef T type;
    };
    template<typename T> struct depointer<T*>
    {
        typedef T type;
    };

    static bool CanClose();

public:
    //! 継承判定
    bool IsDerivedFrom(TypeObj type) const { return (type.GetTypeID() | GetTypeObj().GetTypeID()) == GetTypeObj().GetTypeID(); }

    template<typename DstType>
    DstType DynamicCast()
    {
        // DstType が KAutoObject から派生しているか確認します。
        // 派生していない場合はこの時点でコンパイルエラーが発生します。
        typedef typename depointer<DstType>::type DeDstType;

        // キャスト元、キャスト先の型名が等しい場合はキャスト可能です
        if (IsDerivedFrom(DeDstType::GetTypeObjStatic()))
        {
            return reinterpret_cast<DstType>(this);
        }
        else
        {
            return nullptr;
        }
    }

    template<typename DstType>
    const DstType DynamicCast() const
    {
        // DstType が KAutoObject から派生しているか確認します。
        // 派生していない場合はこの時点でコンパイルエラーが発生します。
        typedef typename depointer<DstType>::type DeDstType;

        // キャスト元、キャスト先の型名が等しい場合はキャスト可能です
        if (IsDerivedFrom(DeDstType::GetTypeObjStatic()))
        {
            return reinterpret_cast<DstType>(this);
        }
        else
        {
            return nullptr;
        }
    }

public:
    NN_AUTOOBJECT_DEFINE_TYPE_FUNC
private:
    NN_AUTOOBJECT_DECLARE_TYPE_NAME(KAutoObject)
    NN_AUTOOBJECT_DECLARE_TYPE_ID(KAutoObject)
};

class KObjectContainer;
class KAutoObjectWithList : public KAutoObject
{
    friend KObjectContainer;
public:
    virtual Bit64 GetId() const { return reinterpret_cast<Bit64>(this); }
private:
    IntrusiveRbTreeNode m_Node;
};

template <typename T>
class KScopedAutoObject
{
private:
    T* m_p;

public:
    explicit KScopedAutoObject(T* p) : m_p(p) {}
    ~KScopedAutoObject() { m_p->Close(); }

    T* operator ->() { return m_p; }
    T& operator *() { return *m_p; }

    void CloseAndReset(T* p)
    {
        m_p->Close();
        m_p = p;
    }
};

}}

