﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include "kern_KRbTree.h"
#include "kern_KThread.h"

namespace nn { namespace kern {

class KThread;
class KProcess;
struct KAddressArbiterCompare
{
    int operator() (const KThread& a, const KThread& b)
    {
        KProcessAddress cvKeyA = a.GetAddressKey();
        KProcessAddress cvKeyB = b.GetAddressKey();

        if (cvKeyA < cvKeyB)
        {
            return -1;
        }
        else if (cvKeyA == cvKeyB)
        {
            int priA = a.GetPriority();
            int priB = b.GetPriority();
            if (priA < priB)
            {
                return -1;
            }
        }
        return 1;
    }
};


class KAddressArbiter
{
private:
    typedef IntrusiveRbTree<KThread, IntrusiveRbTreeMemberNodeTraits<KThread, &KThread::m_AddressArbiterNode>, KAddressArbiterCompare> AddressArbiterList;
    AddressArbiterList m_List;

public:
    Result WaitForAddress(uintptr_t addr, nn::svc::ArbitrationType type, int32_t value, int64_t ns)
    {
        switch (type)
        {
        case nn::svc::ArbitrationType_WaitIfLessThan:
            return WaitIfLessThan(addr, value, false, ns);

        case nn::svc::ArbitrationType_DecrementAndWaitIfLessThan:
            return WaitIfLessThan(addr, value, true, ns);

        case nn::svc::ArbitrationType_WaitIfEqual:
            return WaitIfEqual(addr, value, ns);

        default:
            NN_KERN_ABORT();
            break;
        }
    }

    Result SignalToAddress(uintptr_t addr, nn::svc::SignalType type, int32_t value, int32_t num)
    {
        switch (type)
        {
        case nn::svc::SignalType_SignalOnly:
            return Signal(addr, num);

        case nn::svc::SignalType_IncrementIfEqual:
            return SignalIncrementIfEqual(addr, value, num);

        case nn::svc::SignalType_UpdateByCountIfEqual:
            return SignalUpdateByCountIfEqual(addr, value, num);

        default:
            NN_KERN_ABORT();
            break;
        }
    }

private:
    Result WaitIfLessThan(uintptr_t addr, int32_t value, bool dec, int64_t ns);
    Result WaitIfEqual(uintptr_t addr, int32_t value, int64_t ns);
    Result Signal(uintptr_t addr, int32_t num);
    Result SignalIncrementIfEqual(uintptr_t addr, int32_t value, int32_t num);
    Result SignalUpdateByCountIfEqual(uintptr_t addr, int32_t value, int32_t num);

};

}}

