﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Base.h>
#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include "kern_Platform.h"
#include "kern_KProcess.h"
#include "kern_InitialProcess.h"
#include "kern_Kernel.h"
#include "kern_KKipReader.h"
#include "kern_DebugString.h"
#include "kern_MemoryMap.h"
#include "kern_SystemControl.h"
#include "kern_KMemoryLayout.h"
#include "kern_Panic.h"
#include "kern_Utility.h"

#include <cstring>

using namespace nn::kern;

namespace nn {
    namespace kern {
        namespace
        {
            const bool enableAslr = true;

            KVirtualAddress GetInitialProgramVirtualAddress()
            {
                return KMemoryLayout::ToLinearVirtualAddress(KPhysicalAddress(NN_KERN_P_ADDR_INITIAL_PROCESS_DEST));
            }

            void InitialProcessFileHeader(InitialProcessImageHeader* pHeader)
            {
                if (pHeader->signature != InitialProcessImageHeaderValidSignature)
                {
                    *pHeader = *GetTypedPointer<InitialProcessImageHeader>(GetInitialProgramVirtualAddress());
                }

                NN_KERN_ABORT_UNLESS(pHeader->signature == InitialProcessImageHeaderValidSignature);
                NN_KERN_ABORT_UNLESS(pHeader->numInitialProcesses <= NN_KERN_SLAB_OBJ_NUM_PROCESS);
            }

            struct RunProcessInfo
            {
                KProcess* pProcess;
                size_t stackSize;
                int32_t priority;
            };

            // INITIAL_PROCESS からプロセスを生成する
            void LoadProcess(RunProcessInfo* pInfo, KVirtualAddress pImages, const InitialProcessImageHeader& header)
            {
                Bit8* pData = GetTypedPointer<Bit8>(pImages + sizeof(InitialProcessImageHeader));
                Bit8* pImagesTail = GetTypedPointer<Bit8>(pImages + header.totalLength);

                const uint32_t numInitialProcesses = header.numInitialProcesses;
                for (uint32_t processNum = 0; processNum < numInitialProcesses; processNum++)
                {
                    NN_KERN_ABORT_UNLESS(pData <= pImagesTail - sizeof(KKipReader));
                    KKipReader reader;
                    NN_KERN_ABORT_UNLESS(reader.Bind(pData));

                    nn::svc::CreateProcessParameter params;
                    NN_KERN_ABORT_IF_FAILED(reader.MakeProcessParameter(&params, enableAslr));
                    NN_KERN_ABORT_UNLESS(Kernel::GetSystemResourceLimit().TestLimit(nn::svc::LimitableResource_PhysicalMemoryMax, params.memoryNumPages * NN_KERN_FINEST_PAGE_SIZE));

                    KMemoryManager::Region region = (reader.UseSecureMemory()? KMemoryManager::Region_SecureSystem: KSystemControl::GetCreateProcessMemoryRegion());
                    Bit32 allocateOption = KMemoryManager::MakeAllocateOption(
                            region,
                            KMemoryManager::From_Front);
                    KMemoryManager& mm = Kernel::GetKernelHeapManager();

                    KScopedPageGroup pg(&Kernel::GetBlockInfoManager());
                    NN_KERN_ABORT_IF_FAILED(mm.Allocate(&pg.GetPageGroup(), params.memoryNumPages, allocateOption));
                    pg.GetPageGroup().Open();

                    KProcessAddress addr;
                    NN_KERN_ABORT_IF_FAILED(Kernel::GetKernelPageTable().MapPageGroup(
                                &addr,
                                pg.GetPageGroup(),
                                KMemoryLayout::GetTempRegionBegin(),
                                KMemoryLayout::GetTempRegionSize() / NN_KERN_FINEST_PAGE_SIZE,
                                KMemoryState_Kernel,
                                KMemoryPermission_KernelReadWrite));

                    NN_KERN_ABORT_IF_FAILED(reader.Load(addr, params));
                    NN_KERN_ABORT_IF_FAILED(Kernel::GetKernelPageTable().UnmapPageGroup(addr, pg.GetPageGroup(), KMemoryState_Kernel));

                    KProcess* pProcess = KProcess::Create();
                    NN_KERN_ABORT_UNLESS(pProcess);

                    // プロセスの作成
                    NN_KERN_ABORT_IF_FAILED(pProcess->Initialize(
                            params, pg.GetPageGroup(),
                            reader.GetKernelCapabilities(),
                            reader.GetKernelCapabilitiesSize() / sizeof(uint32_t),
                            &Kernel::GetSystemResourceLimit(),
                            region));

                    pg.GetPageGroup().Close();

                    NN_KERN_ABORT_IF_FAILED(reader.SetPermission(&pProcess->GetPageTable(), params));
                    NN_KERN_ABORT_IF_FAILED(KProcess::Register(pProcess));

                    // プロセスの設定
                    pProcess->SetIdealProcessor(reader.GetIdealProcessor());

                    pInfo[processNum].pProcess = pProcess;
                    pInfo[processNum].stackSize = reader.GetStackSize();
                    pInfo[processNum].priority = reader.GetPriority();

                    pData += reader.GetSize();
                }
            }

            KVirtualAddress s_pProcessImages;
            InitialProcessImageHeader s_ProcessImagesHeader;
            Bit64 s_ProcessIdMin = 0xffffffffffffffffull;
            Bit64 s_ProcessIdMax = 0;
        }

        // 初期プロセス数を返します。
        uint32_t GetNumOfInitialProcesses()
        {
            InitialProcessFileHeader(&s_ProcessImagesHeader);
            return s_ProcessImagesHeader.numInitialProcesses;
        }

        Bit64 GetInitialProcessIdMin()
        {
            return s_ProcessIdMin;
        }

        Bit64 GetInitialProcessIdMax()
        {
            return s_ProcessIdMax;
        }

        size_t GetInitialProcessesNonSecureMemorySize()
        {
            InitialProcessFileHeader(&s_ProcessImagesHeader);
            KVirtualAddress pImages((s_pProcessImages == Null<KVirtualAddress>())? GetInitialProgramVirtualAddress(): s_pProcessImages);

            Bit8* pData = GetTypedPointer<Bit8>(pImages + sizeof(InitialProcessImageHeader));
            Bit8* pImagesTail = GetTypedPointer<Bit8>(pImages + s_ProcessImagesHeader.totalLength);

            size_t size = 0;
            const int numInitialProcesses = s_ProcessImagesHeader.numInitialProcesses;
            for (int processNum = 0; processNum < numInitialProcesses; processNum++)
            {
                NN_KERN_ABORT_UNLESS(pData <= pImagesTail - sizeof(KKipReader));
                KKipReader reader;
                NN_KERN_ABORT_UNLESS(reader.Bind(pData));

                if (!(reader.UseSecureMemory()))
                {
                    size += reader.GetMemorySize();
                    size += RoundUp(reader.GetStackSize(), NN_KERN_FINEST_PAGE_SIZE);
                }
                pData += reader.GetSize();
            }

            return size;
        }

        // メインメモリ上に初期プロセスのイメージをコピーします。
        size_t CopyProcessImagesToMainMemory()
        {
            size_t size = 0;
            InitialProcessFileHeader(&s_ProcessImagesHeader);

            if (s_ProcessImagesHeader.numInitialProcesses > 0)
            {
                KMemoryManager& mm = Kernel::GetKernelHeapManager();
                const size_t numPages = (s_ProcessImagesHeader.totalLength + NN_KERN_FINEST_PAGE_SIZE - 1) / NN_KERN_FINEST_PAGE_SIZE;
                NN_KERN_ABORT_UNLESS(Kernel::GetSystemResourceLimit().TestLimit(nn::svc::LimitableResource_PhysicalMemoryMax, numPages * NN_KERN_FINEST_PAGE_SIZE));
                KVirtualAddress pImages = mm.AllocateContinuous(numPages, 1, KMemoryManager::MakeAllocateOption(KSystemControl::GetCreateProcessMemoryRegion(), KMemoryManager::From_Front));
                NN_KERN_ABORT_UNLESS(pImages != Null<KVirtualAddress>());
                mm.Open(pImages, numPages);

                std::memmove(GetUntypedPointer(pImages), GetUntypedPointer(GetInitialProgramVirtualAddress()), s_ProcessImagesHeader.totalLength);
                std::memset(GetUntypedPointer(GetInitialProgramVirtualAddress()), 0, s_ProcessImagesHeader.totalLength);
                s_pProcessImages = pImages;
                size += numPages * NN_KERN_FINEST_PAGE_SIZE;
            }

            return size;
        }

        // メインメモリ上の初期プロセスをロードして実行します。
        void LoadAndRunProcessesFromMainMemory()
        {
            // ロード
            RunProcessInfo* pInfo = static_cast<RunProcessInfo*>(__builtin_alloca(sizeof(RunProcessInfo) * s_ProcessImagesHeader.numInitialProcesses));
            LoadProcess(pInfo, s_pProcessImages, s_ProcessImagesHeader);

            // 解放
            if (s_ProcessImagesHeader.numInitialProcesses > 0)
            {
                const size_t numPages = (s_ProcessImagesHeader.totalLength + NN_KERN_FINEST_PAGE_SIZE - 1) / NN_KERN_FINEST_PAGE_SIZE;
                Kernel::GetKernelHeapManager().Close(s_pProcessImages, numPages);
                Kernel::GetSystemResourceLimit().ReleaseLimit(nn::svc::LimitableResource_PhysicalMemoryMax, numPages * NN_KERN_FINEST_PAGE_SIZE);
            }

            for (uint32_t i = 0; i < s_ProcessImagesHeader.numInitialProcesses; i++)
            {
                Bit64 pid = pInfo[i].pProcess->GetId();
                if (pid < s_ProcessIdMin)
                {
                    s_ProcessIdMin = pid;
                }
                if (s_ProcessIdMax < pid)
                {
                    s_ProcessIdMax = pid;
                }
            }

            // プロセスが1つでも実行された後は
            // s_ProcessIdMin, s_ProcessIdMax は変更されない。

            // 実行
            for (uint32_t i = 0; i < s_ProcessImagesHeader.numInitialProcesses; i++)
            {
                pInfo[i].pProcess->Run(pInfo[i].priority, pInfo[i].stackSize);
            }
        }
    }
}
