﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include "kern_MemoryMap.h"
#include "../../kern_KTaggedAddress.h"
#include "../ARM64/kern_RegisterAccess.h"
#include "../../kern_KMemoryLayout.h"
#include "kern_KPageTableDefinition.h"
#include "kern_MemoryMap.h"

namespace nn {
    namespace kern {
        namespace ARMv8A {

class KPageTableBody
{
private:
    Bit64* m_pTable;
    bool m_IsKernel;
    uint32_t m_NumEntry;

public:
    struct TraverseData
    {
        KPhysicalAddress    address;
        size_t              size;
    };
    struct TraverseContext
    {
        const Bit64*    pL1;    // 次の L1 エントリへのポインタ
        const Bit64*    pL2;    // 次の L2 エントリへのポインタ
        const Bit64*    pL3;    // 次の L3 エントリへのポインタ
    };

private:
    static KVirtualAddress GetPageTableVirtualAddress(KPhysicalAddress pa)
    {
        return KMemoryLayout::ToLinearVirtualAddress(pa);
    }

    static bool GetAddrFromL1Entry(size_t* pNumPages, KPhysicalAddress* pOut, const Bit64** ppL2, const Bit64** ppL3, Bit64 entryValue, KProcessAddress srcAddress);
    static bool GetAddrFromL2Entry(size_t* pNumPages, KPhysicalAddress* pOut, const Bit64** ppL3, Bit64 entryValue, KProcessAddress srcAddress);
    static bool GetAddrFromL3Entry(size_t* pNumPages, KPhysicalAddress* pOut, Bit64 entryValue, KProcessAddress srcAddress);

public:
    void    InitializeForKernel(void* pTable, KProcessAddress begin, KProcessAddress end);
    void    InitializeForProcess(void* pTable, KProcessAddress begin, KProcessAddress end);
    Bit64*  Finalize();

    void    Dump(uintptr_t begin, size_t size) const;

    bool TraverseBegin(TraverseData* pOut, TraverseContext* pContext, KProcessAddress addr) const;
    bool TraverseNext (TraverseData* pOut, TraverseContext* pContext) const;
    bool GetPhysicalAddress(KPhysicalAddress* pOut, KProcessAddress vaddr) const;

    // TORIAEZU: KPageTable でのキャッシュ操作のため
    void* GetAddress() const { return m_pTable; }

    Bit64 GetL1Entry(KProcessAddress addr)
    {
        uint32_t numEntry = m_NumEntry;
        int32_t index = (GetAsInteger(addr) >> (12 + 9 * 2)) & (numEntry - 1);
        return m_pTable[index];
    }
    void SetL1Entry(KProcessAddress addr, Bit64 entry)
    {
        uint32_t numEntry = m_NumEntry;
        int32_t index = (GetAsInteger(addr) >> (12 + 9 * 2)) & (numEntry - 1);
        m_pTable[index] = entry;
    }
    Bit64 GetL2Entry(KProcessAddress addr, Bit64* pL2Table)
    {
        int32_t index = (GetAsInteger(addr) >> (12 + 9 * 1)) & ((1 << 9) - 1);
        return pL2Table[index];
    }
    void SetL2Entry(KProcessAddress addr, Bit64 entry, Bit64* pL2Table)
    {
        int32_t index = (GetAsInteger(addr) >> (12 + 9 * 1)) & ((1 << 9) - 1);
        pL2Table[index] = entry;
    }
    Bit64 GetL3Entry(KProcessAddress addr, Bit64* pL3Table)
    {
        int32_t index = (GetAsInteger(addr) >> (12 + 9 * 0)) & ((1 << 9) - 1);
        return pL3Table[index];
    }
    void SetL3Entry(KProcessAddress addr, Bit64 entry, Bit64* pL3Table)
    {
        int32_t index = (GetAsInteger(addr) >> (12 + 9 * 0)) & ((1 << 9) - 1);
        pL3Table[index] = entry;
    }
    bool GetTableAddressFromEntry(KPhysicalAddress* pOut, Bit64 entry)
    {
        if ((entry & 0x3) == 0x3)
        {
            *pOut = (entry & 0x0000FFFFFFFFF000ull);
            return true;
        }
        else
        {
            return false;
        }
    }
    bool GetBlockAddressFromEntry(KPhysicalAddress* pOut, Bit64 entry)
    {
        if ((entry & 0x3) == 0x1)
        {
            *pOut = (entry & 0x0000FFFFFFFFF000ull);
            return true;
        }
        else
        {
            return false;
        }
    }
    bool GetPageAddressFromEntry(KPhysicalAddress* pOut, Bit64 entry)
    {
        if ((entry & 0x3) == 0x3)
        {
            *pOut = (entry & 0x0000FFFFFFFFF000ull);
            return true;
        }
        else
        {
            return false;
        }
    }

    size_t CountPageTables() const;
};

        }
    }
}


