﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include "../../kern_CPUSelect.h"
#include "../../kern_KHardwareTimerBase.h"
#include "../ARM64/kern_RegisterAccess.h"

namespace nn { namespace kern {

    namespace ARMv8A {


class KHardwareTimer
    : public KHardwareTimerBase
{
public:
    void    Initialize(int32_t coreNo);
    void    Finalize(int32_t coreNo);

    static int64_t GetTick()
    {
        return GetTimerCount();
    }

    virtual void  DoInterruptTask();

    static uint32_t GetRawTick()
    {
        return GetTimerLowCount();
    }

    void RegisterTaskAbsolute(KTimerTask* pTask, int64_t tick)
    {
        KScopedDisableDispatch dd;
        KScopedSimpleLock lock(&GetLockMutex());
        if (RegisterTaskAbsoluteImpl(pTask, tick))
        {
            SetTimerCount(tick);
            EnableHardwareTimerInterrupt();
        }
    }

private:
    enum {
#if 1
        INTERRUPT_ID_TIMER  = 30, // Non-Secure Timer
#else
        INTERRUPT_ID_TIMER  = 29,
#endif
    };

    static void InitGlobalTimer(int32_t coreNo)
    {
        NN_UNUSED(coreNo);

        // コアごとに初期化が必要
        HW_SET_CNTKCTL_EL1(HW_CNTKCTL_EL0PCTEN);
        HW_SET_CNTP_CTL_EL0(0x00000000ull);
        HW_SET_CNTP_CVAL_EL0(0xffffffffffffffffull);
        HW_SET_CNTP_CTL_EL0(0x00000003ull);
    }

    static void SetTimerCount(int64_t timerValue)
    {
        HW_SET_CNTP_CVAL_EL0(timerValue);
    }

    static int64_t GetTimerCount()
    {
        int64_t val;

        HW_GET_CNTPCT_EL0(val);

        return val;
    }

    static uint32_t GetTimerLowCount()
    {
        return static_cast<uint32_t>(GetTimerCount());
    }

    static void EnableHardwareTimerInterrupt()
    {
        HW_SET_CNTP_CTL_EL0(0x00000001ull);
    }

    static void StopHardwareTimer()
    {
        HW_SET_CNTP_CVAL_EL0(0xffffffffffffffffull);
        HW_SET_CNTP_CTL_EL0(0x00000000ull);
    }

    static void DisableHardwareTimerInterrupt()
    {
        HW_SET_CNTP_CTL_EL0(0x00000003ull);
    }
};

    }
}}

