﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/TargetConfigs/build_Base.h>
#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include "kern_MemoryMap.h"
#include "../../kern_ContextSelect.h"
#include "../../kern_InterlockedSelect.h"
#include "../ARM/kern_RegisterAccess.h"
#include "../../kern_InterruptControllerSelect.h"

namespace nn { namespace kern {

class KThread;

namespace ARMv7A {

class KCPU
{
public:
#if defined NN_BUILD_CONFIG_CPU_CORTEX_A15
    static const size_t INSTRUCTION_CACHE_LINE_SIZE   = 64;
    static const size_t DATA_CACHE_LINE_SIZE          = 64;
    static const int    LOCAL_DATA_CACHE_LEVELS       = 1;
    static const int    NUM_PERFORMANCE_COUNTER       = 6;
#elif defined NN_BUILD_CONFIG_CPU_CORTEX_A7
    static const size_t INSTRUCTION_CACHE_LINE_SIZE   = 32;
    static const size_t DATA_CACHE_LINE_SIZE          = 64;
    static const int    LOCAL_DATA_CACHE_LEVELS       = 1;
    static const int    NUM_PERFORMANCE_COUNTER       = 4;
#elif defined NN_BUILD_CONFIG_CPU_CORTEX_A9
    static const size_t INSTRUCTION_CACHE_LINE_SIZE   = 32;
    static const size_t DATA_CACHE_LINE_SIZE          = 32;
    static const int    LOCAL_DATA_CACHE_LEVELS       = 1;
    static const int    NUM_PERFORMANCE_COUNTER       = 6;
#else
    #error NN_BUILD_CONFIG_CPU_ is not defined
#endif


#if defined(NN_BUILD_CONFIG_HARDWARE_KZMA9)
    static const int32_t    NUM_CORE = 2;
#elif defined(NN_BUILD_CONFIG_HARDWARE_BDSLIMX6)
    static const int32_t    NUM_CORE = 4;
#elif defined(NN_BUILD_CONFIG_HARDWARE_JETSONTK1)
    static const int32_t    NUM_CORE = 4;
#else
#error NN_BUILD_CONFIG_HARDWARE_ is not defined
#endif

public:
    static Result InvalidateDataCache(void* addr, size_t size);
    static Result StoreDataCache(const void* addr, size_t size);
    static Result FlushDataCache(const void* addr, size_t size);
    static Result InvalidateInstructionCache(void* addr, size_t size);

    static void InvalidateEntireDataCache();
    static void StoreEntireDataCache();
    static void FlushEntireDataCache();
    static void InvalidateEntireInstructionCache();

    static void FlushEntireDataCacheLocal();
    static void InvalidateEntireInstructionCacheLocal();

    static void SwithcToHyp();

    static void SynchronizeAllCore();
    static void Initialize0(int32_t coreNo);
    static void Initialize1(int32_t coreNo);

    static void SleepCore()
    {
        asm volatile("wfi");
    }
    static int32_t  GetCurrentCoreNo()
    {
        Bit32 cpuId;
        HW_GET_CP15_CPU_ID(cpuId);
        return (cpuId & HW_C0_AP_CPU_ID_MASK);
    }
    static void OnThreadCoreChanged(KThread* pThread)
    {
        KContext::OnThreadCoreChanged(pThread);
    }

    static void SwitchThreadLocalRegion(uintptr_t tlr)
    {
        HW_SET_CP15_THREAD_ID_USER_READ_ONLY(tlr);
    }

    //-----------------------------------------------------
    // CPU 固有

    static void WaitDataCacheOperation()
    {
        DataSynchronizationBarrier();
    }
    static void CareInstructionConsistency()
    {
        HW_CP15_FLUSH_ENTIRE_BRANCH_TARGET_CACHE();
        DataSynchronizationBarrier();
        InstructionMemoryBarrier();
    }
    static void SwitchProcess(Bit32 asid, Bit32 ttbrValue)
    {
        HW_SET_CP15_CONTEXT_ID(asid);
        HW_SET_CP15_TTB0(ttbrValue, 0);
        HW_CP15_FLUSH_ENTIRE_BRANCH_TARGET_CACHE();
        InstructionMemoryBarrier();
    }
    static uint8_t GetCurrentAsid()
    {
        Bit32 cir;
        HW_GET_CP15_CONTEXT_ID(cir);
        return static_cast<uint8_t>(cir & 0xFF);
    }
    static void InvalidateEntireTlb()
    {
        HW_CP15_INVALIDATE_UNIFIED_TLB();
        CareInstructionConsistency();
    }
    static void CareInstructionConsistencyAllCore()
    {
        DataSynchronizationBarrier();
        InstructionMemoryBarrier();
    }
    static void InvalidateEntireTlbNoCodeModificationAllCore()
    {
        HW_CP15_INVALIDATE_UNIFIED_TLB_ALLCORE();
        DataSynchronizationBarrier();
    }
    static void InvalidateTlbByMvaNoCodeModificationAllCore(KProcessAddress va)
    {
        HW_SET_CP15_INVALIDATE_UNIFIED_TLB_MVA_ALLCORE(GetAsInteger(va));
        DataSynchronizationBarrier();
    }
    static void InvalidateTlbByAsidAllCore(Bit32 asid)
    {
        HW_SET_CP15_INVALIDATE_UNIFIED_TLB_ASID_ALLCORE(asid);
        HW_CP15_FLUSH_ENTIRE_BRANCH_TARGET_CACHE_ALLCORE();
        CareInstructionConsistencyAllCore();
    }
    static void InvalidateTlbByVaAsidAllCore(Bit32 asid, KProcessAddress va)
    {
        HW_SET_CP15_INVALIDATE_UNIFIED_TLB_SINGLE_ALLCORE(GetAsInteger(va) | asid);
        HW_CP15_FLUSH_ENTIRE_BRANCH_TARGET_CACHE_ALLCORE();
        CareInstructionConsistency();
    }

    static void InvalidateTlbByAsidBeforeSwitchProcess(Bit32 asid)
    {
        HW_SET_CP15_INVALIDATE_UNIFIED_TLB_ASID(asid);
        // データ同期バリアを外部で行う必要あり
        // SwitchProcess が呼ばれることを前提に分岐キャッシュフラッシュを行わない
        // SwitchProcess が呼ばれることを前提に命令メモリバリアを行わない
    }
    static Bit32 GetDataFaultAddress()
    {
        Bit32 value;
        HW_GET_CP15_DATA_FAULT_ADDRESS(value);
        return value;
    }
    static Bit32 GetDataFaultStatus()
    {
        Bit32 value;
        HW_GET_CP15_DATA_FAULT_STATUS(value);
        return value;
    }
    static Bit32 GetInstructionFaultAddress()
    {
        Bit32 value;
        HW_GET_CP15_INSTRUCTION_FAULT_ADDRESS(value);
        return value;
    }
    static Bit32 GetInstructionFaultStatus()
    {
        Bit32 value;
        HW_GET_CP15_INSTRUCTION_FAULT_STATUS(value);
        return value;
    }
    static bool GetPhysicalAddressOfCurrentSpace(
            KPhysicalAddress* pPhysicalAddress, KVirtualAddress va, bool privilege = false)
    {
        Bit32 pa;
        Bit32 page = (GetAsInteger(va) & 0xFFFFF000);

        if (privilege)
        {
            HW_SET_CP15_VA_TO_PA_PRIVILEGED_WRITE(page);
        }
        else
        {
            HW_SET_CP15_VA_TO_PA_USER_WRITE(page);
        }
        InstructionMemoryBarrier();
#if defined(NN_BUILD_CONFIG_CPU_CORTEX_A15) || defined(NN_BUILD_CONFIG_CPU_CORTEX_A7)
        Bit32 paHigh;
        HW_GET_CP15_PA(pa, paHigh);
#else
        HW_GET_CP15_PA(pa);
#endif

        if ((pa & 0x1) == 0)
        {
            // LPAE or SuperSectionでない場合
            if ((pa & 0x800) || !(pa & 0x2))
            {
                *pPhysicalAddress = KPhysicalAddress((pa & 0xFFFFF000) | (GetAsInteger(va) & 0x00000FFF));
            }
            else
            {
                *pPhysicalAddress = KPhysicalAddress((pa & 0xFF000000) | (GetAsInteger(va) & 0x00FFFFFF));
            }
            return true;
        }
        else
        {
            return false;
        }
    }

    static bool CanAtomicAccess(KProcessAddress va, bool privilege = false)
    {
        Bit32 pa;
        if (privilege)
        {
            HW_SET_CP15_VA_TO_PA_PRIVILEGED_WRITE((GetAsInteger(va) & 0xFFFFF000));
        }
        else
        {
            HW_SET_CP15_VA_TO_PA_USER_WRITE((GetAsInteger(va) & 0xFFFFF000));
        }
        InstructionMemoryBarrier();
#if defined(NN_BUILD_CONFIG_CPU_CORTEX_A15) || defined(NN_BUILD_CONFIG_CPU_CORTEX_A7)
        Bit32 paHigh;
        HW_GET_CP15_PA(pa, paHigh);
#else
        HW_GET_CP15_PA(pa);
#endif

        if ((pa & 0x1) == 0)
        {
            // (((pa >> 4) & 0x7) == 0x7) || (((pa >> 4) & 0x7) == 0x5)
            return (((pa >> 4) & 0x5) == 0x5);
        }
        else
        {
            return false;
        }
    }


    static void DataSynchronizationBarrier()
    {
        asm volatile("dsb");
    }

    static void DataMemoryBarrier()
    {
        asm volatile("dmb");
    }

    static void InstructionMemoryBarrier()
    {
        asm volatile("isb");
    }

    static uint32_t GetCycleCounter()
    {
        uint32_t x;
        HW_GET_CP15_PMCR(x);
        return x;
    }

    static uint32_t GetPerformanceCounter(int n)
    {
        uint32_t x = 0;
        if (n < NUM_PERFORMANCE_COUNTER)
        {
            Bit32 psr;
            asm volatile("mrs     %0, cpsr":"=r"(psr)::);
            asm volatile("cpsid i");
            HW_SET_CP15_PMSELR(n);
            HW_GET_CP15_PMXEVCNTR(x);
            asm volatile("msr     cpsr_c, %0"::"r"(psr):);
        }
        return x;
    }

    static void EnablePerformanceCounter()
    {
        HW_SET_CP15_PMCR(0);
        DataSynchronizationBarrier();
        InstructionMemoryBarrier();
        switch (NUM_PERFORMANCE_COUNTER)
        {
        default:
        case 6:
            {
                HW_SET_CP15_PMSELR(5);
                HW_SET_CP15_PMXEVTYPER(HW_PMXEVTYPER(1, 0, HW_PM_TYPE_CACHE_REFILL));
                HW_SET_CP15_PMCNTENSET(1ul << 5);
            } NN_FALL_THROUGH;
        case 5:
            {
                HW_SET_CP15_PMSELR(4);
                HW_SET_CP15_PMXEVTYPER(HW_PMXEVTYPER(1, 1, HW_PM_TYPE_CACHE_REFILL));
                HW_SET_CP15_PMCNTENSET(1ul << 4);
            } NN_FALL_THROUGH;
        case 4:
            {
                HW_SET_CP15_PMSELR(3);
                HW_SET_CP15_PMXEVTYPER(HW_PMXEVTYPER(1, 0, HW_PM_TYPE_INST));
                HW_SET_CP15_PMCNTENSET(1ul << 3);
            } NN_FALL_THROUGH;
        case 3:
            {
                HW_SET_CP15_PMSELR(2);
                HW_SET_CP15_PMXEVTYPER(HW_PMXEVTYPER(1, 1, HW_PM_TYPE_INST));
                HW_SET_CP15_PMCNTENSET(1ul << 2);
            } NN_FALL_THROUGH;
        case 2:
            {
                HW_SET_CP15_PMSELR(1);
                HW_SET_CP15_PMXEVTYPER(HW_PMXEVTYPER(1, 0, HW_PM_TYPE_CPU_CYCLES));
                HW_SET_CP15_PMCNTENSET(1ul << 1);
            } NN_FALL_THROUGH;
        case 1:
            {
                HW_SET_CP15_PMSELR(0);
                HW_SET_CP15_PMXEVTYPER(HW_PMXEVTYPER(1, 1, HW_PM_TYPE_CPU_CYCLES));
                HW_SET_CP15_PMCNTENSET(1ul << 0);
            } NN_FALL_THROUGH;
        case 0:
            {
                HW_SET_CP15_PMSELR(31);
                HW_SET_CP15_PMXEVTYPER(HW_PMXEVTYPER(1, 1, HW_PM_TYPE_CPU_CYCLES));
                HW_SET_CP15_PMCNTENSET(1ul << 0);
            }
            break;
        }
        DataSynchronizationBarrier();
        InstructionMemoryBarrier();
        HW_SET_CP15_PMCR((1ul << 2) | (1ul << 1) | (1ul << 0));
    }
    static void GetCycleCounter(Bit64* pOut);
    static void GetPerformanceCounter(Bit32* pOut, int n);
    static uintptr_t GetCurrentValue()
    {
        Bit32 x;
        HW_GET_CP15_THREAD_ID_PRIVILEGED_ONLY(x);
        return x;
    }
    static void SetCurrentValue(uintptr_t value)
    {
        HW_SET_CP15_THREAD_ID_PRIVILEGED_ONLY(value);
    }
};

}
}}

