﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/nn_Common.h>
#include <nn/svc/svc_Kernel.h>
#include <nn/nn_BitTypes.h>

#include "../../kern_Platform.h"
#include "kern_ExceptionHandler.h"
#include "kern_ExceptionHandlerPrint.h"
#include "../ARM/kern_RegisterAccess.h"
#include "../ARM/kern_ExceptionContext.h"
#include "../../kern_DebugSelect.h"
#include "../../kern_KProcess.h"
#include "../../kern_KThread.h"
#include "../../kern_MemoryCopySelect.h"
#include "../../kern_DpcManager.h"
#include "kern_SystemControl.h"
#include "../../kern_KTrace.h"
#include <cstring>

namespace nn { namespace kern { namespace svc {
void RestoreContext(uintptr_t sp);
}}}

namespace nn { namespace kern { namespace ARMv7A {

namespace
{
    Bit32 GetCode(const ARM::ExceptionContext* pContext)
    {
        Bit32 code = 0;
        if (pContext->cpsr & HW_PSR_THUMB_STATE)
        {
            code = *reinterpret_cast<Bit16*>(pContext->pc & ~0x1);
            // Thumb-2 の場合は32bit取得する
            switch ((code >> 11) & 0x1F)
            {
            case 0x1D:
            case 0x1E:
            case 0x1F:
                {
                    code = (code << 16) | *reinterpret_cast<Bit16*>((pContext->pc & ~0x1) + 2);
                }
                break;
            default:
                break;
            }
        }
        else
        {
            code = *reinterpret_cast<Bit32*>(pContext->pc);
        }
        return code;
    }

    void CommonExceptionHandlerForReturnFromException(const nn::kern::ARM::ExceptionContext* pSwiContext, Bit32 param1, Bit32 param2, Bit32 param3)
    {
        Bit32 fs = 0;
        Bit32 fa = 0;
        ExceptionType type = EXCEPTION_TYPE_UNKNOWN;
        switch (param1)
        {
        case nn::svc::DebugException_AccessViolationInstruction:
            {
                type = EXCEPTION_TYPE_PABT;
                fa = param2;
                fs = param3;
            }
            break;
        case nn::svc::DebugException_AccessViolationData:
            {
                type = EXCEPTION_TYPE_DABT;
                fa = param2;
                fs = param3;
            }
            break;
        case nn::svc::DebugException_UndefinedInstruction:
            {
                type = EXCEPTION_TYPE_UNDEF;
            }
            break;
        case nn::svc::DebugException_UndefinedSystemCall:
            {
                type = EXCEPTION_TYPE_INVALID_SVC;
            }
            break;
        default:
            break;
        }
        CommonExceptionHandlerDebug(pSwiContext, type, fs, fa);
        KDebug::PrintBacktrace();
    }

    void HandleUserModeException(ARM::ExceptionContext* pContext, ExceptionType type, Bit32 fa, Bit32 fs, Bit32 code)
    {
        KProcess& process = GetCurrentProcess();
        bool isDemandLoadException = false;

        switch (type)
        {
        case EXCEPTION_TYPE_PABT:
        case EXCEPTION_TYPE_DABT:
            {
                // デマンドロード対象かチェック
                KMemoryInfo mi;
                nn::svc::PageInfo pi;
                if (process.GetPageTable().QueryInfo(&mi, &pi, fa).IsSuccess())
                {
                    if (mi.state == KMemoryState_Code && (mi.permission & KMemoryPermission_UserRead) != KMemoryPermission_UserRead)
                    {
                        isDemandLoadException = true;
                    }
                }
            }
            break;
        default:
            break;
        }

        if ((isDemandLoadException || KTargetSystem::IsUserExceptionHandlerEnabled()) &&
                !(type == EXCEPTION_TYPE_UNDEF && process.IsAttachedByDebugger() && KDebug::IsBreakInstruction(code, pContext->cpsr)))
        {
            if (process.EnterUserException())
            {
                Bit32 cause = 0;

                nn::svc::aarch32::ProcessLocalRegion* pProcLocal = GetTypedPointer<nn::svc::aarch32::ProcessLocalRegion>(process.GetProcessLocalRegionAddr());
                nn::svc::aarch32::ExceptionInfo* pInfo = &pProcLocal->info;
                for (size_t i = 0; i < sizeof(pInfo->r) / sizeof(pInfo->r[0]); i++)
                {
                    pInfo->r[i] = pContext->r[i];
                }
                pInfo->sp = pContext->sp_usr;
                pInfo->lr = pContext->lr_usr;
                pInfo->pc = pContext->pc;
                pInfo->flags = 0;
                pInfo->status.on32.cpsr = (pContext->cpsr & HW_PSR_PL0_MASK);
                pInfo->status.on32.fsr = fs;
                pInfo->status.on32.far = fa;
                pInfo->status.on32.fpexc = 0;
                pInfo->status.on32.fpinst = 0;
                pInfo->status.on32.fpinst2 = 0;

                // backup
                switch (type)
                {
                case EXCEPTION_TYPE_PABT:
                    {
                        GetCurrentThread().SaveDebugParams(nn::svc::DebugException_AccessViolationInstruction, fa, fs);
                        cause = nn::svc::ExceptionType_InstructionAbort;
                    }
                    break;
                case EXCEPTION_TYPE_DABT:
                    {
                        GetCurrentThread().SaveDebugParams(nn::svc::DebugException_AccessViolationData, fa, fs);
                        cause = nn::svc::ExceptionType_DataAbort;
                    }
                    break;
                case EXCEPTION_TYPE_UNDEF:
                    {
                        GetCurrentThread().SaveDebugParams(nn::svc::DebugException_UndefinedInstruction, fa, code);
                        cause = nn::svc::ExceptionType_UndefinedInstruction;
                    }
                    break;
                case EXCEPTION_TYPE_INVALID_SVC:
                    {
                        GetCurrentThread().SaveDebugParams(nn::svc::DebugException_UndefinedSystemCall, fa, code & 0xFF);
                        cause = nn::svc::ExceptionType_InvalidSystemCall;
                    }
                    break;
                default:
                    break;
                }

                pContext->pc = GetAsInteger(process.GetEntryPoint());
                pContext->sp_usr = ((reinterpret_cast<uintptr_t>(pProcLocal->work) + sizeof(pProcLocal->work)) & ~(0x8ul - 1));
                pContext->r[0] = cause;
                pContext->r[1] = GetAsInteger(process.GetProcessLocalRegionAddr() + offsetof(nn::svc::aarch32::ProcessLocalRegion, info));
                pContext->cpsr = HW_PSR_ARM_STATE | HW_PSR_DATA_LITTLE_ENDIAN | HW_PSR_USR_MODE;

                return;
            }
        }

        {
            Result result = ResultSuccess();
            nn::svc::DebugException except = nn::svc::DebugException_AccessViolationData;
            uintptr_t param2 = 0;
            uintptr_t param3 = 0;

            switch (type)
            {
            case EXCEPTION_TYPE_PABT:
                {
                    except = nn::svc::DebugException_AccessViolationInstruction;
                    param2 = fa;
                }
                break;
            case EXCEPTION_TYPE_DABT:
                {
                    except = nn::svc::DebugException_AccessViolationData;
                    param2 = fa;
                }
                break;
            case EXCEPTION_TYPE_UNDEF:
                {
                    except = nn::svc::DebugException_UndefinedInstruction;
                    param2 = fa;
                    param3 = code;
                }
                break;
            case EXCEPTION_TYPE_INVALID_SVC:
                {
                    except = nn::svc::DebugException_UndefinedSystemCall;
                    param2 = fa;
                    param3 = code & 0xFF;
                }
                break;
            default:
                break;
            }

            result = KDebug::OnDebugEvent(nn::svc::DebugEvent_Exception, except, param2, param3);

            if (result <= nn::svc::ResultStopProcessingException())
            {
                return;
            }

            NN_KERN_FATAL_LOG("Exception occurred. %016lx\n", process.GetProgramId());
            CommonExceptionHandlerDebug(pContext, type, fs, fa);
            KDebug::PrintBacktrace();

            NN_KERN_KTRACE_STOP();
            if (!(result <= nn::svc::ResultNotHandled()))
            {
                if (process.EnterJitDebug(nn::svc::DebugEvent_Exception, except, param2, param3))
                {
                    NN_KERN_KTRACE_DELAYED_DUMP_WITH_ADDITIONAL_INFO();
                    return;
                }
            }
        }

        NN_KERN_KTRACE_DUMP_WITH_ADDITIONAL_INFO();
        process.Exit();
    }

    void HandleException(ARM::ExceptionContext* pContext, ExceptionType type)
    {
        // INFO: 割り込み禁止状態
        NN_KERN_ASSERT(!KInterruptManager::IsInterruptEnabled());

        const bool inUserMode = ((pContext->cpsr & HW_PSR_CPU_MODE_MASK) == HW_PSR_USR_MODE);
        Bit32 code = 0;
        Bit32 fa = 0;
        Bit32 fs = 0;

        switch (type)
        {
        case EXCEPTION_TYPE_PABT:
            {
                fs = KCPU::GetInstructionFaultStatus();
                fa = KCPU::GetInstructionFaultAddress();
            }
            break;
        case EXCEPTION_TYPE_DABT:
            {
                fs = KCPU::GetDataFaultStatus();
                fa = KCPU::GetDataFaultAddress();
            }
            break;
        case EXCEPTION_TYPE_UNDEF:
            {
                code = GetCode(pContext);
                fa = (pContext->pc & ~1);
            }
            break;
        case EXCEPTION_TYPE_INVALID_SVC:
            {
                if (pContext->cpsr & HW_PSR_THUMB_STATE)
                {
                    pContext->pc -= 2;
                }
                else
                {
                    pContext->pc -= 4;
                }
                code = GetCode(pContext);
                fa = (pContext->pc & ~1);
            }
            break;
        default:
            break;
        }

        if (inUserMode)
        {
            KEnableInterrupt interruptEnabled;

            HandleUserModeException(pContext, type, fa, fs, code);

        }
        else
        {
            if (KTargetSystem::IsDevelopmentHardware())
            {
                CommonExceptionHandlerDebug(pContext, type, fs, fa);
            }
            NN_KERNEL_PANIC("");
        }

        // INFO: 割り込み禁止状態
        NN_KERN_ASSERT(!KInterruptManager::IsInterruptEnabled());

        while (GetCurrentThread().IsDpcRegistered())
        {
            DpcManager::DpcHandler();
        }
    }
}

void HandleDataAbort(ARM::ExceptionContext* pContext)
{
    HandleException(pContext, EXCEPTION_TYPE_DABT);
}

void HandleInstructionAbort(ARM::ExceptionContext* pContext)
{
    HandleException(pContext, EXCEPTION_TYPE_PABT);
}

void HandleUndefinedInstruction(ARM::ExceptionContext* pContext)
{
    HandleException(pContext, EXCEPTION_TYPE_UNDEF);
}

void HandleInvalidSvc(ARM::ExceptionContext* pContext)
{
    HandleException(pContext, EXCEPTION_TYPE_INVALID_SVC);
}

void PanicByImpreciseDataAbort()
{
    NN_KERNEL_PANIC("Imprecise data abort.");
}

void PanicByTooSmallStack()
{
    NN_KERNEL_PANIC("svc stack is too small.");
}

void ReturnFromException(Result resultValue)
{
    KThread* pCurrent = &GetCurrentThread();

    nn::kern::ARM::ExceptionContext* swiContext = reinterpret_cast<nn::kern::ARM::ExceptionContext*>(
            reinterpret_cast<uintptr_t>(pCurrent->GetKernelStackBottom()) - sizeof(KThread::ParamsOnStack)) - 1;
    KProcess& process = GetCurrentProcess();

    nn::svc::aarch32::ProcessLocalRegion* pProcLocal = GetTypedPointer<nn::svc::aarch32::ProcessLocalRegion>(process.GetProcessLocalRegionAddr());
    const nn::svc::aarch32::ExceptionInfo info = pProcLocal->info;  // コピーする

    if (process.LeaveUserException())
    {
        swiContext->r[0]        = info.r[0];
        swiContext->r[1]        = info.r[1];
        swiContext->r[2]        = info.r[2];
        swiContext->r[3]        = info.r[3];
        swiContext->r[4]        = info.r[4];
        swiContext->r[5]        = info.r[5];
        swiContext->r[6]        = info.r[6];
        swiContext->r[7]        = info.r[7];
        swiContext->sp_usr      = info.sp;
        swiContext->lr_usr      = info.lr;
        swiContext->pc          = info.pc;
        swiContext->cpsr        = ((info.status.on32.cpsr & HW_PSR_PL0_MASK) | (swiContext->cpsr & ~HW_PSR_PL0_MASK));
        swiContext->write       = 1;

        if (resultValue.IsSuccess())
        {
            nn::kern::svc::RestoreContext(reinterpret_cast<uintptr_t>(swiContext));
        }
        else
        {
            uintptr_t param1;
            uintptr_t param2;
            uintptr_t param3;
            GetCurrentThread().RestoreDebugParams(&param1, &param2, &param3);
            Result result;

            result = KDebug::OnDebugEvent(nn::svc::DebugEvent_Exception, param1,  param2,  param3);

            if (result <= nn::svc::ResultStopProcessingException())
            {
                nn::kern::svc::RestoreContext(reinterpret_cast<uintptr_t>(swiContext));
            }

            CommonExceptionHandlerForReturnFromException(swiContext, param1, param2, param3);

            NN_KERN_KTRACE_STOP();

            // デバッガでアタッチされているが、ハンドリングされなかった
            if (!(result <= nn::svc::ResultNotHandled()))
            {
                if (GetCurrentProcess().EnterJitDebug(nn::svc::DebugEvent_Exception, static_cast<nn::svc::DebugException>(param1), param2,  param3))
                {
                    NN_KERN_KTRACE_DELAYED_DUMP_WITH_ADDITIONAL_INFO();
                    nn::kern::svc::RestoreContext(reinterpret_cast<uintptr_t>(swiContext));
                }
            }

            if (resultValue <= nn::svc::ResultDebug())
            {
                nn::kern::svc::RestoreContext(reinterpret_cast<uintptr_t>(swiContext));
            }
        }
    }
    else
    {
        Bit32 code = 0;
        Bit32 fa = 0;

        if (swiContext->cpsr & HW_PSR_THUMB_STATE)
        {
            swiContext->pc -= 2;
        }
        else
        {
            swiContext->pc -= 4;
        }
        code = GetCode(swiContext);
        fa = (swiContext->pc & ~1);

        HandleUserModeException(swiContext, EXCEPTION_TYPE_INVALID_SVC, fa, 0, code);
        nn::kern::svc::RestoreContext(reinterpret_cast<uintptr_t>(swiContext));
    }

    NN_KERN_FATAL_LOG("Exception occurred. %016lx\n", GetCurrentProcess().GetProgramId());
    NN_KERN_KTRACE_DUMP_WITH_ADDITIONAL_INFO();
    GetCurrentProcess().Exit();
}

}}}

