﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#include <nn/TargetConfigs/build_Compiler.h>
#include <nn/TargetConfigs/build_Fpu.h>
#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include "../../kern_Platform.h"
#include "kern_KContext.h"
#include "../../kern_Kernel.h"
#include "../../kern_KThread.h"
#include "../../kern_KProcess.h"
#include "../../kern_KScheduler.h"
#include "kern_ExceptionContext.h"
#include "kern_UserContext.h"

namespace nn { namespace kern {
namespace ARM64 {

void GetUserContext(nn::svc::ThreadContext* pUserContext, const KThread* pThread)
{
    const ExceptionContext* pExcContext = GetExceptionContext(pThread);
    const KContext* pContext = pThread->GetContext();
    NN_KERN_ASSERT(KScheduler::IsSchedulerLocked());
    NN_KERN_ASSERT(pThread->IsSuspended());
    NN_KERN_ASSERT(pThread->GetParentPointer());

    if (pThread->GetParent().Is64Bit())
    {
        pUserContext->fp = pExcContext->x[29];
        pUserContext->lr = pExcContext->x[30];
        pUserContext->sp = pExcContext->sp;
        pUserContext->pc = pExcContext->pc;
        pUserContext->pstate = (pExcContext->psr & HW_PSR_EL0_MASK);

        if (pThread->IsCallingSvc())
        {
            for (size_t i = 19; i < 29; i++)
            {
                pUserContext->r[i] = pExcContext->x[i];
            }
            if (pExcContext->write == 0)
            {
                pUserContext->pc -= 4;
            }
        }
        else
        {
            for (size_t i = 0; i < 29; i++)
            {
                pUserContext->r[i] = pExcContext->x[i];
            }
        }
        pUserContext->tpidr = pExcContext->tpidr;
        const Bit128 *fpr = pContext->GetFpuRegisters();
        for (size_t i = 0; i < pContext->GetNumOfVfpRegisters(); i++)
        {
            pUserContext->v[i] = fpr[i];
        }
    }
    else
    {
        pUserContext->pc =  static_cast<uint32_t>(pExcContext->pc);
        pUserContext->pstate = (pExcContext->psr & HW_PSR_EL0_MASK);

        for (size_t i = 0; i < 15; i++)
        {
            pUserContext->r[i] = static_cast<uint32_t>(pExcContext->x[i]);
        }

        if (pThread->IsCallingSvc())
        {
            if (pExcContext->write == 0)
            {
                pUserContext->pc -= ((pExcContext->psr & HW_PSR_THUMB_STATE)? 2: 4);
            }
        }
        pUserContext->tpidr = pExcContext->tpidr & 0xFFFFFFFF;
        const Bit128 *fpr = pContext->GetFpuRegisters();
        for (size_t i = 0; i < pContext->GetNumOfVfpRegisters() / 2; i++)
        {
            pUserContext->v[i] = fpr[i];
        }
    }

    pUserContext->fpcr = pContext->GetFpcr();
    pUserContext->fpsr = pContext->GetFpsr();
}

uintptr_t GetUserStackPointer(const KThread* pThread)
{
    const ExceptionContext* pExcContext = GetExceptionContext(pThread);

    if (pThread->GetParent().Is64Bit())
    {
        return pExcContext->sp;
    }
    else
    {
        return static_cast<uint32_t>(pExcContext->x[13]);
    }
}
}
}}

