﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   タイマーイベントに関する C 言語用 API の宣言
 */

#pragma once

#include <nn/os/os_Config.h>

#include <nnc/nn_Common.h>
#include <nnc/os/os_Macro.h>
#include <nnc/os/os_EventCommon.h>
#include <nnc/os/os_TimerEventTypes.h>
#include <nnc/os/os_MultipleWaitTypes.h>
#include <nnc/os/detail/os_MacroImpl.h>

//--------------------------------------------------------------------------
//  C 向けの宣言
//--------------------------------------------------------------------------

//! @name タイマーイベント関連 C 言語用 API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   nnosTimerEventType オブジェクトを初期化します。
 *
 * @details
 *  詳細は nn::os::InitializeTimerEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosInitializeTimerEvent(nnosTimerEventType* event, nnosEventClearMode clearMode);


//--------------------------------------------------------------------------
/**
 * @brief   nnosTimerEventType オブジェクトを破棄します。
 *
 * @details
 *  詳細は nn::os::FinalizeTimerEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosFinalizeTimerEvent(nnosTimerEventType* event);


//--------------------------------------------------------------------------
/**
 * @brief   nnosTimerEventType オブジェクトのワンショットタイマー動作を開始します。
 *
 * @details
 *  詳細は nn::os::StartOneShotTimerEvent() を参照して下さい。
 *  本 API では、タイマーの時間をナノ秒単位で指定します。
 *
 */
NN_OS_EXTERN_C void nnosStartOneShotTimerEvent(nnosTimerEventType* event, int64_t first);


//--------------------------------------------------------------------------
/**
 * @brief   nnosTimerEventType オブジェクトの周期タイマー動作を開始します。
 *
 * @details
 *  詳細は nn::os::StartPeriodicTimerEvent() を参照して下さい。
 *  本 API では、タイマーの時間をナノ秒単位で指定します。
 *
 */
NN_OS_EXTERN_C void nnosStartPeriodicTimerEvent(nnosTimerEventType* event, int64_t first, int64_t interval);


//--------------------------------------------------------------------------
/**
 * @brief   nnosTimerEventType オブジェクトのタイマー動作を停止します。
 *
 * @details
 *  詳細は nn::os::StopTimerEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosStopTimerEvent(nnosTimerEventType* event);


//--------------------------------------------------------------------------
/**
 * @brief   nnosTimerEventType オブジェクトがシグナル状態になるまで待機します。
 *
 * @details
 *  詳細は nn::os::WaitTimerEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosWaitTimerEvent(nnosTimerEventType* event);


//--------------------------------------------------------------------------
/**
 * @brief   nnosTimerEventType オブジェクトがシグナル状態かをポーリングします。
 *
 * @details
 *  詳細は nn::os::TryWaitTimerEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C bool nnosTryWaitTimerEvent(nnosTimerEventType* event);


//--------------------------------------------------------------------------
/**
 * @brief   nnosTimerEventType オブジェクトを即座にシグナル状態にします。
 *
 * @details
 *  詳細は nn::os::SignalTimerEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosSignalTimerEvent(nnosTimerEventType* event);


//--------------------------------------------------------------------------
/**
 * @brief   nnosTimerEventType オブジェクトを非シグナル状態にクリアします。
 *
 * @details
 *  詳細は nn::os::ClearTimerEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosClearTimerEvent(nnosTimerEventType* event);

//! @}


//! @name 多重待ち関連 API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   multiWaitHolder を初期化し、タイマーイベントと関連付けます。
 *
 * @details
 *  詳細は nn::os::InitializeMultiWaitHolder(MultiWaitHolderType* multiWaitHolder, TimerEventType* timerEvent) を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosInitializeMultiWaitHolderForTimerEvent(nnosMultiWaitHolderType* multiWaitHolder, nnosTimerEventType* timerEvent);

//! @}

