﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   SystemEvent に関する C 言語用 API の宣言
 */

#pragma once

#include <nnc/nn_Macro.h>
#include <nnc/nn_Result.h>
#include <nnc/os/os_NativeHandleTypes.h>
#include <nnc/os/os_EventCommon.h>
#include <nnc/os/os_MultipleWaitTypes.h>


//--------------------------------------------------------------------------
//  C 向けの宣言
//--------------------------------------------------------------------------

//! @name システムイベント関連 API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   SystemEventType オブジェクトを初期化します。
 *
 * @details
 *  詳細は nn::os::CreateSystemEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C nnResult nnosCreateSystemEvent(nnosSystemEventType* event, nnosEventClearMode clearMode, bool interProcess);


//--------------------------------------------------------------------------
/**
 * @brief   指定されたハンドルを使って SystemEventType オブジェクトをプロセス間同期用として初期化します。
 *
 * @details
 *  詳細は nn::os::AttachSystemEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosAttachSystemEvent(nnosSystemEventType* event, nnosNativeHandle readableHandle, bool isReadableHandleManaged, nnosNativeHandle writableHandle, bool isWritableHandleManaged, nnosEventClearMode clearMode);


//--------------------------------------------------------------------------
/**
 * @brief   指定された readableHandle を使って SystemEventType オブジェクトをプロセス間同期用として初期化します。
 *
 * @details
 *  詳細は nn::os::AttachReadableHandleToSystemEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosAttachReadableHandleToSystemEvent(nnosSystemEventType* event, nnosNativeHandle readableHandle, bool isReadableHandleManaged, nnosEventClearMode clearMode);


//--------------------------------------------------------------------------
/**
 * @brief   指定された writableHandle を使って SystemEventType オブジェクトをプロセス間同期用として初期化します。
 *
 * @details
 *  詳細は nn::os::AttachWritableHandleToSystemEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosAttachWritableHandleToSystemEvent(nnosSystemEventType* event, nnosNativeHandle writableHandle, bool isWritableHandleManaged, nnosEventClearMode clearMode);


//--------------------------------------------------------------------------
/**
 * @brief    SystemEventType オブジェクトから ReadableEvent ハンドルを切り離します。
 *
 * @details
 *  詳細は nn::os::DetachReadableHandleOfSystemEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C nnosNativeHandle nnosDetachReadableHandleOfSystemEvent(nnosSystemEventType* event);


//--------------------------------------------------------------------------
/**
 * @brief   SystemEventType オブジェクトから WritableEvent ハンドルを切り離します。
 *
 * @details
 *  詳細は nn::os::DetachWritableHandleOfSystemEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C nnosNativeHandle nnosDetachWritableHandleOfSystemEvent(nnosSystemEventType* event);


//--------------------------------------------------------------------------
/**
 * @brief   SystemEventType オブジェクトを破棄します。
 *
 * @details
 *  詳細は nn::os::DestroySystemEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosDestroySystemEvent(nnosSystemEventType* event);


//--------------------------------------------------------------------------
/**
 * @brief   システムイベントがシグナル状態になるまで待機します。
 *
 * @details
 *  詳細は nn::os::WaitSystemEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosWaitSystemEvent(nnosSystemEventType* event);


//--------------------------------------------------------------------------
/**
 * @brief   システムイベントがシグナル状態かをポーリングします。
 *
 * @details
 *  詳細は nn::os::TryWaitSystemEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C bool nnosTryWaitSystemEvent(nnosSystemEventType* event);


//--------------------------------------------------------------------------
/**
 * @brief   システムイベントがシグナル状態になるまで待機します。
 *
 * @details
 *  詳細は nn::os::TimedWaitSystemEvent() を参照して下さい。@n
 *  本 API では、タイムアウト時間をナノ秒単位で指定します。
 *
 */
NN_OS_EXTERN_C bool nnosTimedWaitSystemEvent(nnosSystemEventType* event, int64_t timeout);


//--------------------------------------------------------------------------
/**
 * @brief   システムイベントをシグナル状態にセットします。
 *
 * @details
 *  詳細は nn::os::SignalSystemEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosSignalSystemEvent(nnosSystemEventType* event);


//--------------------------------------------------------------------------
/**
 * @brief   システムイベントを非シグナル状態にクリアします。
 *
 * @details
 *  詳細は nn::os::ClearSystemEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosClearSystemEvent(nnosSystemEventType* event);


//--------------------------------------------------------------------------
/**
 * @brief   SystemEventType オブジェクトの ReadableHandle を返します。
 *
 * @details
 *  詳細は nn::os::GetReadableHandleOfSystemEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C nnosNativeHandle nnosGetReadableHandleOfSystemEvent(nnosSystemEventType* event);


//--------------------------------------------------------------------------
/**
 * @brief   SystemEventType オブジェクトの WritableHandle を返します。
 *
 * @details
 *  詳細は nn::os::GetWritableHandleOfSystemEvent() を参照して下さい。
 *
 */
NN_OS_EXTERN_C nnosNativeHandle nnosGetWritableHandleOfSystemEvent(nnosSystemEventType* event);

//! @}


//! @name 多重待ち関連 API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   multiWaitHolder を初期化し SystemEventType オブジェクトと関連付けます。
 *
 * @details
 *  詳細は nn::os::InitializeMultiWaitHolder(MultiWaitHolderType* multiWaitHolder, SystemEventType* event) を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosInitializeMultiWaitHolderForSystemEvent(nnosMultiWaitHolderType* multiWaitHolder, nnosSystemEventType* event);

//! @}
