﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   セマフォに関する C 言語用 API の宣言
 */

#pragma once

#include <nnc/nn_Common.h>
#include <nnc/os/os_Macro.h>
#include <nnc/os/os_SemaphoreTypes.h>
#include <nnc/os/os_MultipleWaitTypes.h>
#include <nnc/os/detail/os_MacroImpl.h>

//--------------------------------------------------------------------------
//  C 向けの宣言
//--------------------------------------------------------------------------

//! @name セマフォ関連 C 言語用 API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   nnosSemaphoreType オブジェクトを初期化します。
 *
 * @details
 *  詳細は nn::os::InitializeSemaphore() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosInitializeSemaphore(nnosSemaphoreType* semaphore, int initialCount, int maxCount);


//--------------------------------------------------------------------------
/**
 * @brief   nnosSemaphoreType オブジェクトをファイナライズします。
 *
 * @details
 *  詳細は nn::os::FinalizeSemaphore() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosFinalizeSemaphore(nnosSemaphoreType* semaphore);


//--------------------------------------------------------------------------
/**
 * @brief   nnosSemaphoreType を獲得します（Ｐ操作）
 *
 * @details
 *  詳細は nn::os::AcquireSemaphore() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosAcquireSemaphore(nnosSemaphoreType* semaphore);

//--------------------------------------------------------------------------
/**
 * @brief   nnosSemaphoreType の獲得を試みます（Ｐ操作）
 *
 * @details
 *  詳細は nn::os::TryAcquireSemaphore() を参照して下さい。
 *
 */
NN_OS_EXTERN_C bool nnosTryAcquireSemaphore(nnosSemaphoreType* semaphore);

//--------------------------------------------------------------------------
/**
 * @brief   nnosSemaphoreType を時限付きで獲得を試みます（Ｐ操作）
 *
 * @details
 *  詳細は nn::os::TimedAcquireSemaphore() を参照して下さい。@n
 *  本 API では、タイムアウト時間をナノ秒単位で指定します。
 *
 */
NN_OS_EXTERN_C bool nnosTimedAcquireSemaphore(nnosSemaphoreType* semaphore, int64_t timeout);


//--------------------------------------------------------------------------
/**
 * @brief   nnosSemaphoreType を１カウントだけ返却します（Ｖ操作）
 *
 * @details
 *  詳細は @ref nn::os::ReleaseSemaphore(nn::os::SemaphoreType* semaphore) を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosReleaseSemaphore(nnosSemaphoreType* semaphore);


//--------------------------------------------------------------------------
/**
 * @brief   SemaphoreType を複数カウント返却します（Ｖ操作）
 *
 * @details
 *  詳細は @ref nn::os::ReleaseSemaphore(nn::os::SemaphoreType* semaphore, int count) を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosReleaseSemaphoreCount(nnosSemaphoreType* semaphore, int count);


//--------------------------------------------------------------------------
/**
 * @brief   SemaphoreType の現在のカウンタ値を返します
 *
 * @details
 *  詳細は nn::os::GetCurrentSemaphoreCount() を参照して下さい。
 *
 */
NN_OS_EXTERN_C int nnosGetCurrentSemaphoreCount(const nnosSemaphoreType* semaphore);

//! @}


//! @name 多重待ち関連 API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   multiWaitHolder を初期化し、semaphore と関連付けます。
 *
 * @details
 *  詳細は nn::os::InitializeMultiWaitHolder(MultiWaitHolderType* multiWaitHolder, SemaphoreType* semaphore) を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosInitializeMultiWaitHolderForSemaphore(nnosMultiWaitHolderType* multiWaitHolder, nnosSemaphoreType* semaphore);

//! @}

