﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   ミューテックスに関する C 言語用 API の宣言
 */

#pragma once

#include <nn/os/os_Config.h>

#include <nnc/nn_Common.h>
#include <nnc/os/os_Macro.h>
#include <nnc/os/os_MutexCommon.h>
#include <nnc/os/os_MutexTypes.h>
#include <nnc/os/detail/os_MacroImpl.h>

#if defined(NN_BUILD_CONFIG_OS_WIN32)
    #include <nnc/os/detail/os_MutexApi-os.win32.h>
#elif defined(NN_BUILD_CONFIG_OS_HORIZON)
    #include <nnc/os/detail/os_MutexApi-os.horizon.h>
#endif

//--------------------------------------------------------------------------
//  C 向けの宣言
//--------------------------------------------------------------------------

//! @name ミューテックス関連 C 言語用 API
//! @{

//--------------------------------------------------------------------------
/**
 * @brief   nnosMutexType オブジェクトを初期化します。
 *
 * @details
 *  詳細は nn::os::InitializeMutex() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosInitializeMutex(nnosMutexType* mutex, bool recursive, int lockLevel);


//--------------------------------------------------------------------------
/**
 * @brief   nnosMutexType オブジェクトを Finalize します。
 *
 * @details
 *  詳細は nn::os::FinalizeMutex() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosFinalizeMutex(nnosMutexType* mutex);


//--------------------------------------------------------------------------
/**
 * @brief   nnosMutexType をロックします。
 *
 * @details
 *  詳細は nn::os::LockMutex() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosLockMutex(nnosMutexType* mutex);

//--------------------------------------------------------------------------
/**
 * @brief   nnosMutexType のロック操作を試みます。
 *
 * @details
 *  詳細は nn::os::TryLockMutex() を参照して下さい。
 *
 */
NN_OS_EXTERN_C bool nnosTryLockMutex(nnosMutexType* mutex);

//--------------------------------------------------------------------------
/**
 * @brief   nnosMutexType をアンロックします。
 *
 * @details
 *  詳細は nn::os::UnlockMutex() を参照して下さい。
 *
 */
NN_OS_EXTERN_C void nnosUnlockMutex(nnosMutexType* mutex);


#if defined(NN_BUILD_FOR_DOCUMENT_GENERATION)
//--------------------------------------------------------------------------
/**
 * @brief   nnosMutexType オブジェクトを静的に初期化します。
 *
 * @param[in] recursive ミューテックスの再帰ロックの可否指定
 *
 * @details
 *  nnosMutexType オブジェクトを静的に初期化するための初期化リストに
 *  展開されるマクロです。recursive には再帰ロック可否を bool 型で指定します。
 *  ロックレベル値は 0 となります。
 *
 *  ミューテックスの詳細は nnosInitializeMutex() を参照して下さい。
 *
 *  本マクロは下記例のように、静的な nnosMutexType オブジェクトの初期化
 *  にのみ使用できます。こうして初期化したオブジェクトは nnosFinalizeMutex
 *  呼び出しによる明示的な破棄も不要です。
 *
 *  使用例：
 *      @code
 *      nnosMutexType g_StaticMutex = NN_OS_MUTEX_INITIALIZER(false);
 *      @endcode
 *
 */
#define NN_OS_MUTEX_INITIALIZER(recursive)
#endif

//! @}

