﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <atomic>

#include <nn/nn_Common.h>
#include <nn/os/os_Event.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/os/os_SdkMutex.h>
#include <nn/os/os_SystemEvent.h>
#include <nn/util/util_Optional.h>

namespace nn { namespace util {

class Cancelable
{
public:
    class Listener
    {
    public:
        virtual void Signal() NN_NOEXCEPT = 0;
    };

private:
    mutable os::SdkMutex m_Lock;
    std::atomic<bool> m_UserCancel {false};
    Listener* m_pListener {nullptr};

public:
    void Cancel() NN_NOEXCEPT;
    bool IsCanceled() const NN_NOEXCEPT;

    // TODO あとで消す
    bool IsCancelled() const NN_NOEXCEPT
    {
        return IsCanceled();
    }

    void AttachListener(Listener* pListener) NN_NOEXCEPT;
    void DetachListener() NN_NOEXCEPT;
};

class CancelInjectable
{
private:
    const Cancelable* const m_pCancelable;

protected:
    explicit CancelInjectable(const Cancelable* pCancelable) NN_NOEXCEPT;

    bool IsCanceled() const NN_NOEXCEPT;

    // TODO あとで消す
    bool IsCancelled() const NN_NOEXCEPT
    {
        return IsCanceled();
    }
};

class Executable
    : public Cancelable
{
    NN_DISALLOW_COPY(Executable);
    NN_DISALLOW_MOVE(Executable);

private:
    os::SdkMutex m_Lock;

    // 状態管理
    bool m_Initialized {false};
    os::SystemEventType m_Event;

    // 結果
    std::atomic<bool> m_Done {false};
    util::optional<Result> m_pResult {util::nullopt};

protected:
    Executable() NN_NOEXCEPT = default;
    ~Executable() NN_NOEXCEPT;

    // 各実装にて内部を記述される
    virtual Result ExecuteImpl(void* buffer, size_t bufferSize) NN_NOEXCEPT = 0;

public:
    Result Initialize() NN_NOEXCEPT;

    // ユーザー定義の処理 (ExecuteImpl) の実行
    void Execute(void* buffer, size_t bufferSize) NN_NOEXCEPT;

    // 実行完了
    os::NativeHandle GetReadableHandle() NN_NOEXCEPT;
    bool HasDone() const NN_NOEXCEPT;
    util::optional<Result> TryGetResult() const NN_NOEXCEPT;

    // Executor での排他処理用
    bool TryLock() NN_NOEXCEPT;
    void Lock() NN_NOEXCEPT;
    void Unlock() NN_NOEXCEPT;
};

class AbstractExecutor
{
private:
    os::Event m_Event;

protected:
    AbstractExecutor() NN_NOEXCEPT;
    ~AbstractExecutor() NN_NOEXCEPT = default;

    // 実行待ちリストの操作
    virtual Result AddToWaitingList(Executable* pTask) NN_NOEXCEPT = 0;
    virtual void DeleteFromWaitingList(Executable* pTask) NN_NOEXCEPT = 0;
    virtual Executable* LockAndGetWaiting() NN_NOEXCEPT = 0;

public:
    // タスク登録側
    Result Register(Executable* pTask) NN_NOEXCEPT;
    void Unregister(Executable* pTask) NN_NOEXCEPT;
    bool TryUnregister(Executable* pTask) NN_NOEXCEPT;

    // タスク実行側
    void Execute(void* buffer, size_t bufferSize) NN_NOEXCEPT;

    void InitializeMultiWaitHolder(os::MultiWaitHolderType* pMultiWaitHolder) NN_NOEXCEPT;
    bool TryWaitSignal() NN_NOEXCEPT;
    void WaitSignal() NN_NOEXCEPT;
    void ClearSignal() NN_NOEXCEPT;
};

inline void InitializeMultiWaitHolder(os::MultiWaitHolderType* pMultiWaitHolder, AbstractExecutor& obj) NN_NOEXCEPT
{
    obj.InitializeMultiWaitHolder(pMultiWaitHolder);
}

/*
    Config は次を提供する
    - static const int QueueCapacity
        タスクリストの容量
    - typename ResultOutOfQueueCapacity
        タスクリストの容量が不足した場合に Register() 関数から返る Result 型
 */
template <typename Config>
class Executor
    : public AbstractExecutor
{
    NN_DISALLOW_COPY(Executor);
    NN_DISALLOW_MOVE(Executor);

private:
    os::SdkMutex m_Lock;
    Executable* m_TaskPtrs[Config::QueueCapacity] = {};
    int m_NumTask = {0};

    // 実行待ちリストの操作
    virtual Result AddToWaitingList(Executable* pTask) NN_NOEXCEPT final NN_OVERRIDE;
    virtual void DeleteFromWaitingList(Executable* pTask) NN_NOEXCEPT final NN_OVERRIDE;
    virtual Executable* LockAndGetWaiting() NN_NOEXCEPT final NN_OVERRIDE;

public:
    Executor() NN_NOEXCEPT = default;
    ~Executor() NN_NOEXCEPT;
};

class AsyncExecution
{
protected:
    bool IsInitialized() const NN_NOEXCEPT;
    void Finalize() NN_NOEXCEPT;

private:
    Executable* m_pExecutable;
    AbstractExecutor* m_pExecutor = {nullptr};

public:
    explicit AsyncExecution(Executable* pExecutable) NN_NOEXCEPT;
    AsyncExecution(AsyncExecution&& rhs) NN_NOEXCEPT;
    ~AsyncExecution() NN_NOEXCEPT;

    AsyncExecution& operator =(AsyncExecution&& rhs) NN_NOEXCEPT;

    Result Initialize(AbstractExecutor* pExecutor) NN_NOEXCEPT;
    void Cancel() NN_NOEXCEPT;
};

}} // ~namespace nn::util

/* --------------------------------------------------------------------------------------------
    実装
 */

#include <mutex>
#include <nn/nn_Abort.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace util {
/* ------------------------------------------------------------
    Executor
 */
template <typename Config>
inline Executor<Config>::~Executor() NN_NOEXCEPT
{
    NN_ABORT_UNLESS(m_NumTask == 0, "[nn::util] ABORT: %d job(s) still exist on the queue. (internal)\n", m_NumTask);
}

#define NN_UTIL_DEFINE_EXECUTOR_METHOD(rtype, methodInfo) \
template <typename Config> \
inline rtype Executor<Config>::methodInfo NN_NOEXCEPT

NN_UTIL_DEFINE_EXECUTOR_METHOD(Result, AddToWaitingList(Executable* pTask))
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
#if !defined(NN_SDK_BUILD_RELEASE)
    for (auto i = 0; i < m_NumTask; ++ i)
    {
        NN_SDK_ASSERT(m_TaskPtrs[i]);
        NN_SDK_ASSERT(m_TaskPtrs[i] != pTask, "Duplicated registration of task (internal)");
    }
#endif

    NN_RESULT_THROW_UNLESS(m_NumTask + 1 <= std::extent<decltype(m_TaskPtrs)>::value, typename Config::ResultOutOfQueueCapacity());
    NN_SDK_ASSERT(m_TaskPtrs[m_NumTask] == nullptr);
    m_TaskPtrs[m_NumTask ++] = pTask;
    NN_RESULT_SUCCESS;
}
NN_UTIL_DEFINE_EXECUTOR_METHOD(void, DeleteFromWaitingList(Executable* pTask))
{
    /* NOTE: 本関数は 1 回あるいは 2 回呼ばれる。
    - 1 回: タスクの実行開始前に、タスクが Unregister される場合
    - 2 回: スレッド A でタスクを処理中/後に、スレッド B から Unregister が呼ばれる場合
    */
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    NN_SDK_REQUIRES(m_NumTask >= 0);
    for (auto i = 0; i < m_NumTask; ++ i)
    {
        NN_SDK_ASSERT(m_TaskPtrs[i] != nullptr);
        if (m_TaskPtrs[i] == pTask)
        {
            for (auto j = i; j < m_NumTask - 1; ++ j)
            {
                NN_SDK_ASSERT(m_TaskPtrs[j + 1] != nullptr);
                m_TaskPtrs[j] = m_TaskPtrs[j + 1];
            }
            m_TaskPtrs[m_NumTask - 1] = nullptr;
            -- m_NumTask;
            return;
        }
    }
}
NN_UTIL_DEFINE_EXECUTOR_METHOD(Executable*, LockAndGetWaiting())
{
    std::lock_guard<decltype(m_Lock)> lock(m_Lock);
    for (auto i = 0; i < m_NumTask; ++ i)
    {
        NN_SDK_ASSERT(m_TaskPtrs[i] != nullptr);
        auto pTask = m_TaskPtrs[i];
        if (pTask->TryLock())
        {
            // Executor の利用者が 1 の限り、完了済みタスクがリストに残ることはない。
            NN_SDK_ASSERT(!pTask->HasDone());
            return pTask;
        }
    }
    return nullptr;
}

#undef NN_UTIL_DEFINE_EXECUTOR_METHOD

}} // ~namespace nn::util
