﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/TargetConfigs/build_Cpu.h>
#include <nn/nn_Common.h>
#include <nn/nn_BitTypes.h>
#include <nn/nn_StaticAssert.h>

namespace nn { namespace svc {

typedef uint32_t    UIntPtr32;
typedef uint32_t    Size32;
typedef uint64_t    UIntPtr64;
typedef uint64_t    Size64;
typedef Bit64       ProgramId;
typedef Bit64       ProgramId;
typedef Bit32       RegType32;
typedef Bit64       RegType64;

enum MemoryState: uint32_t
{
    MemoryState_Free,
    MemoryState_Io,
    MemoryState_Static,
    MemoryState_Code,
    MemoryState_CodeData,
    MemoryState_Normal,
    MemoryState_Shared,
    MemoryState_Alias,
    MemoryState_AliasCode,
    MemoryState_AliasCodeData,
    MemoryState_Ipc,
    MemoryState_Stack,
    MemoryState_ThreadLocal,
    MemoryState_Transfered,
    MemoryState_SharedTransfered,
    MemoryState_SharedCode,
    MemoryState_Inaccessible,
    MemoryState_NonSecureIpc,
    MemoryState_NonDeviceIpc,
    MemoryState_Kernel,
    MemoryState_GeneratedCode,
    MemoryState_CodeOut,
};

enum InterruptType: uint32_t
{
    InterruptType_Edge,
    InterruptType_Level
};

enum MemoryPermission: uint32_t
{
    MemoryPermission_None           = 0,
    MemoryPermission_Read           = (1u <<  0),
    MemoryPermission_Write          = (1u <<  1),
    MemoryPermission_Execute        = (1u <<  2),
    MemoryPermission_ReadWrite      = (MemoryPermission_Read | MemoryPermission_Write),
    MemoryPermission_ReadExecute    = (MemoryPermission_Read | MemoryPermission_Execute),
    MemoryPermission_DontCare       = (1u << 28)
};

enum MemoryAttribute: uint32_t
{
    MemoryAttribute_Locked          = (1u << 0),
    MemoryAttribute_IpcLocked       = (1u << 1),
    MemoryAttribute_DeviceShared    = (1u << 2),
    MemoryAttribute_Uncached        = (1u << 3)
};

enum BreakReason: uint32_t
{
    BreakReason_Panic,
    BreakReason_Assert,
    BreakReason_User,
    BreakReason_PreLoadDll,
    BreakReason_PostLoadDll,
    BreakReason_PreUnloadDll,
    BreakReason_PostUnloadDll,
    BreakReason_CppException,

    BreakReason_NotificationOnlyFlag = 0x80000000,
};

struct PageInfo
{
    Bit32 flags;
};

typedef uint64_t PhysicalAddress;

struct DMADeviceConfig
{
    int8_t  id;
    int8_t  accessUnit;
    int16_t burstSize;
    int16_t transUnit;
    int16_t burstStride;
    int16_t transStride;
};

struct DmaConfig
{
    int8_t          dmaNo;
    int8_t          swapSize;
    Bit8            flags;
    Bit8            padding;
    DMADeviceConfig srcDevice;
    DMADeviceConfig dstDevice;
};

enum DebugEvent: uint32_t
{
    DebugEvent_CreateProcess,
    DebugEvent_CreateThread,
    DebugEvent_ExitProcess,
    DebugEvent_ExitThread,
    DebugEvent_Exception
};

enum DebugException: uint32_t
{
    DebugException_UndefinedInstruction,
    DebugException_AccessViolationInstruction,
    DebugException_AccessViolationData,
    DebugException_DataTypeMissaligned,
    DebugException_AttachBreak,
    DebugException_BreakPoint,
    DebugException_UserBreak,
    DebugException_DebuggerBreak,
    DebugException_UndefinedSystemCall,
    DebugException_MemorySystemError,
};

enum ContinueFlag: uint32_t
{
    ContinueFlag_ExceptionHandled       = (1u << 0),
    ContinueFlag_EnableExceptionEvent   = (1u << 1),
    ContinueFlag_ContinueAll            = (1u << 2),
    ContinueFlag_ContinueOthers         = (1u << 3)
};

enum ProcessExitReason: uint32_t
{
    ProcessExitReason_ExitProcess,
    ProcessExitReason_TerminateProcess,
    ProcessExitReason_UnhandledException
};

enum ThreadExitReason: uint32_t
{
    ThreadExitReason_ExitThread,
    ThreadExitReason_TerminateThread,
    ThreadExitReason_ExitProcess,
    ThreadExitReason_TerminateProccess
};

enum BreakPointType: uint32_t
{
    BreakPointType_HardwareInstruction,
    BreakPointType_HardwareData
};

enum DebugThreadParam: uint32_t
{
    DebugThreadParam_Priority,
    DebugThreadParam_State,
    DebugThreadParam_IdealProcessor,
    DebugThreadParam_CurrentCore,
    DebugThreadParam_AffinityMask
};

enum DebugEventFlag: uint32_t
{
    DebugEventFlag_Stopped  = (1u << 0)
};

enum ThreadSuspend: uint32_t
{
    ThreadSuspend_Debug = (1u << 0),
    ThreadSuspend_User  = (1u << 1),
};

enum ThreadState: uint32_t
{
    ThreadState_Wait,
    ThreadState_Runnable,
    ThreadState_Terminated = 4,
    ThreadState_Initializing,
};

enum ContextFlag: uint32_t
{
    ContextFlag_General     = (1u << 0),
    ContextFlag_Control     = (1u << 1),
    ContextFlag_Fpu         = (1u << 2),
    ContextFlag_FpuControl  = (1u << 3)
};

#if defined(NN_BUILD_CONFIG_CPU_ARM_V8A)
struct ThreadContext
{
    Bit64 r[29];
    Bit64 fp;
    Bit64 lr;
    Bit64 sp;
    Bit64 pc;
    Bit32 pstate;
    Bit32 padding0[1];
    Bit128 v[32];
    Bit32 fpcr;
    Bit32 fpsr;
    Bit64 tpidr;
};

NN_STATIC_ASSERT(sizeof(ThreadContext) == 0x320);
#elif defined(NN_BUILD_CONFIG_CPU_ARM_V7A)
struct ThreadContext
{
    Bit32 r[13];
    Bit32 sp;
    Bit32 lr;
    Bit32 pc;
    Bit32 cpsr;
    Bit8 padding[4];
    Bit64 fpuRegisters[32];
    Bit32 fpscr;
    Bit32 fpexc;
    Bit32 tpidr;
};
#else
#endif

enum CreateProcessParameterFlag: uint32_t
{
    CreateProcessParameterFlag_64Bit                        = (1u << 0),

    CreateProcessParameterFlag_AddressSpaceMask             = (7u << 1),
    CreateProcessParameterFlag_AddressSpace32Bit            = (0u << 1),
    CreateProcessParameterFlag_AddressSpace64BitOld         = (1u << 1),
    CreateProcessParameterFlag_AddressSpace32BitNoReserved  = (2u << 1),
    CreateProcessParameterFlag_AddressSpace64Bit            = (3u << 1),

    CreateProcessParameterFlag_EnableJitDebug               = (1u << 4),
    CreateProcessParameterFlag_EnableAslr                   = (1u << 5),

    CreateProcessParameterFlag_IsApplication                = (1u << 6),

    CreateProcessParameterFlag_UseMemoryShift               = 7,
    CreateProcessParameterFlag_UseMemoryMask                = (0xFu << CreateProcessParameterFlag_UseMemoryShift),
    CreateProcessParameterFlag_UseMemory0                   = (0x0u << CreateProcessParameterFlag_UseMemoryShift),
    CreateProcessParameterFlag_UseMemory1                   = (0x1u << CreateProcessParameterFlag_UseMemoryShift),
    CreateProcessParameterFlag_UseMemory2                   = (0x2u << CreateProcessParameterFlag_UseMemoryShift),
    CreateProcessParameterFlag_UseMemory3                   = (0x3u << CreateProcessParameterFlag_UseMemoryShift),

    CreateProcessParameterFlag_All =
        CreateProcessParameterFlag_64Bit |
        CreateProcessParameterFlag_AddressSpaceMask |
        CreateProcessParameterFlag_EnableJitDebug |
        CreateProcessParameterFlag_EnableAslr |
        CreateProcessParameterFlag_IsApplication |
        CreateProcessParameterFlag_UseMemoryMask
};

enum ProcessInfoType: uint32_t
{
    ProcessInfoType_State
};

enum ProcessState: uint32_t
{
    ProcessState_Initializing,
    ProcessState_PreAttached,
    ProcessState_Running,
    ProcessState_WaitAttach,
    ProcessState_Attached,
    ProcessState_Terminating,
    ProcessState_Terminated,
    ProcessState_Breaked,
};

enum ExceptionType: uint32_t
{
    ExceptionType_Init                  = 0x0000,
    ExceptionType_InstructionAbort      = 0x0100,
    ExceptionType_DataAbort             = 0x0101,
    ExceptionType_UnalignedInstruction  = 0x0102,
    ExceptionType_UnalignedData         = 0x0103,
    ExceptionType_UndefinedInstruction  = 0x0104,
    ExceptionType_ExceptionInstruction  = 0x0105,
    ExceptionType_MemorySystemError     = 0x0106,
    ExceptionType_FpuException          = 0x0200,
    ExceptionType_InvalidSystemCall     = 0x0301,
    ExceptionType_BreakSystemCall       = 0x0302
};

enum InfoType: uint32_t
{
    InfoType_CoreMask,
    InfoType_PriorityMask,
    InfoType_ReservedRegionAddress,
    InfoType_ReservedRegionSize,
    InfoType_HeapRegionAddress,
    InfoType_HeapRegionSize,
    InfoType_AssingedUserPhysicalMemorySize,
    InfoType_UsingUserPhysicalMemorySize,
    InfoType_DebuggerPresence,
    InfoType_ResourceLimit,
    InfoType_IdleTickCount,
    InfoType_Random,
    InfoType_AslrRegionAddress,
    InfoType_AslrRegionSize,
    InfoType_StackRegionAddress,
    InfoType_StackRegionSize,
    InfoType_ExtraResourceSize,
    InfoType_ExtraResourceUsage,
    InfoType_ProgramId,
    InfoType_InitialProcessIdRangeObsolete, // 未使用にするためあける
    InfoType_ProcessRegionAddress,

    InfoType_PerformanceCounter = 0xF0000000,
    /*
     * param = 0, cycle count (起動してからの時間。指標に使う)
     * param = 1, cycle count
     * param = 2, cycle count in user
     * param = 3, instruction count
     * param = 4, instruction count in user
     * param = 5, l2 miss
     * param = 6, l2 miss in user
     */
    InfoType_IpcProfile,
    /*
     * param = 0, IPC Send
     * param = 1, IPC Recv
     * param = 2, IPC Reply
     */

    InfoType_ThreadTickCount,
    InfoType_CoreSwitchCount,
};

enum SystemInfoType: uint32_t
{
    SystemInfoType_PhysicalMemorySize,
    SystemInfoType_UsingPhysicalMemorySize,
    SystemInfoType_InitialProcessIdRange,
};

enum KernelDebugType: uint32_t
{
    KernelDebugType_Thread,
    KernelDebugType_ThreadCallstack,
    KernelDebugType_KernelObject,
    KernelDebugType_Handle,
    KernelDebugType_Memory,
    KernelDebugType_PageTable,
    KernelDebugType_CpuUtilization,
    KernelDebugType_Process,
    KernelDebugType_SuspendProcess,
    KernelDebugType_ResumeProcess,
    KernelDebugType_Port,

    KernelDebugType_KTraceSetTypeFilter,
    KernelDebugType_KTraceSetProcessFilter0,
    KernelDebugType_KTraceSetProcessFilter1,
    KernelDebugType_KTraceStart,
    KernelDebugType_KTraceStop,
    KernelDebugType_KTraceInfo,
};

enum HardwareBreakPointRegisterName: uint32_t
{
    HardwareBreakPointRegisterName_I0,
    HardwareBreakPointRegisterName_I1,
    HardwareBreakPointRegisterName_I2,
    HardwareBreakPointRegisterName_I3,
    HardwareBreakPointRegisterName_I4,
    HardwareBreakPointRegisterName_I5,
    HardwareBreakPointRegisterName_I6,
    HardwareBreakPointRegisterName_I7,
    HardwareBreakPointRegisterName_I8,
    HardwareBreakPointRegisterName_I9,
    HardwareBreakPointRegisterName_I10,
    HardwareBreakPointRegisterName_I11,
    HardwareBreakPointRegisterName_I12,
    HardwareBreakPointRegisterName_I13,
    HardwareBreakPointRegisterName_I14,
    HardwareBreakPointRegisterName_I15,
    HardwareBreakPointRegisterName_D0,
    HardwareBreakPointRegisterName_D1,
    HardwareBreakPointRegisterName_D2,
    HardwareBreakPointRegisterName_D3,
    HardwareBreakPointRegisterName_D4,
    HardwareBreakPointRegisterName_D5,
    HardwareBreakPointRegisterName_D6,
    HardwareBreakPointRegisterName_D7,
    HardwareBreakPointRegisterName_D8,
    HardwareBreakPointRegisterName_D9,
    HardwareBreakPointRegisterName_D10,
    HardwareBreakPointRegisterName_D11,
    HardwareBreakPointRegisterName_D12,
    HardwareBreakPointRegisterName_D13,
    HardwareBreakPointRegisterName_D14,
    HardwareBreakPointRegisterName_D15,
};

enum LastThreadInfoFlag: uint32_t
{
    LastThreadInfoFlag_ThreadInSystemCall = 0x00000001,
};

enum LimitableResource: uint32_t
{
    LimitableResource_PhysicalMemoryMax,        // 物理メモリ量の上限を表します。
    LimitableResource_ThreadCountMax,           // スレッド数の上限を表します。
    LimitableResource_EventCountMax,            // Event 数の上限を表します。
    LimitableResource_TransferMemoryCountMax,   // TransferMemory 数の上限を表します。
    LimitableResource_SessionCountMax,          // セッション数(Lightを含む)の上限を表します。

    LimitableResource_Num
};

enum ThreadActivity: uint32_t
{
    ThreadActivity_Runnable,
    ThreadActivity_Paused
};

enum ProcessActivity: uint32_t
{
    ProcessActivity_Runnable,
    ProcessActivity_Paused
};

enum KernelTraceState: uint32_t
{
    KernelTraceState_Paused,
    KernelTraceState_Running
};

enum CodeMemoryOperation: uint32_t
{
    CodeMemoryOperation_Map,
    CodeMemoryOperation_MapToOwner,
    CodeMemoryOperation_Unmap,
    CodeMemoryOperation_UnmapFromOwner,
};

enum ArbitrationType: uint32_t
{
    ArbitrationType_WaitIfLessThan,
    ArbitrationType_DecrementAndWaitIfLessThan,
    ArbitrationType_WaitIfEqual,
};

enum SignalType: uint32_t
{
    SignalType_SignalOnly,
    SignalType_IncrementIfEqual,
    SignalType_UpdateByCountIfEqual,
};

enum YieldType: int64_t
{
    YieldType_WithoutCoreMigration = 0,
    YieldType_WithCoreMigration = -1,
    YieldType_ToAnyThread = -2,
};
}}

