﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/socket/socket_Types.h>
#include <nn/socket/socket_ConstantsPrivate.h>

#include <nn/socket/private/net/socket_Bpf.h>
#include <nn/socket/private/net/socket_Ethernet.h>
#include <nn/socket/private/net/socket_If.h>
#include <nn/socket/private/net/socket_If_Arp.h>
#include <nn/socket/private/net/socket_If_Dl.h>
#include <nn/socket/private/net/socket_Route.h>
#include <nn/socket/private/netinet/socket_In.h>
#include <nn/socket/private/netinet/socket_Ip.h>
#include <nn/socket/private/netinet/socket_Ip_Icmp.h>
#include <nn/socket/private/netinet/socket_Tcp.h>
#include <nn/socket/private/netinet/socket_Udp.h>
#include <nn/socket/private/sys/socket.h>
#include <nn/socket/private/sys/socket_Ioccom.h>

namespace nn {
namespace socket {

/**
    Open flags.
*/
enum class OpenFlag : int32_t
{
    O_RdOnly = 0,    ///< Open for read-only.
    O_WrOnly = 1,    ///< Open for write-only.
    O_RdWr   = 2     ///< Open for reading and writing.
};

/**
    Socket type, which specifies the semantics of communication.
*/
enum class TypePrivate : uint32_t
{
    Sock_CloExec = 0x10000000
};

/**
    Address and protocol families.
*/
enum class FamilyPrivate : uint8_t
{
    Af_Iso          = 7,            ///< ISO protocols
    Pf_Iso          = Af_Iso,       ///< ISO protocols
    Af_Ccitt        = 10,           ///< CCITT protocols, X.25 etc
    Pf_Ccitt        = Af_Ccitt,     ///< CCITT protocols, X.25 etc
    Af_AppleTalk    = 16,           ///< Apple Talk
    Pf_AppleTalk    = Af_AppleTalk, ///< Apple Talk
    Af_Ipx          = 23,           ///< Novell Internet Protocol
    Pf_Ipx          = Af_Ipx,       ///< Novell Internet Protocol
    Af_NetGraph     = 32,           ///< Netgraph sockets
    Pf_NetGraph     = Af_NetGraph   ///< Netgraph sockets
};

/**
    Commands used for Ioctl().
*/
enum class IoctlCommandPrivate : uint32_t
{
    SiocSHiWat      = NN_SOCKET_IOW('s',  0, int),           ///< set high watermark
    SiocGHiWat      = NN_SOCKET_IOR('s',  1, int),           ///< get high watermark
    SiocSLoWat      = NN_SOCKET_IOW('s',  2, int),           ///< set low watermark
    SiocGLoWat      = NN_SOCKET_IOR('s',  3, int),           ///< get low watermark
    SiocAtMark      = NN_SOCKET_IOR('s',  7, int),           ///< at oob mark?
    SiocSPGrp       = NN_SOCKET_IOW('s',  8, int),           ///< set process group
    SiocGPGrp       = NN_SOCKET_IOR('s',  9, int),           ///< get process group
    SiocSIfAddr     = NN_SOCKET_IOW('i', 12, IfReq),         ///< set ifnet address
    OSiocGIfAddr    = NN_SOCKET_IOWR('i', 13, IfReq),        ///< get ifnet address
    SiocGIfAddr     = NN_SOCKET_IOWR('i', 33, IfReq),        ///< get ifnet address
    SiocSIfDstAddr  = NN_SOCKET_IOW('i', 14, IfReq),         ///< set p-p address
    OSiocGIfDstAddr = NN_SOCKET_IOWR('i', 15, IfReq),        ///< get p-p address
    SiocGIfDstAddr  = NN_SOCKET_IOWR('i', 34, IfReq),        ///< get p-p address
    SiocSIfFlags    = NN_SOCKET_IOW('i', 16, IfReq),         ///< set ifnet flags
    SiocGIfFlags    = NN_SOCKET_IOWR('i', 17, IfReq),        ///< get ifnet flags
    OSiocGIfBrdAddr = NN_SOCKET_IOWR('i', 18, IfReq),        ///< get broadcast addr
    SiocGIfBrdAddr  = NN_SOCKET_IOWR('i', 35, IfReq),        ///< get broadcast addr
    SiocSIfBrdAddr  = NN_SOCKET_IOW('i', 19, IfReq),         ///< set broadcast addr
    OSiocGIfConf    = NN_SOCKET_IOWR('i', 20, IfConf),       ///< get ifnet list
    SiocGIfConf     = NN_SOCKET_IOWR('i', 36, IfConf),       ///< get ifnet list
    OSiocGIfNetmask = NN_SOCKET_IOWR('i', 21, IfReq),        ///< get net addr mask
    SiocGIfNetmask  = NN_SOCKET_IOWR('i', 37, IfReq),        ///< get net addr mask
    SiocSIfNetmask  = NN_SOCKET_IOW('i', 22, IfReq),         ///< set net addr mask
    SiocGIfMetric   = NN_SOCKET_IOWR('i', 23, IfReq),        ///< get IF metric
    SiocSIfMetric   = NN_SOCKET_IOW('i', 24, IfReq),         ///< set IF metric
    SiocDIfAddr     = NN_SOCKET_IOW('i', 25, IfReq),         ///< delete IF addr
    OSiocAIfAddr    = NN_SOCKET_IOW('i', 26, OIfAliasReq),   ///< add/chg IF alias
    SiocALIfAddr    = NN_SOCKET_IOW('i', 27, IfLAddrReq),    ///< add IF addr
    SiocGLIfAddr    = NN_SOCKET_IOWR('i', 28, IfLAddrReq),   ///< get IF addr
    SiocDLIfAddr    = NN_SOCKET_IOW('i', 29, IfLAddrReq),    ///< delete IF addr
    SiocSIfCap      = NN_SOCKET_IOW('i', 30, IfReq),         ///< set IF features
    SiocGIfCap      = NN_SOCKET_IOWR('i', 31, IfReq),        ///< get IF features
    SiocGIfIndex    = NN_SOCKET_IOWR('i', 32, IfReq),        ///< get IF index
    SiocGIfMAC      = NN_SOCKET_IOWR('i', 38, IfReq),        ///< get IF MAC label
    SiocSIfMAC      = NN_SOCKET_IOW('i', 39, IfReq),         ///< set IF MAC label
    SiocSIfName     = NN_SOCKET_IOW('i', 40, IfReq),         ///< set IF name
    SiocSIfDescr    = NN_SOCKET_IOW('i', 41, IfReq),         ///< set ifnet descr
    SiocGIfDescr    = NN_SOCKET_IOWR('i', 42, IfReq),        ///< get ifnet descr
    SiocAIfAddr     = NN_SOCKET_IOW('i', 43, IfAliasReq),    ///< add/chg IF alias
    SiocAddMulti    = NN_SOCKET_IOW('i', 49, IfReq),         ///< add m'cast addr
    SiocDelMulti    = NN_SOCKET_IOW('i', 50, IfReq),         ///< del m'cast addr
    SiocGIfMtu      = NN_SOCKET_IOWR('i', 51, IfReq),        ///< get IF mtu
    SiocSIfMtu      = NN_SOCKET_IOW('i', 52, IfReq),         ///< set IF mtu
    SiocGIfPhys     = NN_SOCKET_IOWR('i', 53, IfReq),        ///< get IF wire
    SiocSIfPhys     = NN_SOCKET_IOW('i', 54, IfReq),         ///< set IF wire
    SiocSIfMedia    = NN_SOCKET_IOWR('i', 55, IfReq),        ///< set net media
    SiocGIfMedia    = NN_SOCKET_IOWR('i', 56, IfMediaReq),   ///< get net media
    SiocSIfGeneric  = NN_SOCKET_IOW('i', 57, IfReq),         ///< generic IF set op
    SiocGIfGeneric  = NN_SOCKET_IOWR('i', 58, IfReq),        ///< generic IF get op
    SiocGIfStatus   = NN_SOCKET_IOWR('i', 59, IfStat),       ///< get IF status
    SiocSIfLlAddr   = NN_SOCKET_IOW('i', 60, IfReq),         ///< set linklevel addr
    SiocSIfPhyAddr  = NN_SOCKET_IOW('i', 70, IfAliasReq),    ///< set gif addres
    SiocGIfPSrcAddr = NN_SOCKET_IOWR('i', 71, IfReq),        ///< get gif psrc addr
    SiocGIfPDstAddr = NN_SOCKET_IOWR('i', 72, IfReq),        ///< get gif pdst addr
    SiocDIfPhyAddr  = NN_SOCKET_IOW('i', 73, IfReq),         ///< delete gif addrs
    SiocSLIfPhyAddr = NN_SOCKET_IOW('i', 74, IfLAddrReq),    ///< set gif addrs
    SiocGLIfPhyAddr = NN_SOCKET_IOWR('i', 75, IfLAddrReq),   ///< get gif addrs
    SiocGPrivate_0  = NN_SOCKET_IOWR('i', 80, IfReq),        ///< device private 0
    SiocGPrivate_1  = NN_SOCKET_IOWR('i', 81, IfReq),        ///< device private 1
    SiocSIfVnet     = NN_SOCKET_IOWR('i', 90, IfReq),        ///< move IF jail/vnet
    SiocSIfRVnet    = NN_SOCKET_IOWR('i', 91, IfReq),        ///< reclaim vnet IF
    SiocGIfFib      = NN_SOCKET_IOWR('i', 92, IfReq),        ///< get IF fib
    SiocSIfFib      = NN_SOCKET_IOW('i', 93, IfReq),         ///< set IF fib
    SiocSDrvSpec    = NN_SOCKET_IOW('i', 123, IfDrv),        ///< set driver-specific parameters
    SiocGDrvSpec    = NN_SOCKET_IOWR('i', 123, IfDrv),       ///< get driver-specific parameters
    SiocIfCreate    = NN_SOCKET_IOWR('i', 122, IfReq),       ///< create clone if
    SiocIfCreate2   = NN_SOCKET_IOWR('i', 124, IfReq),       ///< create clone if
    SiocIfDestroy   = NN_SOCKET_IOW('i', 121, IfReq),        ///< destroy clone if
    SiocIfGCloners  = NN_SOCKET_IOWR('i', 120, IfCloneReq),  ///< get cloners
    SiocAIfGroup    = NN_SOCKET_IOW('i', 135, IfGroupReq),   ///< add an ifgroup
    SiocGIfGroup    = NN_SOCKET_IOWR('i', 136, IfGroupReq),  ///< get ifgroups
    SiocDIfGroup    = NN_SOCKET_IOW('i', 137, IfGroupReq),   ///< delete ifgroup
    SiocGIfGMemb    = NN_SOCKET_IOWR('i', 138, IfGroupReq),  ///< get members
    SiocGIfDupInfo  = NN_SOCKET_IOWR('i', 139, IfDuplicateIpInfo),   ///< get duplicate ip information

    BiocGBLen       = NN_SOCKET_IOR('B', 102, uint32_t),
    BiocSBLen       = NN_SOCKET_IOWR('B', 102, uint32_t),
    BiocSetf        = NN_SOCKET_IOW('B', 103, BpfProgram),
    BiocFlush       = NN_SOCKET_IO('B', 104),
    BiocPromisc     = NN_SOCKET_IO('B', 105),
    BiocGDlt        = NN_SOCKET_IOR('B', 106, uint32_t),
    BiocGetIf       = NN_SOCKET_IOR('B', 107, IfReq),
    BiocSetIf       = NN_SOCKET_IOW('B', 108, IfReq),
    BiocSRTimeout   = NN_SOCKET_IOW('B', 109, TimeVal),
    BiocGRTimeout   = NN_SOCKET_IOR('B', 110, TimeVal),
    BiocGStats      = NN_SOCKET_IOR('B', 111, BpfStat),
    BiocImmediate   = NN_SOCKET_IOW('B', 112, uint32_t),
    BiocVersion     = NN_SOCKET_IOR('B', 113, BpfVersion),
    BiocGRSig       = NN_SOCKET_IOR('B', 114, uint32_t),
    BiocSRSig       = NN_SOCKET_IOW('B', 115, uint32_t),
    BiocGHdrCmplt   = NN_SOCKET_IOR('B', 116, uint32_t),
    BiocSHdrCmplt   = NN_SOCKET_IOW('B', 117, uint32_t),
    BiocGDirection  = NN_SOCKET_IOR('B', 118, uint32_t),
    BiocSDirection  = NN_SOCKET_IOW('B', 119, uint32_t),
    BiocSDlt        = NN_SOCKET_IOW('B', 120, uint32_t),
    BiocGDltList    = NN_SOCKET_IOWR('B', 121, BpfDltList),
    BiocLock        = NN_SOCKET_IO('B', 122),
    BiocSetWf       = NN_SOCKET_IOW('B', 123, BpfProgram),
    BiocFeedback    = NN_SOCKET_IOW('B', 124, uint32_t),
    BiocGetBufMode  = NN_SOCKET_IOR('B', 125, uint32_t),
    BiocSetBufMode  = NN_SOCKET_IOW('B', 126, uint32_t),
    BiocGetZMax     = NN_SOCKET_IOR('B', 127, size_t),
    BiocRotZBuf     = NN_SOCKET_IOR('B', 128, BpfZBuf),
    BiocSetZBuf     = NN_SOCKET_IOW('B', 129, BpfZBuf),
    BiocSetFnr      = NN_SOCKET_IOW('B', 130, BpfProgram),
    BiocGTstamp     = NN_SOCKET_IOR('B', 131, uint32_t),
    BiocSTstamp     = NN_SOCKET_IOW('B', 132, uint32_t)
};

/**
 * @brief This type defines the ResolverOptionKey type and
 * constants that are used to delienate one option from another.
 *
 * @details
 * The range of ResolverOptionKeys are divided into three sub-groups.
 * The naming scheme of each option describes which context it can
 * be used in. There are three contexts: Get, Set, and Request.  Any
 * attempt to use an option outside of its allowed context results
 * in an error being returned to the caller and
 * @ref nn::socket::GetLastError() set
 *
 * - Get keys: Key names that start with "ResolverOptionKey_Get"
 * (range 10000-19999) are used with options that retrieve information
 * from the resolver. The result is a @ref ResolverOption that
 * corresponds to the data described. For more information see
 * @ref nn::socket::ResolverGetOption() and
 * @ref nn::socket::ResolverSetOption().
 *
 * - Set keys: Keys constants that start with "ResolverOptionKey_Set"
 * (range 20000-29999) are used with options that perform an action
 * or set a configuration value in the resolver. For more information
 * see @ref nn::socket::ResolverGetOption() and @ref
 * nn::socket::ResolverSetOption().
 *
 * - Request keys: Key that begin with "ResolverOptionKey_Request"
 * (range 30000-39999) provide options that are used only with a
 * resolver request lookup function such as
 * @ref nn::socket::GetHostEntByName GetHostEntByName(),
 * @ref nn::socket::GetHostEntByAddr GetHostEntByAddr(),
 * @ref nn::socket::GetAddrInfo GetAddrInfo(),
 * and @ref nn::socket::GetNameInfo GetNameInfo().
 */
enum class ResolverOptionKey : uint32_t
{
    // ---------------------------------------------------------------
    // Get Option Block
    // ---------------------------------------------------------------

    /**
     * @brief
     * Defines the minimum boundary for Get key values.
     *
     * @details
     * Minimum and maximum boundaries are not valid key values and
     * produce an error if used.
     */
    GetMinimumValue = 10000,

    /**
     * @brief
     * Get a cancel handle from the resolver for use by the client.
     *
     * @details
     * You can use the returned handle to immediately cancel a long-term
     * resolver activity such as nn::socket::GetHostEntByName().  The API is provided
     * for situations where one no longer wishes to wait for a
     * response from the resolver.
     *
     * @code
     * // Get the handle
     * nn::socket::ResolverOption option, optionSave;
     * nn::socket::ResolverGetOption(&option, ResolverOptionKey::GetCancelHandleInteger);
     * optionSave = option;
     *
     * // Use the cancel handle
     * option.key = ResolverOptionKey::RequestCancelHandleInteger;
     * nn::socket::GetHostEntByName(host, &option, 1);
     *
     * // In another thread you can cancel the request in progress by calling
     * // @ref ResolverSetOption().
     * optionSave.key = SetCancelHandleInteger;
     * nn::socket::ResolverSetOption(optionSave);
     * @endcode
     */
    GetCancelHandleInteger = 10001,

    /**
     * @brief
     * Defines a maximum boundary for Get key values.
     *
     * @details
     * Minimum and maximum boundaries are not valid key values and
     * produce an error if used.
     */
    GetMaximumValue = 10002,


    // ---------------------------------------------------------------
    // Set Option Block
    // ---------------------------------------------------------------

    /**
     * @brief
     * Defines the minimum boundary for Get key values.
     *
     * @details
     * Minimum and maximum boundaries are not valid key values and
     * produce an error if used.
     */
    SetMinimumValue = 20000,

    /**
     * @brief
     * Cancels an active DNS request.
     *
     * @code
     * // Get the cancel handle to be used with request
     * nn::socket::ResolverOption option = {
     *     .key = nn::socket::ResolverOptionKey::SetCancelHandleInteger,
     *     .type = nn::socket::ResolverOptionType::Integer,
     *     .data.integerValue = cancelHandle
     * };
     *
     * @endcode
     */
    SetCancelHandleInteger = 20001,

    /**
     * @brief
     * Removes a domain name from the DNS cache, which
     * effectively forces the resolver client to hit the DNS server
     * on subsequent lookups.
     *
     * @code
     * const char* domainname = "www.nintendo.com";
     *
     * nn::socket::ResolverOption option = {
     *     .key = nn::socket::ResolverOptionKey::SetRemoveDomainnameFromCachePointer,
     *     .type = nn::socket::ResolverOptionType::Pointer,
     *     .size = strlen(domainname) + 1;
     *     .data = domainname,
     * };
     *
     *  rc = nn::socket::SetResolverOption(option);
     * @endcode
     */
    SetRemoveDomainnameFromCachePointer = 20002,

    /**
     * @brief
     * Removes an ip address in network-byte order from
     * the DNS cache, which effectively forces the resolver client to
     * hit the DNS server on subsequent lookups.
     *
     * @code
     * // Get the cancel handle
     * nn::socket::ResolverOption option = {
     *     .key = nn::socket::ResolverOptionKey::SetRemoveIpAddressFromCacheUnsigned32
     *     .type = nn::socket::ResolverOptionType::Unsigned32,
     *     .size = sizeof(uint32_t)
     * };
     *
     * inet_ntop(AF_INET, "1.2.3.4", &option.data.unsigned32Value,
     *           sizeof(uint32_t));
     *
     * rc = nn::socket::SetResolverOption(option);
     * @endcode
     */
    SetRemoveIpAddressFromCacheUnsigned32 = 20003,

    /**
     * @brief
     * Defines the maximum boundary for Get key values.
     *
     * @details
     * Minimum and maximum boundaries are not valid key values and
     * produce an error if used.
     */
    SetMaximumValue = 20004,

    // ---------------------------------------------------------------
    // Request Option Block
    // ---------------------------------------------------------------

    /**
     * @brief
     * Defines the minimum boundary for Request key values.
     *
     * @details
     * Minimum and maximum boundaries are not valid key values and
     * produce an error if used.
     */
    RequestMinimumValue = 30000,

    /**
     * @brief This option enables or disables Network Service
     * Discovery for a single resolver request.
     *
     * @details By default NSD is enabled for the following calls:
     * - @ref nn::socket::GetHostEntByName
     * - @ref nn::socket::GetAddrInfo
     *
     * @code
     * nn::socket::ResolverOption option = {
     *     .key = nn::socket::ResolverOptionKey::RequestEnableServiceDiscoveryBoolean,
     *     .type = nn::socket::ResolverOptionType::Boolean,
     *     .size = sizeof(bool),
     *     .data.boolValue = true
     * };
     *
     * // then make a resolver call such as:
     * nn::socket::GetHostEntByName(name, &option, 1);
     * @endcode
     */
    RequestEnableServiceDiscoveryBoolean = 30001,

    /**
     * @brief This option contains a unique resolver cancellation
     * handle that can be used to cancel a request. It is made
     * available only on the Horizon platform and is ignored on other
     * platforms.
     *
     * @details The total time spent in a DNS call depend on multiple
     * conditions such as DNS delegation and recursion, network
     * availability, server resources, and even security options.
     * Therefore time spent is a resolver call is not deterministic
     * and it might be advisable to abort a DNS. For this reason
     * Nintendo provides a DNS request cancellation API via
     * the ResolverOption interface. The value for this key is a
     * unique DNS handle that is passed to the resolver that allows
     * a developer to stop the request from another thread. It is
     * still necessary to check the result from the resolver API to
     * determine whether the cancellation was successful.
     *
     * @code
     *
     * // Get the handle
     * nn::socket::ResolverOption option, optionSave;
     * nn::socket::ResolverGetOption(&option, ResolverOptionKey::GetCancelHandleInteger);
     * optionSave = option;
     *
     * // Use the cancel handle
     * option.key = nn::socket::ResolverOptionKey::RequestCancelHandleInteger;
     * nn::socket::GetHostEntByName(host, &option, 1);
     *
     * struct hostent* entry =
     *     nn::socket::GetHostEntByName(name, &option, 1);
     * if (NULL == entry && nn::socket::Errno::ECanceled == nn::socket::GetLastError())
     * {
     *      // ... request was cancelled
     * }
     *
     * @endcode
     */
    RequestCancelHandleInteger = 30002,

    /**
     * @brief This option contains a flag that enables or disables the
     * DNS cache for the given resolver request.
     *
     * @details
     * By default this flag is true.
     *
     * @code
     * nn::socket::ResolverOption option = {
     *     .key = nn::socket::ResolverOptionKey::RequestEnableDnsCacheBoolean,
     *     .type = nn::socket::ResolverOptionType::Boolean,
     *     .size = sizeof(bool),
     *     .data.boolValue = false
     * };
     * @endcode
     */
    RequestEnableDnsCacheBoolean = 30003,

    /**
     * @brief
     * Defines the maximum boundary for Request key values.
     *
     * @details
     * Minimum and maximum boundaries are not valid key values and
     * produce an error if used.
     */
    RequestMaximumValue = 30004,
};

/**
 * @brief This enum lists all the ResolverOption value types.
 *
 * @details The ResolverOptions Container union is used to contain values of
 * multiple types.
 *
 * To reduce error:
 * - With the exception of strings the ResolverOption size
 *   is compared against the size of the type.
 *
 * - Strings are compared to the string length.
 *
 * - The @ref ResolverOptionKey key also contains the type name.
 *
 * For example:
 *
 * -- RequestEnableServiceDiscoveryBoolean --> ResolverOptionType::Boolean
 * -- GetCancelHandleInteger --> ResolverOptionType::Integer
 *
 * More generally:
 * -- ResolverOptionKey::OptionNameAndTypeName --> ResolverOptionType::TypeName
 */
enum class ResolverOptionType : uint32_t
{
    /**
     * @brief The data value is a boolean.
     *
     * @details Valid size: sizeof(bool).
     */
    Boolean = 100,

    /**
     * @brief The data value is an integer.
     *
     * @details Valid size: sizeof(int).
     */
    Integer = 101,

    /**
     * @brief The data value is an unsigned 32 bit integer.
     *
     * @details Valid size: sizeof(uint32_t).
     */
    Unsigned32 = 102,

    /**
     * @brief The data value is an unsigned 64 bit integer.
     *
     * @details Valid size: sizeof(uint64_t).
     */
    Unsigned64 = 103,

    /**
     * @brief The data value is a double.
     *
     * @details Valid size: sizeof(double).
     */
    Double = 104,

    /**
     * @brief The data value is a string or pointer value.
     *
     * @details The value is a pointer. The size parameter reflects
     * the size in bytes. Strings are considered pointers and must
     * include the null-terminator in the size.
     */
    Pointer = 105,
};

/**
 * @brief Structure for resolver options.
 *
 * @details This structure is a container class that contains a @ref
 * ResolverOptionKey value that specifies how the structure should
 * be interpreted. The @ref ResolverOptionType value specifies the
 * type of data that the container either <i>does</i> contain such as
 * in the case of a set or request operation) or <i>will</i> contain
 * such as in the case of a get operation.
 */
struct ResolverOption
{
    /**
     * @brief The resolver option key corresponding to this option.
     *
     * @details See @ref ResolverOptionKey.
     */
    ResolverOptionKey key;

    /**
     * @brief The resolver option type corresponding to this option.
     *
     * @details see @ref ResolverOptionType.
     * @details See @ref ResolverOptionType.
     */
    ResolverOptionType type;

    /**
     * @brief the resolver option size
     *
     * @details proper sizing for each @ref ResolverOptionType are listed below
     *
     * - nn::socket::ResolverOptionType::Boolean    = sizeof(bool)
     * - nn::socket::ResolverOptionType::Integer    = sizeof(int)
     * - nn::socket::ResolverOptionType::Unsigned32 = sizeof(uint32_t)
     * - nn::socket::ResolverOptionType::Unsigned64 = sizeof(uint64_t)
     * - nn::socket::ResolverOptionType::Double     = sizeof(double)
     * - nn::socket::ResolverOptionType::Pointer    = pointer data length such as a string length from @ref nn::util::Strnlen
     */
    size_t size;

    /**
     * @brief The resolver option container.
     * @note All Container data values have corresponding type information in
     * the key and type and the size argument also implicitly corresponds to the
     * size of the value or length string (minus the NULL-terminator).
     */
    union Container
    {
        bool        booleanValue;
        int         integerValue;
        uint32_t    unsigned32Value;
        uint64_t    unsigned64Value;
        double      doubleValue;
        const char* pointerValue;
    }
    data;
};

inline OpenFlag operator | (OpenFlag a, OpenFlag b) NN_NOEXCEPT
{
    return static_cast<OpenFlag>(static_cast<int>(a) | static_cast<int>(b));
}

inline OpenFlag operator |= (OpenFlag &a, OpenFlag b) NN_NOEXCEPT
{
    return a = a | b;
}

inline OpenFlag operator & (OpenFlag a, OpenFlag b) NN_NOEXCEPT
{
    return static_cast<OpenFlag>(static_cast<int>(a) & static_cast<int>(b));
}

inline OpenFlag operator &= (OpenFlag &a, OpenFlag b) NN_NOEXCEPT
{
    return a = a & b;
}

inline OpenFlag operator ^ (OpenFlag a, OpenFlag b) NN_NOEXCEPT
{
    return static_cast<OpenFlag>(static_cast<int>(a) ^ static_cast<int>(b));
}

inline OpenFlag operator ^= (OpenFlag &a, OpenFlag b) NN_NOEXCEPT
{
    return a = a ^ b;
}

inline OpenFlag operator ~ (OpenFlag a) NN_NOEXCEPT
{
    return static_cast<OpenFlag>(~static_cast<int>(a));
}

inline Type operator | (Type a, TypePrivate b) NN_NOEXCEPT
{
    return static_cast<Type>(static_cast<int>(a) | static_cast<int>(b));
}

inline Type operator |= (Type &a, TypePrivate b) NN_NOEXCEPT
{
    return a = a | b;
}

inline Type operator & (Type a, TypePrivate b) NN_NOEXCEPT
{
    return static_cast<Type>(static_cast<int>(a) & static_cast<int>(b));
}

inline Type operator &= (Type &a, TypePrivate b) NN_NOEXCEPT
{
    return a = a & b;
}

inline Type operator ^ (Type a, TypePrivate b) NN_NOEXCEPT
{
    return static_cast<Type>(static_cast<int>(a) ^ static_cast<int>(b));
}

inline Type operator ^= (Type &a, TypePrivate b) NN_NOEXCEPT
{
    return a = a ^ b;
}

inline IfrFlag operator | (IfrFlag a, IfrFlag b) NN_NOEXCEPT
{
    return static_cast<IfrFlag>(static_cast<int>(a) | static_cast<int>(b));
}

inline IfrFlag operator |= (IfrFlag &a, IfrFlag b) NN_NOEXCEPT
{
    return a = a | b;
}

inline IfrFlag operator & (IfrFlag a, IfrFlag b) NN_NOEXCEPT
{
    return static_cast<IfrFlag>(static_cast<int>(a) & static_cast<int>(b));
}

inline IfrFlag operator &= (IfrFlag &a, IfrFlag b) NN_NOEXCEPT
{
    return a = a & b;
}

inline IfrFlag operator ^ (IfrFlag a, IfrFlag b) NN_NOEXCEPT
{
    return static_cast<IfrFlag>(static_cast<int>(a) ^ static_cast<int>(b));
}

inline IfrFlag operator ^= (IfrFlag &a, IfrFlag b) NN_NOEXCEPT
{
    return a = a ^ b;
}

inline IfrFlag operator ~ (IfrFlag a) NN_NOEXCEPT
{
    return static_cast<IfrFlag>(~static_cast<int>(a));
}

}} /* nn::socket */
