﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/
#pragma once

/**
 * @file This file implements a logging facility for the resolver
 *
 * @details
 * The use of macros are so that certain functions are compiled out
 * at runtime. The logger has been designed in such a way that it
 * can be used with the telemetry API.
 */

/**
 * @brief This preprocessor macro is used as a log level constant
 * for major errors.
 */
#define LOG_LEVEL_MAJOR 0 // NOLINT(preprocessor/const)

/**
 * @brief This preprocessor macro is used as a log level constant
 * for minor errors.
 */
#define LOG_LEVEL_MINOR 1 // NOLINT(preprocessor/const)

/**
 * @brief This preprocessor macro is used as a log level constant
 * for informational messages.
 */
#define LOG_LEVEL_INFO  2 // NOLINT(preprocessor/const)

/**
 * @brief This preprocessor macro is used as a log level constant
 * for debug messages.
 */
#define LOG_LEVEL_DEBUG 3 // NOLINT(preprocessor/const)

/**
 * @brief This preprocessor macro is used as a log level constant
 * for for debug messages that include a hex string
 */
#define LOG_LEVEL_HEX   4 // NOLINT(preprocessor/const)

/**
 * @brief This preprocessor macro is used to determine the maximum
 * log level; it is currently synonymous with @ref LOG_LEVEL_HEX
 * LOG_LEVEL_HEX
 */
#define LOG_LEVEL_MAX   LOG_LEVEL_HEX // NOLINT(preprocessor/const)

/**
 * @brief This preprocessor definition is used to determine which
 * messages will be logged.
 *
 * @detail LOG_LEVEL defines the highest level that is logged. If it
 * is not already defined when this file is included then it defaults
 * to LOG_LEVEL_INFO.
 */
#ifndef LOG_LEVEL
#define LOG_LEVEL LOG_LEVEL_INFO // NOLINT(preprocessor/const)
#endif

/**
 * @brief A module is composed of one or more files; it is a logical
 * definition.
 */
#ifndef LOG_MODULE_NAME
#define LOG_MODULE_NAME "undefined" // NOLINT(preprocessor/const)
#endif

#if defined(__cplusplus)
extern "C"
{
#endif

#include <stdio.h>
#include <stdint.h>

/**
 * @brief this function provides the short file name by searching
 * in reverse for a path delimiter ('/' or '\') and returning the
 * address.
 *
 * @param line the line that contains the file name from __FILE__
 *
 */
const char* GetFilename(const char* line);

/**
 * @brief Return the string representation for the level provided
 *
 * @param level the level
 */
const char* GetLevelString(unsigned level);

/**
 * @brief This function is a printf-style log with an optional hexdump.
 *
 * @param linebreak should the hex formatter use linebreaks or not
 *
 * @param hexbuffer A pointer to a buffer that contains bytes to print.
 *
 * @param hexsize the size of the hexbuffer
 *
 * @param format a printf-style format string
 */
void ResolverLog(bool linebreak, const void* hexbuffer, unsigned hexsize, const char* format, ...);

#if defined(__cplusplus)
};
#endif

#define RESOLVER_LOG(format, level, ...)                             \
    do                                                               \
    {                                                                \
        ResolverLog(false, NULL, 0,                                  \
                    "%s:%u %s (%s) %s " format,                      \
                    GetFilename(__FILE__),                           \
                    __LINE__,                                        \
                    __FUNCTION__,                                    \
                    LOG_MODULE_NAME,                                 \
                    GetLevelString(level),                           \
                    ##__VA_ARGS__);                                  \
    } while (NN_STATIC_CONDITION(false))

/**
 * @brief Define the Major Error Resolver Log facility
 *
 * @detail
 * When enabled this block of code is used to log major errors. It
 * prints to the console and is designed to be used with the telemetry
 * API.
 */
#if defined LogMajor
#undef LogMajor
#endif
#if LOG_LEVEL >= LOG_LEVEL_MAJOR
#define LogMajor(format, ...) RESOLVER_LOG(format, LOG_LEVEL_MAJOR, ##__VA_ARGS__)
#else
#define LogMajor(ignore, ...)
#endif


/**
 * @brief Define the Minor Error Resolver Log facility
 *
 * @detail
 * When enabled this block of code is used to log major errors to the
 * console. It is designed to be used with the telemetry API.
 */
#if defined LogMinor
#undef LogMinor
#endif
#if LOG_LEVEL >= LOG_LEVEL_MINOR
#define LogMinor(format, ...) RESOLVER_LOG(format, LOG_LEVEL_MINOR, ##__VA_ARGS__)
#else
#define LogMinor(ignore, ...)
#endif

/**
 * @brief Define the Informational Resolver Log facility
 *
 * @detail
 * When enabled this block of code is used to log informational
 * messages. It prints to the console and is designed to be used with
 * the telemetry API.
 */
#if defined LogInfo
#undef LogInfo
#endif
#if LOG_LEVEL >= LOG_LEVEL_INFO
#define LogInfo(format, ...) RESOLVER_LOG(format, LOG_LEVEL_INFO, ##__VA_ARGS__)
#else
#define LogInfo(ignore, ...)
#endif

/**
 * @brief Define the Debug Resolver Log facility
 *
 * @detail
 * When enabled this block of code is used to log debug messages. It
 * prints to the console and is designed to be used with the telemetry
 * API.
 *
 * Even though it is designed to be used with the telemetry API it
 * will only be enabled in rare cases if ever used at all because
 * debug logging is expensive.
 */
#if defined LogDebug
#undef LogDebug
#endif
#if LOG_LEVEL >= LOG_LEVEL_DEBUG
#define LogDebug(format, ...) RESOLVER_LOG(format, LOG_LEVEL_DEBUG, ##__VA_ARGS__)
#else
#define LogDebug(ignore, ...)
#endif

/**
 * @brief Define the Debug with Hexdump Resolver Log facility
 *
 * @detail
 * When enabled this block of code is used to log debug messages with
 * a hexdump of data up to 1024 bytes. It prints data to the console
 * and is designed to be used with the telemetry API.
 *
 * Even though it is designed to be used with the telemetry API it
 * will only be enabled in rare cases if ever used at all because
 * debug logging is expensive.
 */
#if defined LogHex
#undef LogHex
#endif
#if LOG_LEVEL >= LOG_LEVEL_HEX
#define LogHex(buffer, size, format, ...)                            \
    do                                                               \
    {                                                                \
        ResolverLog(false, (const uint8_t*)buffer, size,             \
                    "%s:%u %s (%s) %s " format,                      \
                    GetFilename(__FILE__),                           \
                    __LINE__,                                        \
                    __FUNCTION__,                                    \
                    LOG_MODULE_NAME,                                 \
                    GetLevelString(LOG_LEVEL_HEX),                   \
                    ##__VA_ARGS__);                                  \
    } while (NN_STATIC_CONDITION(false))

#else
#define LogHex(ignore, ...)
#endif

#if defined LogFormattedHex
#undef LogFormattedHex
#endif
#if LOG_LEVEL >= LOG_LEVEL_HEX
#define LogFormattedHex(buffer, size, format, ...)                   \
    do                                                               \
    {                                                                \
        ResolverLog(true, (const uint8_t*)buffer, size,              \
                    "%s:%u %s (%s) %s " format,                      \
                    GetFilename(__FILE__),                           \
                    __LINE__,                                        \
                    __FUNCTION__,                                    \
                    LOG_MODULE_NAME,                                 \
                    GetLevelString(LOG_LEVEL_HEX),                   \
                    ##__VA_ARGS__);                                  \
    } while (NN_STATIC_CONDITION(false))

#else
#define LogFormattedHex(ignore, ...)
#endif
