﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/socket/sys/socket.h>

namespace nn {
namespace socket {

/*
 * Constants and structures defined by the internet system,
 * Per RFC 790, September 1981, and numerous additions.
 */

/*
 * Protocols (RFC 1700)
 */
enum class ProtocolPrivate : int32_t
{
    IpProto_HopOpts         = 0,           /* IP6 hop-by-hop options */
    IpProto_Igmp            = 2,           /* group mgmt protocol */
    IpProto_Ggp             = 3,           /* gateway^2 (deprecated) */
    IpProto_Ipv4            = 4,           /* IPv4 encapsulation */
    IpProto_Ipip            = IpProto_Ipv4,    /* for compatibility */
    IpProto_St              = 7,           /* Stream protocol II */
    IpProto_Egp             = 8,           /* exterior gateway protocol */
    IpProto_Pigp            = 9,           /* private interior gateway */
    IpProto_RccMon          = 10,          /* BBN RCC Monitoring */
    IpProto_Nvpii           = 11,          /* network voice protocol*/
    IpProto_Pup             = 12,          /* pup */
    IpProto_Argus           = 13,          /* Argus */
    IpProto_Emcon           = 14,          /* EMCON */
    IpProto_Xnet            = 15,          /* Cross Net Debugger */
    IpProto_Chaos           = 16,          /* Chaos*/
    IpProto_Mux             = 18,          /* Multiplexing */
    IpProto_Meas            = 19,          /* DCN Measurement Subsystems */
    IpProto_Hmp             = 20,          /* Host Monitoring */
    IpProto_Prm             = 21,          /* Packet Radio Measurement */
    IpProto_Idp             = 22,          /* xns idp */
    IpProto_Trunk1          = 23,          /* Trunk-1 */
    IpProto_Trunk2          = 24,          /* Trunk-2 */
    IpProto_Leaf1           = 25,          /* Leaf-1 */
    IpProto_Leaf2           = 26,          /* Leaf-2 */
    IpProto_Rdp             = 27,          /* Reliable Data */
    IpProto_Irtp            = 28,          /* Reliable Transaction */
    IpProto_Tp              = 29,          /* tp-4 w/ class negotiation */
    IpProto_Blt             = 30,          /* Bulk Data Transfer */
    IpProto_Nsp             = 31,          /* Network Services */
    IpProto_Inp             = 32,          /* Merit Internodal */
    IpProto_Sep             = 33,          /* Sequential Exchange */
    IpProto_3Pc             = 34,          /* Third Party Connect */
    IpProto_Idpr            = 35,          /* InterDomain Policy Routing */
    IpProto_Xtp             = 36,          /* XTP */
    IpProto_Ddp             = 37,          /* Datagram Delivery */
    IpProto_Cmtp            = 38,          /* Control Message Transport */
    IpProto_Tpxx            = 39,          /* TP++ Transport */
    IpProto_Il              = 40,          /* IL transport protocol */
    IpProto_Ipv6            = 41,          /* IP6 header */
    IpProto_Sdrp            = 42,          /* Source Demand Routing */
    IpProto_Routing         = 43,          /* IP6 routing header */
    IpProto_Fragment        = 44,          /* IP6 fragmentation header */
    IpProto_Idrp            = 45,          /* InterDomain Routing*/
    IpProto_Rsvp            = 46,          /* resource reservation */
    IpProto_Gre             = 47,          /* General Routing Encap. */
    IpProto_Mhrp            = 48,          /* Mobile Host Routing */
    IpProto_Bha             = 49,          /* BHA */
    IpProto_Esp             = 50,          /* IP6 Encap Sec. Payload */
    IpProto_Ah              = 51,          /* IP6 Auth Header */
    IpProto_Inlsp           = 52,          /* Integ. Net Layer Security */
    IpProto_Swipe           = 53,          /* IP with encryption */
    IpProto_Nhrp            = 54,          /* Next Hop Resolution */
    IpProto_Mobile          = 55,          /* IP Mobility */
    IpProto_Tlsp            = 56,          /* Transport Layer Security */
    IpProto_Skep            = 57,          /* SKIP */
    IpProto_Icmpv6          = 58,          /* ICMP6 */
    IpProto_None            = 59,          /* IP6 no next header */
    IpProto_DstOpts         = 60,          /* IP6 destination option */
    IpProto_Ahip            = 61,          /* any host internal protocol */
    IpProto_Cftp            = 62,          /* CFTP */
    IpProto_Hello           = 63,          /* "hello" routing protocol */
    IpProto_SatExpak        = 64,          /* SATNET/Backroom EXPAK */
    IpProto_KryptoLan       = 65,          /* Kryptolan */
    IpProto_Rvd             = 66,          /* Remote Virtual Disk */
    IpProto_Ippc            = 67,          /* Pluribus Packet Core */
    IpProto_Adfs            = 68,          /* Any distributed FS */
    IpProto_SatMon          = 69,          /* Satnet Monitoring */
    IpProto_Visa            = 70,          /* VISA Protocol */
    IpProto_Ipcv            = 71,          /* Packet Core Utility */
    IpProto_Cpnx            = 72,          /* Comp. Prot. Net. Executive */
    IpProto_Cphb            = 73,          /* Comp. Prot. HeartBeat */
    IpProto_Wsn             = 74,          /* Wang Span Network */
    IpProto_Pvp             = 75,          /* Packet Video Protocol */
    IpProto_BrSatMon        = 76,          /* BackRoom SATNET Monitoring */
    IpProto_Nd              = 77,          /* Sun net disk proto (temp.) */
    IpProto_WbMon           = 78,          /* WIDEBAND Monitoring */
    IpProto_WbExpak         = 79,          /* WIDEBAND EXPAK */
    IpProto_Eon             = 80,          /* ISO cnlp */
    IpProto_Vmtp            = 81,          /* VMTP */
    IpProto_Svmtp           = 82,          /* Secure VMTP */
    IpProto_Vines           = 83,          /* Banyon VINES */
    IpProto_Ttp             = 84,          /* TTP */
    IpProto_Igp             = 85,          /* NSFNET-IGP */
    IpProto_Dgp             = 86,          /* dissimilar gateway prot. */
    IpProto_Tcf             = 87,          /* TCF */
    IpProto_Igrp            = 88,          /* Cisco/GXS IGRP */
    IpProto_Ospfigp         = 89,          /* OSPFIGP */
    IpProto_Srpc            = 90,          /* Strite RPC protocol */
    IpProto_Larp            = 91,          /* Locus Address Resoloution */
    IpProto_Mtp             = 92,          /* Multicast Transport */
    IpProto_Ax25            = 93,          /* AX.25 Frames */
    IpProto_IpEIp           = 94,          /* IP encapsulated in IP */
    IpProto_Micp            = 95,          /* Mobile Int.ing control */
    IpProto_Sccsp           = 96,          /* Semaphore Comm. security */
    IpProto_EtherIp         = 97,          /* Ethernet IP encapsulation */
    IpProto_Encap           = 98,          /* encapsulation header */
    IpProto_Apes            = 99,          /* any private encr. scheme */
    IpProto_Gmtp            = 100,         /* GMTP*/
/* 101-254: Partly Unassigned */
    IpProto_Pim             = 103,         /* Protocol Independent Mcast */
    IpProto_IpComp          = 108,         /* payload compression (IPComp) */
    IpProto_Carp            = 112,         /* CARP */
    IpProto_Pgm             = 113,         /* PGM */
    IpProto_Sctp            = 132,         /* SCTP */
    IpProto_Mh              = 135,         /* IPv6 Mobility Header */
    IpProto_Mpls            = 137,         /* MPLS-in-IP */
    IpProto_Hip             = 139,         /* IP6 Host Identity Protocol */
    IpProto_Shim6           = 140,         /* IP6 Shim6 Protocol */
    IpProto_PfSync          = 240,         /* PFSYNC */
    IpProto_Reserved_253    = 253,         /* Reserved */
    IpProto_Reserved_254    = 254,         /* Reserved */
/* 255: Reserved */
/* BSD Private, local use, namespace incursion, no longer used */
    IpProto_Old_Divert      = 254,         /* OLD divert pseudo-proto */
    IpProto_Max             = 256,

/* last return value of *_input(), meaning "all job for this pkt is done".  */
    IpProto_Done            = 257,

/* Only used internally, so can be outside the range of valid IP protocols. */
    IpProto_Divert          = 258,         /* divert pseudo-protocol */
    IpProto_Send            = 259,         /* SeND pseudo-protocol */

/*
 * Defined to avoid confusion.  The master value is defined by
 * PROTO_SPACER in sys/protosw.h.
 */
    IpProto_Spacer          = 32767,       /* spacer for loadable protos */
};

/*
 * Local port number conventions:
 *
 * When a user does a Bind() or Connect() with a port number of zero,
 * a non-conflicting local port address is chosen.
 * The default range is IpPort_HiFirstAuto through
 * IpPort_HiLastAuto, although that is settable by sysctl.
 *
 * A user may set the IPPROTO_IP option Ip_PortRange to change this
 * default assignment range.
 *
 * The value Ip_PortRange_Default causes the default behavior.
 *
 * The value Ip_PortRange_Hign changes the range of candidate port numbers
 * into the "high" range.  These are reserved for client outbound connections
 * which do not want to be filtered by any firewalls.
 *
 * The value Ip_PortRange_Low changes the range to the "low" are
 * that is (by convention) restricted to privileged processes.  This
 * convention is based on "vouchsafe" principles only.  It is only secure
 * if you trust the remote host to restrict these ports.
 *
 * The default range of ports and the high range can be changed by
 * sysctl(3).  (net.inet.ip.port{hi,low}{first,last}_auto)
 *
 * Changing those values has bad security implications if you are
 * using a stateless firewall that is allowing packets outside of that
 * range in order to allow transparent outgoing connections.
 *
 * Such a firewall configuration will generally depend on the use of these
 * default values.  If you change them, you may find your Security
 * Administrator looking for you with a heavy object.
 *
 * For a slightly more orthodox text view on this:
 *
 *            ftp://ftp.isi.edu/in-notes/iana/assignments/port-numbers
 *
 *    port numbers are divided into three ranges:
 *
 *                0 -  1023 Well Known Ports
 *             1024 - 49151 Registered Ports
 *            49152 - 65535 Dynamic and/or Private Ports
 *
 */

/*
 * Ports < IPPORT_RESERVED are reserved for
 * privileged processes (e.g. root).         (Ip_PortRange_Low)
 */
enum class IpPort : uint16_t
{
    IpPort_Reserved         = 1024,

/*
 * Default local port range, used by Ip_PortRange_Default
 */
    IpPort_EphemeralFirst   = 10000,
    IpPort_EphemeralLast    = 65535,

/*
 * Dynamic port range, used by Ip_PortRange_High.
 */
    IpPort_HiFirstAuto      = 49152,
    IpPort_HiLastAuto       = 65535,

/*
 * Scanning for a free reserved port return a value below IpPort_Reserved,
 * but higher than IpPort_ReservedStart.  Traditionally the start value was
 * 512, but that conflicts with some well-known-services that firewalls may
 * have a fit if we use.
 */
    IpPort_ReservedStart    = 600,

    IpPort_Max              = 65535
};

/*
 * Definitions of bits in internet address integers.
 * On subnets, the decomposition of addresses to host and net parts
 * is done according to subnet mask, not the masks here.
 */
inline bool In_ClassA(uint32_t addr)
{
    return (addr & 0x80000000) == 0;
}

enum class InClassA : uint32_t
{
    In_ClassA_Net         =  0xff000000,
    In_ClassA_NShift      =  24,
    In_ClassA_Host        =  0x00ffffff,
    In_ClassA_Max         =  128
};

inline bool In_ClassB(uint32_t addr)
{
    return (addr & 0xc0000000) == 0x80000000;
}

enum class InClassB : uint32_t
{
    In_ClassB_Net         =  0xffff0000,
    In_ClassB_NShift      =  16,
    In_ClassB_Host        =  0x0000ffff,
    In_ClassB_Max         =  65536
};

inline bool In_ClassC(uint32_t addr)
{
    return (addr & 0xe0000000) == 0xc0000000;
}

enum class InClassC : uint32_t
{
    In_ClassC_Net         =  0xffffff00,
    In_ClassC_NShift      =  8,
    In_ClassC_Host        =  0x000000ff
};

inline bool In_ClassD(uint32_t addr)
{
    return (addr & 0xf0000000) == 0xe0000000;
}

enum class InClassD : uint32_t
{
    In_ClassD_Net         =  0xf0000000,  /* These ones aren't really */
    In_ClassD_NShift      =  28,          /* net and host fields, but */
    In_ClassD_Host        =  0x0fffffff   /* routing needn't know.    */
};

inline bool In_Multicast(uint32_t addr)
{
    return In_ClassD(addr);
}

inline bool In_Experimental(uint32_t addr)
{
    return (addr & 0xf0000000) == 0xf0000000;
}
inline bool In_BadClass(uint32_t addr)
{
    return (addr & 0xf0000000) == 0xf0000000;
}

inline bool In_LinkLocal(uint32_t addr)
{
    return (addr & 0xffff0000) == 0xa9fe0000;
}
inline bool In_Loopback(uint32_t addr)
{
    return (addr & 0xff000000) == 0x7f000000;
}
inline bool In_ZeroNet(uint32_t addr)
{
    return (addr & 0xff000000) == 0;
}

inline bool In_Private(uint32_t addr)
{
    return ((addr & 0xff000000) == 0x0a000000) ||
           ((addr & 0xfff00000) == 0xac100000) ||
           ((addr & 0xffff0000) == 0xc0a80000);
}

inline bool In_Local_Group(uint32_t addr)
{
    return (addr & 0xffffff00) == 0xe0000000;
}

inline bool In_Any_Local(uint32_t addr)
{
    return In_LinkLocal(addr) || In_Local_Group(addr);
}

const uint32_t In_LoopbackNet   = 127;     /* official! */
const uint32_t In_Rfc3021_Mask  = 0xfffffffe;

/* Protocol Independent Multicast API [RFC3678] */
enum class Multicast : uint32_t
{
    Mcast_Join_Group          =  80,    /* join an any-source group */
    Mcast_Leave_Group         =  81,    /* leave all sources for group */
    Mcast_Join_Source_Group   =  82,    /* join a source-specific group */
    Mcast_Leave_Source_Group  =  83,    /* leave a single source */
    Mcast_Block_Source        =  84,    /* block a source */
    Mcast_Unblock_Source      =  85     /* unblock a source */
};

/*
 * Defaults and limits for options
 */
const uint32_t Ip_Default_Multicast_Ttl    = 1;    /* normally limit m'casts to 1 hop  */
const uint32_t Ip_Default_Multicast_Loop   = 1;    /* normally hear sends if a member  */

/*
 * The imo_membership vector for each socket is now dynamically allocated at
 * run-time, bounded by USHRT_MAX, and is reallocated when needed, sized
 * according to a power-of-two increment.
 */
const uint32_t Ip_Min_Memberships       = 31;
const uint32_t Ip_Max_Memberships       = 4095;
const uint32_t Ip_Max_Source_Filter     = 1024;  /* XXX to be unused */

/*
 * Default resource limits for IPv4 multicast source filtering.
 * These may be modified by sysctl.
 */
const uint32_t Ip_Max_Group_Src_Filter  = 512;   /* sources per group */
const uint32_t Ip_Max_Sock_Src_Filter   = 128;   /* sources per socket/group */
const uint32_t Ip_Max_Sock_Mute_Filter  = 128;   /* XXX no longer used */

/*
 * Argument structure for IP_ADD_MEMBERSHIP and IP_DROP_MEMBERSHIP.
 */
struct IpMreq
{
    InAddr imr_multiaddr;       /* IP multicast address of group */
    InAddr imr_interface;       /* local IP address of interface */
};

/*
 * Modified argument structure for IP_MULTICAST_IF, obtained from Linux.
 * This is used to specify an interface index for multicast sends, as
 * the IPv4 legacy APIs do not support this (unless IP_SENDIF is available).
 */
struct IpMreqn
{
    InAddr imr_multiaddr;       /* IP multicast address of group */
    InAddr imr_address;         /* local IP address of interface */
    int    imr_ifindex;                 /* Interface index; cast to uint32_t */
};

/*
 * Argument structure for IPv4 Multicast Source Filter APIs. [RFC3678]
 */
struct IpMreqSource
{
    InAddr imr_multiaddr;       /* IP multicast address of group */
    InAddr imr_sourceaddr;      /* IP address of source */
    InAddr imr_interface;       /* local IP address of interface */
};

/*
 * Argument structures for Protocol-Independent Multicast Source
 * Filter APIs. [RFC3678]
 */
struct GroupReq
{
    uint32_t           gr_interface;    /* interface index */
    SockAddrStorage    gr_group;        /* group address */
};

struct GroupSourceReq
{
    uint32_t           gsr_interface;   /* interface index */
    SockAddrStorage    gsr_group;       /* group address */
    SockAddrStorage    gsr_source;      /* source address */
};

/*
 * Argument for Ip_PortRange:
 * - which range to search when port is unspecified at bind() or connect()
 */
enum class IpPortRange : uint16_t
{
    Ip_PortRange_Default   = 0,         /* default range */
    Ip_PortRange_High      = 1,         /* "high" - request firewall bypass */
    Ip_PortRange_Low       = 2,         /* "low" - vouchsafe security */
};

}} /* nn::socket */
