﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

namespace nn {
namespace socket {

enum class RtFlag : uint32_t
{
    Rtf_None            = 0x0,
    Rtf_Up              = 0x1,          /* route usable */
    Rtf_Gateway         = 0x2,          /* destination is a gateway */
    Rtf_Host            = 0x4,          /* host entry (net otherwise) */
    Rtf_Reject          = 0x8,          /* host or net unreachable */
    Rtf_Dynamic         = 0x10,         /* created dynamically (by redirect) */
    Rtf_Modified        = 0x20,         /* modified dynamically (by redirect) */
    Rtf_Done            = 0x40,         /* message confirmed */
/*                      = 0x80,            unused, was Rtf_DelClone */
/*                      = 0x100,           unused, was Rtf_Cloning */
    Rtf_XResolve        = 0x200,        /* external daemon resolves name */
    Rtf_LlInfo          = 0x400,        /* DEPRECATED - exists ONLY for backward
                                           compatibility */
    Rtf_LlData          = 0x400,        /* used by apps to add/del L2 entries */
    Rtf_Static          = 0x800,        /* manually added */
    Rtf_BlackHole       = 0x1000,       /* just discard pkts (during updates) */
    Rtf_Proto2          = 0x4000,       /* protocol specific routing flag */
    Rtf_Proto1          = 0x8000,       /* protocol specific routing flag */
/*                      = 0x10000,         unused, was Rtf_PrCloning */
/*                      = 0x20000,         unused, was Rtf_WasCloned */
    Rtf_Proto3          = 0x40000,      /* protocol specific routing flag */
/*                      = 0x80000,         unused */
    Rtf_Pinned          = 0x100000,     /* route is immutable */
    Rtf_Local           = 0x200000,     /* route represents a local address */
    Rtf_Broadcast       = 0x400000,     /* route represents a bcast address */
    Rtf_Multicast       = 0x800000,     /* route represents a mcast address */
/*                      = 0x8000000,       and up unassigned */
    Rtf_Sticky          = 0x10000000,   /* always route dst->src */

    Rtf_Rnh_Locked      = 0x40000000,   /* radix node head is locked */

    Rtf_GwFlag_Compat   = 0x80000000,   /* a compatibility bit for interacting
                                           with existing routing apps */

/* Mask of Rtf flags that are allowed to be modified by Rtm_Change. */
    Rtf_FMask           = (Rtf_Proto1 | Rtf_Proto2 | Rtf_Proto3 | Rtf_BlackHole | \
                           Rtf_Reject | Rtf_Static | Rtf_Sticky)
};

/*
 * Bitmask values for rtm_inits and rmx_locks.
 */
enum class RtvFlag : uint32_t
{
    Rtv_None        = 0x0,
    Rtv_Mtu         = 0x1,          /* init or lock _mtu */
    Rtv_HopCount    = 0x2,          /* init or lock _hopcount */
    Rtv_Expire      = 0x4,          /* init or lock _expire */
    Rtv_RPipe       = 0x8,          /* init or lock _recvpipe */
    Rtv_SPipe       = 0x10,         /* init or lock _sendpipe */
    Rtv_SsThresh    = 0x20,         /* init or lock _ssthresh */
    Rtv_Rtt         = 0x40,         /* init or lock _rtt */
    Rtv_RttVar      = 0x80,         /* init or lock _rttvar */
    Rtv_Weight      = 0x100,        /* init or lock _weight */
};

/*
 * Bitmask values for rtm_addrs.
 */
enum class RtaFlag : int
{
    Rta_None    = 0x0,
    Rta_Dst     = 0x1,          /* destination sockaddr present */
    Rta_Gateway = 0x2,          /* gateway sockaddr present */
    Rta_NetMask = 0x4,          /* netmask sockaddr present */
    Rta_GenMask = 0x8,          /* cloning mask sockaddr present */
    Rta_Ifp     = 0x10,         /* interface name sockaddr present */
    Rta_Ifa     = 0x20,         /* interface addr sockaddr present */
    Rta_Author  = 0x40,         /* sockaddr for author of redirect */
    Rta_Brd     = 0x80,         /* for NEWADDR, broadcast or p-p dest addr */
};

struct RtMetrics
{
    RtvFlag     rmx_locks;              /* Kernel must leave these values alone */
    uint32_t    rmx_mtu;                /* MTU for this path */
    uint32_t    rmx_hopcount;           /* max hops expected */
    uint32_t    rmx_expire;             /* lifetime for route, e.g. redirect */
    uint32_t    rmx_recvpipe;           /* inbound delay-bandwidth product */
    uint32_t    rmx_sendpipe;           /* outbound delay-bandwidth product */
    uint32_t    rmx_ssthresh;           /* outbound gateway buffer limit */
    uint32_t    rmx_rtt;                /* estimated round trip time */
    uint32_t    rmx_rttvar;             /* estimated rtt variance */
    uint32_t    rmx_pksent;             /* packets sent using this route */
    uint32_t    rmx_weight;             /* route weight */
    uint32_t    rmx_filler[3];          /* will be used for T/TCP later */
};

/*
 * Routing statistics.
 */
struct RtStat
{
    int16_t    rts_badredirect;         /* bogus redirect calls */
    int16_t    rts_dynamic;             /* routes created by redirects */
    int16_t    rts_newgateway;          /* routes modified by redirects */
    int16_t    rts_unreach;             /* lookups which failed */
    int16_t    rts_wildcard;            /* lookups satisfied by a wildcard */
};

/*
 * Message types.
 */
enum RtMsgType : uint8_t
{
    Rtm_Add             = 0x1,          /* Add Route */
    Rtm_Delete          = 0x2,          /* Delete Route */
    Rtm_Change          = 0x3,          /* Change Metrics or flags */
    Rtm_Get             = 0x4,          /* Report Metrics */
    Rtm_Losing          = 0x5,          /* Kernel Suspects Partitioning */
    Rtm_Redirect        = 0x6,          /* Told to use different route */
    Rtm_Miss            = 0x7,          /* Lookup failed on this address */
    Rtm_Lock            = 0x8,          /* fix specified metrics */
/*                      = 0x9,          */
/*                      = 0xa,          */
    Rtm_Resolve         = 0xb,          /* req to resolve dst to LL addr */
    Rtm_NewAddr         = 0xc,          /* address being added to iface */
    Rtm_DelAddr         = 0xd,          /* address being removed from iface */
    Rtm_IfInfo          = 0xe,          /* iface going up/down etc. */
    Rtm_NewMAddr        = 0xf,          /* mcast group membership being added to if */
    Rtm_DelMAddr        = 0x10,         /* mcast group membership being deleted */
    Rtm_IfAnnounce      = 0x11,         /* iface arrival/departure */
    Rtm_Ieee80211       = 0x12,         /* IEEE80211 wireless event */
};

/*
 * Structures for routing messages.
 */
struct RtMsgHdr
{
    uint16_t    rtm_msglen;     /* to skip over non-understood messages */
    uint8_t     rtm_version;    /* future binary compatibility */
    RtMsgType   rtm_type;       /* message type */
    uint16_t    rtm_index;      /* index for associated ifp */
    RtFlag      rtm_flags;      /* flags, incl. kern & message, e.g. DONE */
    RtaFlag     rtm_addrs;      /* bitmask identifying sockaddrs in msg */
    int         rtm_pid;        /* identify sender */
    int         rtm_seq;        /* for sender to identify action */
    int         rtm_errno;      /* why failed */
    int         rtm_fmask;      /* bitmask used in Rtm_Change message */
    RtvFlag     rtm_inits;      /* which metrics we are initializing */
    RtMetrics   rtm_rmx;        /* metrics themselves */
};

const uint32_t Rtm_Version = 5;  /* Up the ante and ignore older versions */

/*
 * Index offsets for sockaddr array for alternate internal encoding.
 */
enum class Rtax : uint8_t
{
    Rtax_Dst        = 0,            /* destination sockaddr present */
    Rtax_Gateway    = 1,            /* gateway sockaddr present */
    Rtax_NetMask    = 2,            /* netmask sockaddr present */
    Rtax_GenMask    = 3,            /* cloning mask sockaddr present */
    Rtax_Ifp        = 4,            /* interface name sockaddr present */
    Rtax_Ifa        = 5,            /* interface addr sockaddr present */
    Rtax_Author     = 6,            /* sockaddr for author of redirect */
    Rtax_Brd        = 7,            /* for NEWADDR, broadcast or p-p dest addr */
    Rtax_Max        = 8             /* size of array to allocate */
};

struct RtAddrInfo
{
    int         rti_addrs;
    nn::socket::SockAddr *rti_info[static_cast<uint8_t>(Rtax::Rtax_Max)];
    int         rti_flags;
    struct      ifaddr *rti_ifa;
    struct      ifnet *rti_ifp;
};

/*
 * This macro returns the size of a struct sockaddr when passed
 * through a routing socket. Basically we round up sa_len to
 * a multiple of sizeof(long), with a minimum of sizeof(long).
 * The check for a NULL pointer is just a convenience, probably never used.
 * The case sa_len == 0 should only apply to empty structures.
 */
struct SockAddrIn;
inline size_t Sa_Size(SockAddrIn *sa)
{
    return (!(sa) || ((SockAddr *)(sa))->sa_len == 0) ?
        sizeof(long) :
        1 + ((((SockAddr *)(sa))->sa_len - 1) | (sizeof(long) - 1));
}
struct SockAddrDl;
inline size_t Sa_Size(SockAddrDl *sa)
{
    return (!(sa) || ((SockAddr *)(sa))->sa_len == 0) ?
        sizeof(long) :
        1 + ((((SockAddr *)(sa))->sa_len - 1) | (sizeof(long) - 1));
}

inline RtFlag operator | (RtFlag a, RtFlag b) NN_NOEXCEPT
{
    return static_cast<RtFlag>(static_cast<int>(a) | static_cast<int>(b));
}

inline RtFlag operator |= (RtFlag &a, RtFlag b) NN_NOEXCEPT
{
    return a = a | b;
}

inline RtFlag operator & (RtFlag a, RtFlag b) NN_NOEXCEPT
{
    return static_cast<RtFlag>(static_cast<int>(a) & static_cast<int>(b));
}

inline RtFlag operator &= (RtFlag &a, RtFlag b) NN_NOEXCEPT
{
    return a = a & b;
}

inline RtFlag operator ^ (RtFlag a, RtFlag b) NN_NOEXCEPT
{
    return static_cast<RtFlag>(static_cast<int>(a) ^ static_cast<int>(b));
}

inline RtFlag operator ^= (RtFlag &a, RtFlag b) NN_NOEXCEPT
{
    return a = a ^ b;
}

inline RtFlag operator ~ (RtFlag a) NN_NOEXCEPT
{
    return static_cast<RtFlag>(~static_cast<int>(a));
}

inline bool operator ! (RtFlag a) NN_NOEXCEPT
{
    return (RtFlag::Rtf_None == a);
}

inline RtvFlag operator | (RtvFlag a, RtvFlag b) NN_NOEXCEPT
{
    return static_cast<RtvFlag>(static_cast<int>(a) | static_cast<int>(b));
}

inline RtvFlag operator |= (RtvFlag &a, RtvFlag b) NN_NOEXCEPT
{
    return a = a | b;
}

inline RtvFlag operator & (RtvFlag a, RtvFlag b) NN_NOEXCEPT
{
    return static_cast<RtvFlag>(static_cast<int>(a) & static_cast<int>(b));
}

inline RtvFlag operator &= (RtvFlag &a, RtvFlag b) NN_NOEXCEPT
{
    return a = a & b;
}

inline RtvFlag operator ^ (RtvFlag a, RtvFlag b) NN_NOEXCEPT
{
    return static_cast<RtvFlag>(static_cast<int>(a) ^ static_cast<int>(b));
}

inline RtvFlag operator ^= (RtvFlag &a, RtvFlag b) NN_NOEXCEPT
{
    return a = a ^ b;
}

inline RtvFlag operator ~ (RtvFlag a) NN_NOEXCEPT
{
    return static_cast<RtvFlag>(~static_cast<int>(a));
}

inline bool operator ! (RtvFlag a) NN_NOEXCEPT
{
    return (RtvFlag::Rtv_None == a);
}

inline RtaFlag operator | (RtaFlag a, RtaFlag b) NN_NOEXCEPT
{
    return static_cast<RtaFlag>(static_cast<int>(a) | static_cast<int>(b));
}

inline RtaFlag operator |= (RtaFlag &a, RtaFlag b) NN_NOEXCEPT
{
    return a = a | b;
}

inline RtaFlag operator & (RtaFlag a, RtaFlag b) NN_NOEXCEPT
{
    return static_cast<RtaFlag>(static_cast<int>(a) & static_cast<int>(b));
}

inline RtaFlag operator &= (RtaFlag &a, RtaFlag b) NN_NOEXCEPT
{
    return a = a & b;
}

inline RtaFlag operator ^ (RtaFlag a, RtaFlag b) NN_NOEXCEPT
{
    return static_cast<RtaFlag>(static_cast<int>(a) ^ static_cast<int>(b));
}

inline RtaFlag operator ^= (RtaFlag &a, RtaFlag b) NN_NOEXCEPT
{
    return a = a ^ b;
}

inline RtaFlag operator ~ (RtaFlag a) NN_NOEXCEPT
{
    return static_cast<RtaFlag>(~static_cast<int>(a));
}

inline bool operator ! (RtaFlag a) NN_NOEXCEPT
{
    return (RtaFlag::Rta_None == a);
}

}} /* nn::socket */
