﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief サービスフレームワークで HIPC のポートとセッションを一括して管理する機能を提供します。
    @details
     このファイルは <nn/sf/sf_HipcServer.h> をインクルードすることでインクルードされます。
*/

#include <nn/sf/hipc/server/sf_HipcAllInOneServerManager.h>
#include <nn/sf/sf_HipcPortCommon.h>

#include <utility>
#include <nn/nn_Abort.h>

namespace nn { namespace sf {

/**
    @brief HipcSimpleAllInOneServerManager のオプションがとるデフォルトの型です。

    @see HipcSimpleAllInOneServerManager
*/
struct DefaultHipcSimpleAllInOneServerManagerOption
{
    /**
        @brief ポインタ転送に使用するバッファのサイズです。
    */
    static const size_t PointerTransferBufferSize = 0;

    /**
        @brief ProcessInvokeRequest が Defer できるかどうかを表します。
    */
    static const bool CanDeferInvokeRequest = false;

    /**
        @brief サブドメインの最大数です。
    */
    static const int SubDomainCountMax = 0;

    /**
        @brief サブドメイン内オブジェクトの最大数です。
    */
    static const int ObjectInSubDomainCountMax = 0;
};

namespace detail {

template <typename User>
class HipcSimpleAllInOneServerManagerOptionsGetter
{
private:

    typedef DefaultHipcSimpleAllInOneServerManagerOption Default;

    template <typename T>
    static decltype(T::PointerTransferBufferSize, User()) CheckPointerTransferBufferSize(T*) NN_NOEXCEPT;
    static Default CheckPointerTransferBufferSize(...) NN_NOEXCEPT;
    typedef decltype(CheckPointerTransferBufferSize(static_cast<User*>(nullptr))) OptionForPointerTransferBufferSize;

    template <typename T>
    static decltype(T::CanDeferInvokeRequest, User()) CheckCanDeferInvokeRequest(T*) NN_NOEXCEPT;
    static Default CheckCanDeferInvokeRequest(...) NN_NOEXCEPT;
    typedef decltype(CheckCanDeferInvokeRequest(static_cast<User*>(nullptr))) OptionForCanDeferInvokeRequest;

    template <typename T>
    static decltype(T::SubDomainCountMax, User()) CheckSubDomainCountMax(T*) NN_NOEXCEPT;
    static Default CheckSubDomainCountMax(...) NN_NOEXCEPT;
    typedef decltype(CheckSubDomainCountMax(static_cast<User*>(nullptr))) OptionForSubDomainCountMax;

    template <typename T>
    static decltype(T::ObjectInSubDomainCountMax, User()) CheckObjectInSubDomainCountMax(T*) NN_NOEXCEPT;
    static Default CheckObjectInSubDomainCountMax(...) NN_NOEXCEPT;
    typedef decltype(CheckObjectInSubDomainCountMax(static_cast<User*>(nullptr))) OptionForObjectInSubDomainCountMax;

public:

    static const size_t PointerTransferBufferSize = OptionForPointerTransferBufferSize::PointerTransferBufferSize;
    static const bool CanDeferInvokeRequest = OptionForCanDeferInvokeRequest::CanDeferInvokeRequest;
    static const int SubDomainCountMax = OptionForSubDomainCountMax::SubDomainCountMax;
    static const int ObjectInSubDomainCountMax = OptionForObjectInSubDomainCountMax::ObjectInSubDomainCountMax;

};

template <typename User>
const size_t HipcSimpleAllInOneServerManagerOptionsGetter<User>::PointerTransferBufferSize;

template <typename User>
const bool HipcSimpleAllInOneServerManagerOptionsGetter<User>::CanDeferInvokeRequest;

template <typename User>
const int HipcSimpleAllInOneServerManagerOptionsGetter<User>::SubDomainCountMax;

template <typename User>
const int HipcSimpleAllInOneServerManagerOptionsGetter<User>::ObjectInSubDomainCountMax;

}

class HipcServerSessionManagerCommonBase
{
public:

    virtual sf::hipc::server::HipcServerSessionManager* CastHipcServerSessionManagerInternal() NN_NOEXCEPT = 0;

};

/**
    @brief HIPC のポートとセッションを一括して管理するクラステンプレートです。

    @tparam SessionCountMax 管理する最大セッション数を指定します。
    @tparam PortCountMax 管理する最大ポート数を指定します。
    @tparam Option その他のオプションを指定します。この型が含むことのできる変数はデフォルト型引数である DefaultHipcSimpleAllInOneServerManagerOption を参照してください。

    @details
     このクラステンプレートの仕様は今後変更される(破壊的な変更も含む)可能性があります。
     特に、現実装は終了処理に関して未実装であり、正しい動作は保証されていません。

     内部にセッションとポート管理用の領域を持つため、
     引数の値によってはサイズが大きくなります。
     オブジェクトインスタンスをスタックなどに置く際には十分に注意してください。

     各メンバ関数のスレッド安全性は以下の通りです。

     - RegisterObjectForPort() と InitializePort() はスレッドセーフではありません。Start() 呼び出し前に、一つのスレッドから呼び出すようにしてください。
     - Wait() は複数のスレッドから同時に呼び出しても安全です。
     - 異なる os::MultiWaitHolderType* に対する ProcessAcceptRequest(), ProcessInvokeRequest(), ProcessInvokeRequestWithDefer() の呼び出しは、複数のスレッドから同時に呼び出しても安全です。
     - 同一の os::MultiWaitHolderType* に対する ProcessAcceptRequest(), ProcessInvokeRequest(), ProcessInvokeRequestWithDefer() の呼び出しは、複数のスレッドから同時に呼び出すことはできません。

     このクラスは、サブドメインに対応することができます。
     サブドメインに対応するためには、型引数 Option に最大ドメイン数 SubDomainCountMax と、
     最大サブドメイン内オブジェクト数 ObjectInSubDomainCountMax を指定する必要があります。

    @see DefaultHipcSimpleAllInOneServerManagerOption
*/
template <size_t SessionCountMax, size_t PortCountMax, typename Option = DefaultHipcSimpleAllInOneServerManagerOption>
class HipcSimpleAllInOneServerManager
    : private hipc::server::HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, detail::HipcSimpleAllInOneServerManagerOptionsGetter<Option>>
    , public HipcServerSessionManagerCommonBase
{
private:

    typedef detail::HipcSimpleAllInOneServerManagerOptionsGetter<Option> EffectiveOption;
    typedef hipc::server::HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, detail::HipcSimpleAllInOneServerManagerOptionsGetter<Option>> Base;

public:
    virtual ~HipcSimpleAllInOneServerManager() NN_NOEXCEPT
    {
    }

    //! @name オブジェクトのポートへの登録
    //! @{

    /**
        @brief サービス名をシステムに登録し、サービス名によって指定したオブジェクト参照を得られるようにします。

        @tparam Interface オブジェクトが実装するインターフェイスを指定します。
        @param[in] p オブジェクトへの共有ポインタを指定します。
        @param[in] sessionCountMax このオブジェクトに対して作成できるセッションの最大数を指定します。
        @param[in] serviceName サービス名を指定します。

        @pre InitializePort() と RegisterObjectForPort() の呼び出し回数が PortCountMax 未満である
        @pre Start() がまだ呼ばれていない
        @pre static_cast<bool>(p) == true
        @pre sessionCountMax > 0
        @pre std::strlen(serviceName) > 0
        @pre std::strlen(serviceName) <= HipcServiceNameLengthMax

        @retresult
        @endretresult

        @details
         サービス名をシステムに登録し、クライアントがこのサービス名に対して CreateHipcProxyByName() を呼ぶことで、
         サービスオブジェクト p に対するプロキシを得ることができるようにします。

         このポートからは同時に最大 sessionCountMax 個のセッションを作成できます。
         Start() を呼ぶ前に呼ぶ必要があります。

         同一プロセスで同一のサービス名に対し、この関数と CreateHipcProxyByName() の両方を呼ぶ場合には、
         必ずこの関数を先に呼ぶようにしてください。
         この関数を後に呼んでしまうと、デッドロックします。

         この関数で登録した p の参照は、このクラスインスタンスが破棄されるまで保持されます。
    */
    template <typename Interface>
    Result RegisterObjectForPort(SharedPointer<Interface> p, int32_t sessionCountMax, const char* serviceName) NN_NOEXCEPT;

    /**
        @brief (内部用) ポートに対しオブジェクトを登録します。

        @tparam Interface オブジェクトが実装するインターフェイスを指定します。
        @param[in] p オブジェクトへの共有ポインタを指定します。
        @param[in] portHandle 管理するサーバポートハンドルを指定します。

        @pre InitializePort() と RegisterObjectForPort() の呼び出し回数が PortCountMax 未満である
        @pre Start() がまだ呼ばれていない
        @pre static_cast<bool>(p) == true

        @details
         ポートをシステムに登録し、クライアントがこのポートに対して CreateHipcProxyByPort() を呼ぶことで、
         サービスオブジェクト p に対するプロキシを得ることができるようにします。

         この関数に渡した portHandle の管理責任はこの関数を呼んだ側にあり、
         不要になった際には、手動で閉じる必要があります。

         この関数で登録した p の参照は、このクラスインスタンスが破棄されるまで保持されます。
    */
    template <typename Interface>
    void RegisterObjectForPort(SharedPointer<Interface> p, HipcServerPortHandle portHandle) NN_NOEXCEPT;

    //! @}

    //! @name (非推奨) ポート初期化
    //! @{

    /**
        @brief (非推奨) サービス名をシステムに登録し、対応するポートを待機可能にします。

        @deprecated 本関数は非推奨です。代わりに RegisterObjectForPort() を使用してください。

        @param[in] acceptIndex ユーザが任意に決定した番号を指定します。
        @param[in] sessionCountMax このポートに対して作成できるセッションの最大数を指定します。
        @param[in] serviceName サービス名を指定します。

        @pre InitializePort() と RegisterObjectForPort() の呼び出し回数が PortCountMax 未満である
        @pre Start() がまだ呼ばれていない
        @pre sessionCountMax > 0
        @pre std::strlen(serviceName) > 0
        @pre std::strlen(serviceName) <= HipcServiceNameLengthMax

        @retresult
        @endretresult

        @details
         サービス名をシステムに登録し、対応するポートを待機可能にします。
         このポートからは同時に最大 sessionCountMax 個のセッションを作成できます。
         Start() を呼ぶ前に呼ぶ必要があります。

         acceptIndex にはユーザが決定した任意の値を指定することが可能です。
         portHandle に対応するポートに接続リクエストがあった際には、
         ここで指定した acceptIndex の値によって OnNeedsToAccept() が呼ばれます。
         この関数を使用する際には OnNeedsToAccept() のオーバライドが必要です。

         同一プロセスで同一のサービス名に対し、この関数と CreateHipcProxyByName() の両方を呼ぶ場合には、
         必ずこの関数を先に呼ぶようにしてください。
         この関数を後に呼んでしまうと、デッドロックします。
    */
    Result InitializePort(int acceptIndex, int32_t sessionCountMax, const char* serviceName) NN_NOEXCEPT;

    /**
        @brief (非推奨)(内部用) ポートを直接登録します。

        @deprecated 本関数は非推奨です。代わりに RegisterObjectForPort() を使用してください。

        @param[in] acceptIndex ユーザが任意に決定した番号を指定します。
        @param[in] portHandle 管理するサーバポートハンドルを指定します。

        @pre InitializePort() と RegisterObjectForPort() の呼び出し回数が PortCountMax 未満である
        @pre Start() がまだ呼ばれていない

        @details
         ポートを登録し、待機可能な状態にします。
         Start() を呼ぶ前に呼ぶ必要があります。

         acceptIndex にはユーザが決定した任意の値を指定することが可能です。
         portHandle に対応するポートに接続リクエストがあった際には、
         ここで指定した acceptIndex の値によって OnNeedsToAccept() が呼ばれます。
         この関数を使用する際には OnNeedsToAccept() のオーバライドが必要です。

         この関数に渡した portHandle の管理責任はこの関数を呼んだ側にあり、
         不要になった際には、手動で閉じる必要があります。
    */
    void InitializePort(int acceptIndex, HipcServerPortHandle portHandle) NN_NOEXCEPT;

    //! @}

    //! @name 開始と終了
    //! @{

    /**
        @brief 処理を開始します。
    */
    void Start() NN_NOEXCEPT;

    /**
        @brief 処理の中止をリクエストします。

        @pre Start() が呼ばれている

        @details
         この関数の仕様は今後変更される(破壊的な変更も含む)可能性があります。

         Wait() のブロックを解除するようリクエストをします。
         この場合 Wait() が nullptr で返ることがあります。
         また、これによって LoopAuto() 関数が返ります。

         現実装は終了処理に関して未実装であり、正しい動作は保証されていません。
         現状、正しく動かすためには、すべてのセッションが閉じられている状態で呼んでください。
    */
    void RequestStop() NN_NOEXCEPT;

    //! @}

    //! @name 待機
    //! @{

    /**
        @brief ポートやセッションがシグナルされるのを待機します。

        @return シグナルされたオブジェクトが関連付けられた同時待機オブジェクトへのポインタを返します。

        @pre Start() が呼ばれている

        @details
         ポートやセッションは os::MultiWaitHolderType に紐づけられて管理されており、
         この関数内で同時にシグナルされるのを待機できます。

         同じインスタンスへのこの関数の呼び出しは互いに排他されています。
         よって、この関数を同時に複数のスレッドから呼ぶことが可能です。

         この返り値に対し os::GetMultiWaitHolderUserData() を呼び、
         シグナルされたオブジェクトがポートであるかセッションであるかを判定することができます。

         os::GetMultiWaitHolderUserData() の返り値が AcceptTag の場合にはポートがシグナルされており、
         ProcessAcceptRequest() を呼ぶことでポートへの処理を行うことができます。
         InitializePort() を呼んでいない場合には、これに該当することはありません。

         os::GetMultiWaitHolderUserData() の返り値が InvokeTag の場合にはセッションがシグナルされており、
         ProcessInvokeRequest() などを呼ぶことでセッションへの処理を行うことができます。

         os::GetMultiWaitHolderUserData() の返り値がそれ以外のときは、
         AddUserWaitHolder() に渡された同期オブジェクトがシグナルされたことを示します。
         返されたオブジェクトは次の Wait() の呼び出しの際に待機対象にはなりません。
         再度待機する必要がある際には、再度 AddUserWaitHolder() を呼びなおす必要があります。

         なお RequestStop() が呼ばれた際には nullptr が返ります。

        @see os::GetMultiWaitHolderUserData()
    */
    os::MultiWaitHolderType* Wait() NN_NOEXCEPT;

    //! @}

    /**
        @brief (非推奨) Wait() の返り値がポートであることを識別するための定数です。

        @deprecated 本関数は非推奨な機能を使用している場合にのみ使用する必要があるものです。
    */
    static const uintptr_t AcceptTag = 2;

    //! @name (非推奨) ポート操作
    //! @{

    /**
        @brief (非推奨) シグナルされたオブジェクトをポートとして処理します。

        @deprecated 本関数は非推奨な機能を使用している場合にのみ使用する必要があるものです。

        @param[in] p Wait() から返されたポートオブジェクトへのポインタを指定します。

        @pre p が Wait() から返されたもので、未処理
        @pre \link os::GetMultiWaitHolderUserData \endlink(p) == \link AcceptTag \endlink

        @retresult
        @endretresult

        @details
         p をポートして処理します。

         p に対応するポートに対して OnNeedsToAccept() を呼び出します。
         OnNeedsToAccept() 内では AcceptImpl() が呼ばれることが想定されており、
         ポートを受理して得たセッションに、オブジェクトを結びつけ、
         セッションへのリクエストをオブジェクトへの呼び出しとするように設定します。
    */
    Result ProcessAcceptRequest(os::MultiWaitHolderType* p) NN_NOEXCEPT;

    //! @}

    /**
        @brief Wait() の返り値がセッションであることを識別するための定数です。
    */
    static const uintptr_t InvokeTag = 1;

    //! @name リクエスト操作
    //! @{

    /**
        @brief シグナルされたオブジェクトをセッションとして、HIPC 用 TLS メッセージバッファ上のメッセージを処理します。

        @param[in] p Wait() から返されたポートオブジェクトへのポインタを指定します。

        @retresult
        @endretresult

        @pre \link os::GetMultiWaitHolderUserData \endlink(p) == \link InvokeTag \endlink
        @pre p で表されるセッションが未受信状態

        @post p で表されるセッションが待機状態
        @post メッセージに応じた p に対する処理が完了している

        @details
         p であらわされるセッションからリクエストを受信し、メッセージの内容に従ってインターフェイスメソッドを呼びだし、メッセージを返信します。
         返信後、セッションは再び待機状態になります。

         インターフェイスメソッドの返り値などでオブジェクトが返された場合、
         オブジェクトに対応するセッションが作成され、そのセッションも待機状態になり、
         返されたオブジェクトへの呼び出しが HIPC 経由でできるようになります。

         サービスオブジェクトの実装が何らかの理由によって処理を後回しにすると判断し DeferProcess() を返した場合、この関数は処理を返さず停止します。
         このような可能性がある場合には、かわりに ProcessInvokeRequestWithDefer() を呼んでください。

        @see ProcessInvokeRequestWithDefer()
    */
    Result ProcessInvokeRequest(os::MultiWaitHolderType* p) NN_NOEXCEPT;

    /**
        @brief シグナルされたオブジェクトをセッションとして、HIPC 用 TLS メッセージバッファ上のメッセージを処理します。処理の遅延が可能です。

        @param[in] p Wait() から返されたポートオブジェクトへのポインタを指定します。

        @retresult
            @handleresult{ResultProcessDeferred,処理が後回しにされました。再度呼び出しなおす必要があります。}
        @endretresult

        @pre Option::CanDeferInvokeRequest == true (static assert)
        @pre \link os::GetMultiWaitHolderUserData \endlink(p) == \link InvokeTag \endlink
        @pre p で表されるセッションが未受信状態または処理延期状態

        @post p で表されるセッションが待機状態
        @post メッセージに応じた p に対する処理が完了している

        @details
         p であらわされるセッションが未受信状態のとき、p であらわされるセッションからリクエストを受信し、メッセージの内容に従ってインターフェイスメソッドを呼びだし、メッセージを返信します。
         返信後、セッションは再び待機状態になります。

         p であらわされるセッションが処理延期状態のとき、メッセージの内容に従ってインターフェイスメソッドを呼びだし、メッセージを返信します。
         返信後、セッションは再び待機状態になります。

         インターフェイスメソッドの返り値などでオブジェクトが返された場合、
         オブジェクトに対応するセッションが作成され、そのセッションも待機状態になり、
         返されたオブジェクトへの呼び出しが HIPC 経由でできるようになります。

         サービスオブジェクトの実装が何らかの理由によって処理を後回しにすると判断し DeferProcess() を返した場合、
         この関数から ResultProcessDeferred が返ります。
         この際 p で表されるセッションは処理延期状態となります。
         別途処理を行える条件を満たすなどした上で、再度本関数を呼び出しなおす必要があります。

         サービスオブジェクトの実装が DeferProcess() を返さないことがわかっている場合には、
         かわりに ProcessInvokeRequest() 関数を呼んだほうが効率的です。

        @see ProcessInvokeRequest()
    */
    Result ProcessInvokeRequestWithDefer(os::MultiWaitHolderType* p) NN_NOEXCEPT;

    //! @}

    //! @name 統合ユーティリティ
    //! @{

    /**
        @brief シグナルされたオブジェクトに対しポートとセッションとを自動判別して処理します。

        @param[in] p Wait() から返されたオブジェクトへのポインタを指定します。

        @retresult
        @endretresult

        @details
         p がポートであるかセッションであるかを判別して、
         ProcessInvokeRequest(os::MultiWaitHolderType*) と ProcessAcceptRequest(os::MultiWaitHolderType*) を呼び分けるユーティリティ関数です。

        @see ProcessInvokeRequest(os::MultiWaitHolderType*), ProcessAcceptRequest(os::MultiWaitHolderType*)
    */
    Result ProcessAuto(os::MultiWaitHolderType* p) NN_NOEXCEPT;

    /**
        @brief (未サポート) 待機と処理を行います。

        @details
         未サポートです。
    */
    void WaitAndProcessAuto() NN_NOEXCEPT;

    /**
        @brief 待機と処理をループします。

        @details
         この関数仕様は今後変更される(破壊的な変更も含む)可能性があります。

         WaitAndProcessAuto() の呼び出しを RequestStop() の呼び出しまでの間ループして行います。

         この関数を複数のスレッドから呼ぶことで、複数の呼び出し処理を複数のスレッドで同時に行うことができます。

        @see WaitAndProcessAuto(), RequestStop()
    */
    void LoopAuto() NN_NOEXCEPT;

    //! @}

    //! @name ユーザ待機オブジェクト追加
    //! @{

    /**
        @brief 処理スレッドで同時に待機する待機オブジェクトを追加します。

        @param[in] p 待機オブジェクトホルダーへのポインタを指定します。

        @pre p が初期化状態である
        @pre os::GetMultiWaitHolderUserData (p) != InvokeTag
        @pre os::GetMultiWaitHolderUserData (p) != AcceptTag

        @details
         Wait() での待機対象に、待機オブジェクトホルダー p に関連付けられた待機オブジェクトを追加します。

         os::SetMultiWaitHolderUserData によって設定される p のユーザデータは、
         InvokeTag や AcceptTag と異なる値が設定されている必要があります。

         この関数によって待機オブジェクトが登録されている場合には、
         シグナルされたオブジェクトに対する手動の処理が必要となります(LoopAuto() を呼ぶことはできません)。
         すなわち、 Wait() によってシグナルされたオブジェクトを取得した後、
         そのオブジェクトに対し os::GetMultiWaitHolderUserData を呼んでユーザデータを取得し、
         その値によってユーザ独自の処理を行う必要があります。
         なお、ユーザデータがそれ以外の InvokeTag や AcceptTag だった場合には、
         ProcessAuto() に渡すことでデフォルトの動作を行うことが可能です。

         AddUserWaitHolder で渡した待機オブジェクトホルダーが Wait() から返った場合、
         そのままでは、次の Wait() の対象には含まれません。
         再度待機対象に加える必要がある場合には、再度本関数を呼んでください。

        @see Wait(), ProcessAuto(), os::SetMultiWaitHolderUserData(), os::GetMultiWaitHolderUserData()
    */
    void AddUserWaitHolder(os::MultiWaitHolderType* p) NN_NOEXCEPT;

    //! @}

    //! @name ManagerHandler 設定
    //! @{

    /**
        @brief IHipcServerSessionManagerHandler を設定します。

        @param[in] pManagerHandler IHipcServerSessionManagerHandler へのポインタを指定します。
    */
    void SetManagerHandler(IHipcServerSessionManagerHandler* pManagerHandler) NN_NOEXCEPT;

    //! @}

#if defined(NN_BUILD_CONFIG_OS_WIN32)

    void SetEnabledIpcOnWindows(bool enabled) NN_NOEXCEPT;

#endif

protected:

    /**
        @brief (非推奨) ポートの内部型を表します。

        @deprecated 本関数は非推奨な機能を使用している場合にのみ使用する必要があるものです。

        @details
         内部型へのエイリアスです。
         OnNeedsToAccept() の実装以外の用途に使用しないでください。
    */
    typedef typename Base::PortForAllInOne AcceptTarget;

    //! @name オーバライド可能関数
    //! @{

    /**
        @brief (非推奨)(オーバライド可) ポートに対する要求を処理します。

        @deprecated 本関数は非推奨な機能を使用している場合にのみ使用する必要があるものです。

        @param[in] acceptIndex InitializePort() で指定したポートインデックスが指定されます。
        @param[in] pAcceptTarget 下位レイヤに渡すためのデータが指定されます。

        @pre acceptIndex が InitializePort() で指定された値である
        @post 指定されたポートに対して AcceptImpl() が適切に呼ばれている

        @return 内部で呼び出した AcceptImpl() の返り値をそのまま返します。

        @details
         この関数は、InitializePort() が呼ばれていた際に ProcessAcceptRequest() の実装内から呼び出されます。

         InitializePort() を呼ぶ場合には、この関数を適切にオーバライドする必要があります。
         InitializePort() を呼ばない場合には、この関数をオーバライドする必要はありません。

         この関数はユーザが実装しオーバライドする必要があります。
         実装内では、適切なオブジェクトを引数として AcceptImpl() を呼ぶ必要があります。
         複数のポートを管理している際には、acceptIndex の値を見ることで、
         どのポートに対する要求であるかを判別し、オブジェクトを選択することができます。
         引数 pAcceptTarget に対しては処理をせず、必ずそのまま AcceptImpl() に渡してください。
         必ず AcceptImpl() の返り値をそのまま return してください。

         この関数を直接呼ばないでください。

        @see AcceptImpl()
    */
    virtual Result OnNeedsToAccept(int acceptIndex, AcceptTarget* pAcceptTarget) NN_NOEXCEPT NN_OVERRIDE;

    /**
        @brief (試験的機能)(オーバライド可) タグに応じた HipcServerSessionManagerCommonBase を取得します。

        @param[in] tag タグ

        @return タグに応じた HipcServerSessionManagerCommonBase を返します。

        @details
         本関数は試験的な機能です。通常は使用しないでください。
    */
    virtual HipcServerSessionManagerCommonBase* GetHipcServerSessionManagerCommonBase(uint32_t tag) NN_NOEXCEPT
    {
        NN_UNUSED(tag);
        return this;
    }

    //! @}

    //! @name オーバロード実装用
    //! @{

    /**
        @brief (非推奨)(オーバライド関数実装用) 指定されたポートを accept し、生成されたセッションとオブジェクトとを結びつけます。

        @deprecated 本関数は非推奨な機能を使用している場合にのみ使用する必要があるものです。

        @tparam Interface オブジェクトが実装するインターフェイスを指定します。
        @param[in] pAcceptTarget OnNeedsToAccept() の引数を指定します。
        @param[in] p オブジェクトへの共有ポインタを指定します。

        @details
         この関数は OnNeedsToAccept() 内から呼ばれることを想定しています。
         これ以外の場所から呼ばないようにしてください。

        @see OnNeedsToAccept()
    */
    template <typename Interface>
    Result AcceptImpl(AcceptTarget* pAcceptTarget, SharedPointer<Interface> p) NN_NOEXCEPT;

    //! @}

    // 互換性維持用
    typedef AcceptTarget PortObjectImpl;
    typedef AcceptTarget PortForAllInOne;

private:

    static_assert(AcceptTag == Base::PortTag, "[SF-Internal]");
    static_assert(InvokeTag == Base::SessionTag, "[SF-Internal]");

    virtual sf::hipc::server::HipcServerSessionManager* CastHipcServerSessionManagerInternal() NN_NOEXCEPT NN_OVERRIDE
    {
        return this;
    }

    virtual sf::hipc::server::HipcServerSessionManager* GetHipcServerSessionManagerInternal(uint32_t tag) NN_NOEXCEPT NN_OVERRIDE final
    {
        return GetHipcServerSessionManagerCommonBase(tag)->CastHipcServerSessionManagerInternal();
    }

};

// 以下実装
// TODO: 後に .cpp に移す

template <size_t SessionCountMax, size_t PortCountMax, typename Option>
template <typename Interface>
inline void HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, Option>::RegisterObjectForPort(SharedPointer<Interface> p, HipcServerPortHandle portHandle) NN_NOEXCEPT
{
    return Base::RegisterObjectForPort(std::move(p), portHandle);
}

template <size_t SessionCountMax, size_t PortCountMax, typename Option>
template <typename Interface>
inline Result HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, Option>::RegisterObjectForPort(SharedPointer<Interface> p, int32_t sessionCountMax, const char* serviceName) NN_NOEXCEPT
{
    return Base::RegisterObjectForPort(std::move(p), sessionCountMax, serviceName);
}

template <size_t SessionCountMax, size_t PortCountMax, typename Option>
inline void HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, Option>::InitializePort(int acceptIndex, HipcServerPortHandle portHandle) NN_NOEXCEPT
{
    return Base::InitializePort(acceptIndex, portHandle);
}

template <size_t SessionCountMax, size_t PortCountMax, typename Option>
inline Result HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, Option>::InitializePort(int acceptIndex, int32_t sessionCountMax, const char* serviceName) NN_NOEXCEPT
{
    return Base::InitializePort(acceptIndex, sessionCountMax, serviceName);
}

template <size_t SessionCountMax, size_t PortCountMax, typename Option>
inline void HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, Option>::Start() NN_NOEXCEPT
{
    return Base::Start();
}

template <size_t SessionCountMax, size_t PortCountMax, typename Option>
inline void HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, Option>::RequestStop() NN_NOEXCEPT
{
    return Base::RequestStop();
}

template <size_t SessionCountMax, size_t PortCountMax, typename Option>
inline os::MultiWaitHolderType* HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, Option>::Wait() NN_NOEXCEPT
{
    return Base::WaitSignaled();
}

template <size_t SessionCountMax, size_t PortCountMax, typename Option>
const uintptr_t HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, Option>::AcceptTag;

template <size_t SessionCountMax, size_t PortCountMax, typename Option>
inline Result HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, Option>::ProcessAcceptRequest(os::MultiWaitHolderType* p) NN_NOEXCEPT
{
    return Base::ProcessAsPort(p);
}

template <size_t SessionCountMax, size_t PortCountMax, typename Option>
const uintptr_t HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, Option>::InvokeTag;

template <size_t SessionCountMax, size_t PortCountMax, typename Option>
inline Result HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, Option>::ProcessInvokeRequest(os::MultiWaitHolderType* p) NN_NOEXCEPT
{
    return Base::ProcessInvokeRequest(p);
}

template <size_t SessionCountMax, size_t PortCountMax, typename Option>
inline Result HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, Option>::ProcessInvokeRequestWithDefer(os::MultiWaitHolderType* p) NN_NOEXCEPT
{
    static_assert(EffectiveOption::CanDeferInvokeRequest, "[SF-HIPC-InvalidServerSettings:CannotDefer] Requires to call this function: Option::CanDeferInvokeRequest == true");
    return Base::ProcessInvokeRequestWithDefer(p);
}

template <size_t SessionCountMax, size_t PortCountMax, typename Option>
inline Result HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, Option>::ProcessAuto(os::MultiWaitHolderType* p) NN_NOEXCEPT
{
    return Base::ProcessAuto(p);
}

template <size_t SessionCountMax, size_t PortCountMax, typename Option>
inline void HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, Option>::WaitAndProcessAuto() NN_NOEXCEPT
{
    Base::WaitAndProcessAuto();
}

template <size_t SessionCountMax, size_t PortCountMax, typename Option>
inline void HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, Option>::LoopAuto() NN_NOEXCEPT
{
    Base::LoopAuto();
}

template <size_t SessionCountMax, size_t PortCountMax, typename Option>
inline Result HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, Option>::OnNeedsToAccept(int, AcceptTarget*) NN_NOEXCEPT
{
    NN_ABORT("[SF-HIPC-InvalidUserImplementation:OnNeedsAccept] OnNeedsToAccept must be overridden when using 'acceptIndex'.");
}

template <size_t SessionCountMax, size_t PortCountMax, typename Option>
template <typename Interface>
inline Result HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, Option>::AcceptImpl(AcceptTarget* pAcceptTarget, SharedPointer<Interface> p) NN_NOEXCEPT
{
    return Base::AcceptImpl(pAcceptTarget, std::move(p));
}

template <size_t SessionCountMax, size_t PortCountMax, typename Option>
inline void HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, Option>::AddUserWaitHolder(os::MultiWaitHolderType* p) NN_NOEXCEPT
{
    Base::AddUserWaitHolder(p);
}


template <size_t SessionCountMax, size_t PortCountMax, typename Option>
inline void HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, Option>::SetManagerHandler(IHipcServerSessionManagerHandler* pManagerHandler) NN_NOEXCEPT
{
    Base::SetManagerHandler(pManagerHandler);
}

#if defined(NN_BUILD_CONFIG_OS_WIN32)

template <size_t SessionCountMax, size_t PortCountMax, typename Option>
inline void HipcSimpleAllInOneServerManager<SessionCountMax, PortCountMax, Option>::SetEnabledIpcOnWindows(bool enabled) NN_NOEXCEPT
{
    if (enabled)
    {
        Base::SetHipcServerApiModel(hipc::server::GetWindowsIpcEnabledServerApiModel());
    }
    else
    {
        Base::SetHipcServerApiModel(hipc::server::GetDefaultServerApiModel());
    }
}

#endif

}}
