﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
    @file
    @brief サービスフレームワークで使用される型なしバッファの型を定義します。
    @details
    このファイルを直接インクルードする必要はありません。 <nn/sf/sf_Types.h> をインクルードしてください。
*/

#include <nn/nn_Common.h>
#include <type_traits>
#include <nn/util/util_Span.h>

namespace nn { namespace sf {

namespace detail {

    struct PointerAndSize
    {
        uintptr_t pointer;
        size_t size;
    };

    struct BufferBaseTag
    {
    };

} // detail

/**
    @brief サービスインターフェイスで型なしの入力バッファを表すためのクラスです。

    @details
     この型であらわされたデータは、型がないことを前提としています。
     このバッファから得られたデータを解釈しないようにしてください。
*/
class InBuffer
    : public detail::BufferBaseTag
{
public:

    /**
        @brief コンストラクタ: 先頭アドレスとサイズを取って初期化します。

        @param[in] p 先頭アドレスを指定します。
        @param[in] size バッファのサイズを指定します。

        @pre [p, p + size) の領域を読むことができる。
        @post this->GetPointerUnsafe() == p
        @post this->GetSize() == size
    */
    InBuffer(const char* p, size_t size) NN_NOEXCEPT
        : m_Address(p)
        , m_Size(size)
    {
    }

    /**
        @brief バッファの先頭アドレスを取得します。

        @return バッファの先頭アドレスを返します。
    */
    const char* GetPointerUnsafe() const NN_NOEXCEPT
    {
        return m_Address;
    }

    /**
        @brief バッファのサイズを取得します。

        @return バッファのサイズを返します。
    */
    size_t GetSize() const NN_NOEXCEPT
    {
        return m_Size;
    }

    /**
        @brief 部分バッファを取得します。

        @param[in] offset 部分バッファのオフセット
        @param[in] size 部分バッファのサイズ

        @return 部分バッファを返します。

        @pre offset + size < GetSize()
    */
    InBuffer MakePart(ptrdiff_t offset, size_t size) const NN_NOEXCEPT
    {
        return InBuffer(m_Address + offset, size);
    }

    // 内部用
    NN_IMPLICIT InBuffer(const detail::PointerAndSize& pas) NN_NOEXCEPT
        : m_Address(reinterpret_cast<decltype(m_Address)>(pas.pointer))
        , m_Size(pas.size)
    {
    }

private:

    const char* m_Address;
    size_t m_Size;

};

/**
    @brief サービスインターフェイスで型なしの入力バッファを表すためのクラスです。

    @details
     この型であらわされたデータは、型がないことを前提としています。
*/
class OutBuffer
    : public detail::BufferBaseTag
{
public:

    /**
        @brief コンストラクタ: 先頭アドレスとサイズを取って初期化します。

        @param[in] p 先頭アドレスを指定します。
        @param[in] size バッファのサイズを指定します。

        @pre [p, p + size) の領域に書き込むことができる。
        @post this->GetPointerUnsafe() == p
        @post this->GetSize() == size
    */
    OutBuffer(char* p, size_t size) NN_NOEXCEPT
        : m_Address(p)
        , m_Size(size)
    {
    }

    /**
        @brief バッファの先頭アドレスを取得します。

        @return バッファの先頭アドレスを返します。
    */
    char* GetPointerUnsafe() const NN_NOEXCEPT
    {
        return m_Address;
    }

    /**
        @brief バッファのサイズを取得します。

        @return バッファのサイズを返します。
    */
    size_t GetSize() const NN_NOEXCEPT
    {
        return m_Size;
    }

    /**
        @brief 部分バッファを取得します。

        @param[in] offset 部分バッファのオフセット
        @param[in] size 部分バッファのサイズ

        @return 部分バッファを返します。

        @pre offset + size < GetSize()
    */
    OutBuffer MakePart(ptrdiff_t offset, size_t size) const NN_NOEXCEPT
    {
        return OutBuffer(m_Address + offset, size);
    }

    // 内部用
    NN_IMPLICIT OutBuffer(const detail::PointerAndSize& pas) NN_NOEXCEPT
        : m_Address(reinterpret_cast<decltype(m_Address)>(pas.pointer))
        , m_Size(pas.size)
    {
    }

private:

    char* m_Address;
    size_t m_Size;

};

/**
    @brief サービスインターフェイスで型付きの入力配列を表すためのクラスです。

    @tparam T 配列の要素型
*/
template <typename T>
class InArray
    : public detail::BufferBaseTag
{
public:

    /**
        @brief デフォルトコンストラクタ: 長さ 0 の配列として初期化します。
    */
    InArray() NN_NOEXCEPT
        : m_Data(nullptr), m_Length(0)
    {
    }

    /**
        @brief コンストラクタ: 配列の先頭要素へのポインタと配列の長さを指定して初期化します。

        @param[in] data 配列の先頭要素へのポインタを指定します。
        @param[in] length 配列の長さを指定します。
    */
    InArray(const T* data, size_t length) NN_NOEXCEPT
        : m_Data(data), m_Length(length)
    {
    }

    /**
        @brief コンストラクタ: 配列の先頭要素と終端要素の次へのポインタを指定して初期化します。

        @param[in] begin 配列の先頭要素へのポインタを指定します。
        @param[in] end 配列の終端要素の次へのポインタを指定します。
    */
    InArray(const T* begin, const T* end) NN_NOEXCEPT
        : m_Data(begin), m_Length(end - begin)
    {
    }

    template <typename U, typename = typename std::enable_if<std::is_convertible<util::Span<U>, util::Span<const T>>::value>::type>
    NN_IMPLICIT InArray(const util::Span<U>& span) NN_NOEXCEPT
        : InArray(span.data(), static_cast<size_t>(span.size()))
    {
    }

    NN_EXPLICIT_OPERATOR util::Span<const T>() const NN_NOEXCEPT
    {
        return {m_Data, static_cast<std::ptrdiff_t>(m_Length)};
    }

    util::Span<const T> ToSpan() const NN_NOEXCEPT
    {
        return {m_Data, static_cast<std::ptrdiff_t>(m_Length)};
    }

    /**
        @brief 配列の先頭要素へのポインタを取得します。

        @return 配列の先頭要素へのポインタを返します。
    */
    const T* GetData() const NN_NOEXCEPT
    {
        return m_Data;
    }

    /**
        @brief 配列の長さを取得します。

        @return 配列の長さを返します。
    */
    size_t GetLength() const NN_NOEXCEPT
    {
        return m_Length;
    }

    /**
        @brief 配列の要素を取得します。

        @param[in] i 取得したい要素のインデックスを指定します。
        @return 指定したインデックスの要素を返します。
    */
    const T& operator[](size_t i) const NN_NOEXCEPT
    {
        return m_Data[i];
    }

    // 内部用
    NN_IMPLICIT InArray(const detail::PointerAndSize& pas) NN_NOEXCEPT
        : m_Data(reinterpret_cast<decltype(m_Data)>(pas.pointer))
        , m_Length(pas.size / sizeof(T))
    {
    }

private:

    const T* m_Data;
    size_t m_Length;

};

template <typename T>
class OutArray
    : public detail::BufferBaseTag
{
public:

    /**
        @brief デフォルトコンストラクタ: 長さ 0 の配列として初期化します。
    */
    OutArray() NN_NOEXCEPT
        : m_Data(nullptr), m_Length(0)
    {
    }

    /**
        @brief コンストラクタ: 配列の先頭要素へのポインタと配列の長さを指定して初期化します。

        @param[in] data 配列の先頭要素へのポインタを指定します。
        @param[in] length 配列の長さを指定します。
    */
    OutArray(T* data, size_t length) NN_NOEXCEPT
        : m_Data(data), m_Length(length)
    {
    }

    /**
        @brief コンストラクタ: 配列の先頭要素と終端要素の次へのポインタを指定して初期化します。

        @param[in] begin 配列の先頭要素へのポインタを指定します。
        @param[in] end 配列の終端要素の次へのポインタを指定します。
    */
    OutArray(T* begin, T* end) NN_NOEXCEPT
        : m_Data(begin), m_Length(end - begin)
    {
    }

    NN_IMPLICIT OutArray(const util::Span<T>& span) NN_NOEXCEPT
        : OutArray(span.data(), static_cast<size_t>(span.size()))
    {
    }

    template <typename U, typename = typename std::enable_if<std::is_convertible<util::Span<T>, util::Span<U>>::value>::type>
    NN_EXPLICIT_OPERATOR util::Span<U>() const NN_NOEXCEPT
    {
        return {m_Data, static_cast<std::ptrdiff_t>(m_Length)};
    }

    util::Span<T> ToSpan() const NN_NOEXCEPT
    {
        return {m_Data, static_cast<std::ptrdiff_t>(m_Length)};
    }

    /**
        @brief 配列の先頭要素へのポインタを取得します。

        @return 配列の先頭要素へのポインタを返します。
    */
    T* GetData() const NN_NOEXCEPT
    {
        return m_Data;
    }

    /**
        @brief 配列の長さを取得します。

        @return 配列の長さを返します。
    */
    size_t GetLength() const NN_NOEXCEPT
    {
        return m_Length;
    }

    /**
        @brief 配列の要素への参照を取得します。

        @param[in] i 取得したい要素のインデックスを指定します。
        @return 指定したインデックスの要素への参照を返します。
    */
    T& operator[](size_t i) const NN_NOEXCEPT
    {
        return m_Data[i];
    }

    // 内部用
    NN_IMPLICIT OutArray(const detail::PointerAndSize& pas) NN_NOEXCEPT
        : m_Data(reinterpret_cast<decltype(m_Data)>(pas.pointer))
        , m_Length(pas.size / sizeof(T))
    {
    }

private:

    T* m_Data;
    size_t m_Length;

};

}}
