﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/os/os_MultipleWaitTypes.h>
#include <nn/sf/cmif/server/sf_CmifServerObjectInfo.h>
#include <nn/sf/hipc/sf_HipcHandleTypes.h>
#include <nn/sf/hipc/server/sf_HipcServerApiModel.h>
#include <nn/sf/sf_HipcServerSessionManagerHandler.h>

#include <utility>
#include <nn/util/util_ScopeExit.h>
#include <nn/sf/hipc/sf_HipcResult.h>
#include <nn/sf/hipc/detail/sf_HipcMessageBufferAccessor.h>

namespace nn { namespace sf { namespace hipc { namespace detail {

    struct HipcMessageHeaderInfo;
    class HipcMessageBufferAccessor;

}}}}

namespace nn { namespace sf { namespace hipc { namespace server {

class HipcServerSessionManagerBase;

class HipcServerSessionBase
    : public nn::os::MultiWaitHolderType // WaitAny 可能
{
    friend class HipcServerSessionManagerBase;
private:
    bool m_Closed;
    HipcServerSessionHandle m_ServerHandle;
public:
    Result Reply(void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT;
};

class HipcServerSessionManagerBase
{
public:

    Result ProcessRequest(HipcServerSessionBase* pSession, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT;

protected:

    Result ReceiveRequestBase(HipcServerSessionBase* p, void* messageBuffer, size_t messageBufferSize, void* pointerBuffer, size_t pointerBufferSize) NN_NOEXCEPT;
    void CloseSessionBase(HipcServerSessionBase* pSession) NN_NOEXCEPT;
    Result RegisterBase(HipcServerSessionBase* pSession, HipcServerSessionHandle serverSessionHandle) NN_NOEXCEPT;
    Result AcceptBase(HipcServerSessionBase* pSession, HipcServerPortHandle serverPortHandle) NN_NOEXCEPT;

    virtual void DestroyServerSession(HipcServerSessionBase* p) NN_NOEXCEPT = 0;

private:

    virtual Result ProcessMessage(HipcServerSessionBase* pSession, const hipc::detail::HipcMessageBufferAccessor* pInAccessor, const hipc::detail::HipcMessageHeaderInfo* pInRequestHeaderInfo, void* outMessageBuffer, size_t outMessageBufferSize) NN_NOEXCEPT = 0;
    virtual Result ProcessMessage2(HipcServerSessionBase* pSession, void* inMessageBuffer, size_t inMessageBufferSize, void* outMessageBuffer, size_t outMessageBufferSize) NN_NOEXCEPT = 0;
    virtual void RegisterServerSessionToWaitBase(HipcServerSessionBase* pSession) NN_NOEXCEPT = 0;

};

template <typename T>
class HipcServerSessionManagerT
    : public HipcServerSessionManagerBase
{
private:

    template <typename F>
    Result CreateServerSessionImpl(HipcServerSessionBase** pOut, const F& f, T&& x) NN_NOEXCEPT
    {
        auto p = this->CreateServerSession(std::forward<T>(x));
        NN_RESULT_THROW_UNLESS(p != nullptr, hipc::ResultOutOfServerSessionInfoMemory());
        auto success = false;
        NN_UTIL_SCOPE_EXIT
        {
            if (!success)
            {
                DestroyServerSession(p);
            }
        };
        NN_RESULT_DO(f(p));
        success = true;
        *pOut = p;
        NN_RESULT_SUCCESS;
    }

public:

    Result Register(HipcServerSessionHandle serverSessionHandle, T&& x) NN_NOEXCEPT
    {
        HipcServerSessionBase* dummy;
        return Register(&dummy, serverSessionHandle, std::forward<T>(x));
    }

    Result Register(HipcServerSessionBase** pOut, HipcServerSessionHandle serverSessionHandle, T&& x) NN_NOEXCEPT
    {
        auto f = [=] (HipcServerSessionBase* p) NN_NOEXCEPT
        {
            return RegisterBase(p, serverSessionHandle);
        };
        return CreateServerSessionImpl(pOut, f, std::forward<T>(x));
    }

    Result Accept(HipcServerPortHandle serverPortHandle, T&& x) NN_NOEXCEPT
    {
        HipcServerSessionBase* dummy;
        return Accept(&dummy, serverPortHandle, std::forward<T>(x));
    }

    Result Accept(HipcServerSessionBase** pOut, HipcServerPortHandle serverPortHandle, T&& x) NN_NOEXCEPT
    {
        auto f = [=] (HipcServerSessionBase* p) NN_NOEXCEPT
        {
            return AcceptBase(p, serverPortHandle);
        };
        return CreateServerSessionImpl(pOut, f, std::forward<T>(x));
    }

private:

    virtual HipcServerSessionBase* CreateServerSession(T&& x) NN_NOEXCEPT = 0;

};

class HipcServerSessionManager;
class HipcServerSessionManagerWithDomain;

class HipcServerSession
    : public HipcServerSessionBase
{
    friend class HipcServerSessionManager;
    friend class HipcServerSessionManagerWithDomain;
private:

    cmif::server::CmifServerObjectInfo m_X;
    void* m_PointerBuffer;
    size_t m_PointerBufferSize;

};

class HipcServerSessionManager
    : private HipcServerSessionManagerT<cmif::server::CmifServerObjectInfo>
    , protected HipcServerApiModelHolder
{
private:

    typedef HipcServerSessionManagerT<cmif::server::CmifServerObjectInfo> Base;

public:

    HipcServerSessionManager()
        : HipcServerApiModelHolder(GetDefaultServerApiModel())
    {
    }

    using Base::ProcessRequest;

    using Base::Register;
    using Base::Accept;

    Result ReceiveRequest(HipcServerSession* p, void* messageBuffer, size_t messageBufferSize) NN_NOEXCEPT
    {
        return Base::ReceiveRequestBase(p, messageBuffer, messageBufferSize, p->m_PointerBuffer, p->m_PointerBufferSize);
    }

    template <typename Interface>
    Result Register(HipcServerSessionHandle serverSessionHandle, SharedPointer<Interface> p) NN_NOEXCEPT
    {
        HipcServerSession dummy;
        return Register(&dummy, serverSessionHandle, std::move(p));
    }

    template <typename Interface>
    Result Register(HipcServerSession** pOut, HipcServerSessionHandle serverSessionHandle, SharedPointer<Interface> p) NN_NOEXCEPT
    {
        HipcServerSessionBase* pSession;
        NN_RESULT_DO(Register(&pSession, serverSessionHandle, cmif::server::CmifServerObjectInfo(std::move(p))));
        *pOut = static_cast<HipcServerSession*>(pSession);
        NN_RESULT_SUCCESS;
    }

    template <typename Interface>
    Result Accept(HipcServerPortHandle serverPortHandle, SharedPointer<Interface> p) NN_NOEXCEPT
    {
        HipcServerSession* dummy;
        return Accept(&dummy, serverPortHandle, std::move(p));
    }

    template <typename Interface>
    Result Accept(HipcServerSession** pOut, HipcServerPortHandle serverPortHandle, SharedPointer<Interface> p) NN_NOEXCEPT
    {
        return Accept(pOut, serverPortHandle, cmif::server::CmifServerObjectInfo(std::move(p)));
    }

protected:

    Result Accept(HipcServerPortHandle serverPortHandle, cmif::server::CmifServerObjectInfo&& x) NN_NOEXCEPT
    {
        HipcServerSession* dummy;
        return Accept(&dummy, serverPortHandle, std::move(x));
    }

    Result Accept(HipcServerSession** pOut, HipcServerPortHandle serverPortHandle, cmif::server::CmifServerObjectInfo&& x) NN_NOEXCEPT
    {
        HipcServerSessionBase* pSession;
        NN_RESULT_DO(Accept(&pSession, serverPortHandle, std::move(x)));
        *pOut = static_cast<HipcServerSession*>(pSession);
        NN_RESULT_SUCCESS;
    }

    Result ProcessInvokeMethodImpl(cmif::server::CmifServerObjectInfo&& targetObject, HipcServerSession* pSession, const hipc::detail::HipcMessageBufferAccessor* pInAccessor, const hipc::detail::HipcMessageHeaderInfo* pInRequestHeaderInfo, void* outMessageBuffer, size_t outMessageBufferSize) NN_NOEXCEPT;
    Result Process2InvokeMethodImpl(cmif::server::CmifServerObjectInfo&& targetObject, HipcServerSession* pSession, void* inMessageBuffer, size_t inMessageBufferSize, void* outMessageBuffer, size_t outMessageBufferSize) NN_NOEXCEPT;

private:

    // 上位でオーバライドが必要な関数

    virtual HipcServerSession* AllocateServerSession() NN_NOEXCEPT = 0;
    virtual void DeallocateServerSession(HipcServerSession* pSession) NN_NOEXCEPT = 0;

    virtual void* AllocatePointerTransferBuffer(size_t* pOut, HipcServerSession* pSession) NN_NOEXCEPT
    {
        NN_UNUSED(pSession);
        *pOut = 0;
        return nullptr;
    }

    virtual void DeallocatePointerTransferBuffer(HipcServerSession* pSession, void* p, size_t size) NN_NOEXCEPT
    {
        NN_UNUSED(pSession);
        NN_UNUSED(p);
        NN_UNUSED(size);
    }

    virtual void RegisterServerSessionToWait(HipcServerSession* pSession) NN_NOEXCEPT = 0;

    virtual Result ProcessManagerInvoke(HipcServerSession* pSession, const hipc::detail::HipcMessageBufferAccessor* pInAccessor, const hipc::detail::HipcMessageHeaderInfo* pInRequestHeaderInfo, void* outMessageBuffer, size_t outMessageBufferSize) NN_NOEXCEPT;
    virtual Result Process2ManagerInvoke(HipcServerSession* pSession, void* inMessageBuffer, size_t inMessageBufferSize, void* outMessageBuffer, size_t outMessageBufferSize) NN_NOEXCEPT;

protected:

    virtual HipcServerSessionManager* GetHipcServerSessionManagerInternal(uint32_t tag) NN_NOEXCEPT
    {
        NN_UNUSED(tag);
        return this;
    }

private:

    // 実装

    virtual Result ProcessMessage(HipcServerSessionBase* pSession, const hipc::detail::HipcMessageBufferAccessor* pInAccessor, const hipc::detail::HipcMessageHeaderInfo* pInRequestHeaderInfo, void* outMessageBuffer, size_t outMessageBufferSize) NN_NOEXCEPT NN_OVERRIDE final;
    virtual Result ProcessMessage2(HipcServerSessionBase* pSession, void* inMessageBuffer, size_t inMessageBufferSize, void* outMessageBuffer, size_t outMessageBufferSize) NN_NOEXCEPT final;

    virtual HipcServerSession* CreateServerSession(cmif::server::CmifServerObjectInfo&& objectInfo) NN_NOEXCEPT NN_OVERRIDE final
    {
        auto ret = AllocateServerSession();
        if (!ret)
        {
            return nullptr;
        }
        ret->m_X = std::move(objectInfo);
        ret->m_PointerBuffer = AllocatePointerTransferBuffer(&ret->m_PointerBufferSize, ret);
        return ret;
    }

    virtual void DestroyServerSession(HipcServerSessionBase* pSession) NN_NOEXCEPT NN_OVERRIDE final
    {
        auto p = static_cast<HipcServerSession*>(pSession);
        p->m_X.Reset();
        DeallocatePointerTransferBuffer(p, p->m_PointerBuffer, p->m_PointerBufferSize);
        DeallocateServerSession(p);
    }

    virtual void RegisterServerSessionToWaitBase(HipcServerSessionBase* pSession) NN_NOEXCEPT NN_OVERRIDE final
    {
        RegisterServerSessionToWait(static_cast<HipcServerSession*>(pSession));
    }

public:

    void SetManagerHandler(IHipcServerSessionManagerHandler*) NN_NOEXCEPT
    {
        // nop
        // not supported
    }

};

}}}}
