﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>

#include <nn/util/util_BitPack.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/cmif/sf_CmifMessageCommon.h>
#include <nn/sf/cmif/sf_InlineContext.h>
#include <nn/sf/hipc/detail/sf_HipcHandleRegistration.h>
#include <nn/os/os_NativeHandle.h>
#include <nn/sf/detail/sf_CommonUtil.h>
#include <nn/sf/hipc/detail/sf_HipcMessageBufferAccessor.h>

namespace nn { namespace sf { namespace hipc { namespace detail {

typedef uint32_t HipcWord;

struct HipcFormat
{
    struct Header0Tag {};
    typedef sf::detail::BitPack<uint32_t, Header0Tag>::Field<0,                      16, Bit16> Tag;
    typedef sf::detail::BitPack<uint32_t, Header0Tag>::Field<Tag::Next,               4, int> PointerCount;
    typedef sf::detail::BitPack<uint32_t, Header0Tag>::Field<PointerCount::Next,      4, int> SendCount;
    typedef sf::detail::BitPack<uint32_t, Header0Tag>::Field<SendCount::Next,         4, int> ReceiveCount;
    typedef sf::detail::BitPack<uint32_t, Header0Tag>::Field<ReceiveCount::Next,      4, int> ExchangeCount;

    struct Header1Tag;
    typedef sf::detail::BitPack<uint32_t, Header1Tag>::Field<0,                      10, int> RawCount;
    typedef sf::detail::BitPack<uint32_t, Header1Tag>::Field<RawCount::Next,          4, int> ReceiveListCount;
    typedef sf::detail::BitPack<uint32_t, Header1Tag>::Field<31,                      1, int> SpecialCount;

    struct SpecialTag;
    typedef sf::detail::BitPack<uint32_t, SpecialTag>::Field<0,                       1, int> Pid;
    typedef sf::detail::BitPack<uint32_t, SpecialTag>::Field<Pid::Next,               4, int> CopyHandleCount;
    typedef sf::detail::BitPack<uint32_t, SpecialTag>::Field<CopyHandleCount::Next,   4, int> MoveHandleCount;

    struct HeaderData
    {
        sf::detail::BitPack<uint32_t, Header0Tag> header0;
        sf::detail::BitPack<uint32_t, Header1Tag> header1;
    };

    struct Pointer0Tag;
    typedef sf::detail::BitPack<uint32_t, Pointer0Tag>::Field<0,                       4, int> PointerIndex;
    typedef sf::detail::BitPack<uint32_t, Pointer0Tag>::Field<6,                       3, uint32_t> PointerAddress36;
    typedef sf::detail::BitPack<uint32_t, Pointer0Tag>::Field<12,                      4, uint32_t> PointerAddress32;
    typedef sf::detail::BitPack<uint32_t, Pointer0Tag>::Field<PointerAddress32::Next, 16, uint32_t> PointerSize;

    struct Pointer1Tag;
    typedef sf::detail::BitPack<uint32_t, Pointer1Tag>::Field<0,                      32, uint32_t> PointerAddress0;

    struct PointerData
    {
        sf::detail::BitPack<uint32_t, Pointer0Tag> data0;
        sf::detail::BitPack<uint32_t, Pointer1Tag> data1;
    };

    struct Map0Tag;
    typedef sf::detail::BitPack<uint32_t, Map0Tag>::Field<0,               32, uint32_t> MapSizeLow;

    struct Map1Tag;
    typedef sf::detail::BitPack<uint32_t, Map1Tag>::Field<0,               32, uint32_t> MapAddress0;

    struct Map2Tag;
    typedef sf::detail::BitPack<uint32_t, Map2Tag>::Field<0,                2, uint32_t> MapTransferAttribute;
    typedef sf::detail::BitPack<uint32_t, Map2Tag>::Field<2,                3, uint32_t> MapAddress36;
    typedef sf::detail::BitPack<uint32_t, Map2Tag>::Field<24,               4, uint32_t> MapSizeHi;
    typedef sf::detail::BitPack<uint32_t, Map2Tag>::Field<MapSizeHi::Next,  4, uint32_t> MapAddress32;

    struct MapData
    {
        sf::detail::BitPack<uint32_t, Map0Tag> data0;
        sf::detail::BitPack<uint32_t, Map1Tag> data1;
        sf::detail::BitPack<uint32_t, Map2Tag> data2;
    };

    struct ReceiveList0Tag;
    typedef sf::detail::BitPack<uint32_t, ReceiveList0Tag>::Field<0,  32, uint32_t> ReceiveListAddressLow;

    struct ReceiveList1Tag;
    typedef sf::detail::BitPack<uint32_t, ReceiveList1Tag>::Field<0,   7, uint32_t> ReceiveListAddressHi;
    typedef sf::detail::BitPack<uint32_t, ReceiveList1Tag>::Field<16, 16, uint32_t> ReceiveListSize;

    struct ReceiveListData
    {
        sf::detail::BitPack<uint32_t, ReceiveList0Tag> data0;
        sf::detail::BitPack<uint32_t, ReceiveList1Tag> data1;
    };

};

struct HipcMessageDataOffsetInfo
{
    // 全て HipcWord 単位
    int pidOffset;
    int copyHandleOffset;
    int moveHandleOffset;
    int pointerOffset;
    int sendOffset;
    int receiveOffset;
    int exchangeCount;
    int rawOffset;
    int receiveListOffset;
    int allCount;
};

struct HipcMessageDataInfo
{
    bool valid;
    bool hasSpecial;
    HipcFormat::HeaderData headerData;
    sf::detail::BitPack<uint32_t, HipcFormat::SpecialTag> specialHeaderData;
    HipcMessageDataOffsetInfo dataOffsetInfo;
};

NN_SF_DETAIL_CONSTEXPR inline HipcMessageDataOffsetInfo MakeHipcMessageDataOffsetInfo(const HipcMessageHeaderInfo& headerInfo, bool hasSpecial) NN_NOEXCEPT
{
    HipcMessageDataOffsetInfo ret = {};

    if (headerInfo.rawDataByteSize % 4 > 0)
    {
        return ret;
    }
    auto rawNum = static_cast<int>(headerInfo.rawDataByteSize / sizeof(uint32_t));

    auto current = 0;

    current += 2;

    if (hasSpecial)
    {
        current += 1;
    }

    if (headerInfo.hasPid)
    {
        ret.pidOffset = current;
        current += 2;
    }

    if (headerInfo.copyHandleCount > 0)
    {
        ret.copyHandleOffset = current;
        current += headerInfo.copyHandleCount * 1;
    }

    if (headerInfo.moveHandleCount > 0)
    {
        ret.moveHandleOffset = current;
        current += headerInfo.moveHandleCount * 1;
    }

    if (headerInfo.pointerCount > 0)
    {
        ret.pointerOffset = current;
        current += headerInfo.pointerCount * 2;
    }

    if (headerInfo.sendCount > 0)
    {
        ret.sendOffset = current;
        current += headerInfo.sendCount * 3;
    }

    if (headerInfo.receiveCount > 0)
    {
        ret.receiveOffset = current;
        current += headerInfo.receiveCount * 3;
    }

    if (headerInfo.exchangeCount > 0)
    {
        ret.exchangeCount = current;
        current += headerInfo.exchangeCount * 3;
    }

    if (headerInfo.rawDataByteSize > 0)
    {
        ret.rawOffset = current;
        current += rawNum;
    }

    if (headerInfo.receiveListCount > 0)
    {
        ret.receiveListOffset = current;
        current += headerInfo.receiveListCount * 2;
    }

    ret.allCount = current;
    return ret;
}

NN_SF_DETAIL_CONSTEXPR inline HipcMessageDataInfo MakeHipcMessageDataInfo(const HipcMessageHeaderInfo& headerInfo) NN_NOEXCEPT
{
    HipcMessageDataInfo ret = {};
    ret.valid = false;

    auto receiveListCountBits = 0;
    switch (headerInfo.receiveBufferMode)
    {
        case HipcMessageReceiveBufferMode_None:
        {
            NN_SDK_ASSERT(headerInfo.receiveListCount == 0, "[SF-Internal]");
            receiveListCountBits = 0;
            break;
        }
        case HipcMessageReceiveBufferMode_MessageBuffer:
        {
            NN_SDK_ASSERT(headerInfo.receiveListCount == 0, "[SF-Internal]");
            receiveListCountBits = 1;
            break;
        }
        case HipcMessageReceiveBufferMode_Single:
        {
            NN_SDK_ASSERT(headerInfo.receiveListCount == 1, "[SF-Internal]");
            receiveListCountBits = 2;
            break;
        }
        case HipcMessageReceiveBufferMode_Multi:
        {
            NN_SDK_ASSERT(headerInfo.receiveListCount <= 13, "[SF-Internal]");
            receiveListCountBits = headerInfo.receiveListCount > 0 ? headerInfo.receiveListCount + 2 : 0;

            break;
        }
        default:
        {
            return ret;
        }
    }
    if (headerInfo.rawDataByteSize % 4 > 0)
    {
        return ret;
    }
    auto rawNum = headerInfo.rawDataByteSize / sizeof(uint32_t);

    auto hasSpecial = headerInfo.hasPid || headerInfo.copyHandleCount > 0 || headerInfo.moveHandleCount > 0;
    ret.dataOffsetInfo = MakeHipcMessageDataOffsetInfo(headerInfo, hasSpecial);

    auto current = 0;

    {
        auto& w = ret.headerData.header0;
        current += 1;
        w.Set<HipcFormat::PointerCount>(headerInfo.pointerCount);
        w.Set<HipcFormat::SendCount>(headerInfo.sendCount);
        w.Set<HipcFormat::ReceiveCount>(headerInfo.receiveCount);
        w.Set<HipcFormat::ExchangeCount>(headerInfo.exchangeCount);
    }

    {
        auto& w = ret.headerData.header1;
        current += 1;
        w.Set<HipcFormat::RawCount>(static_cast<int>(rawNum));
        w.Set<HipcFormat::ReceiveListCount>(receiveListCountBits);
        w.Set<HipcFormat::SpecialCount>(hasSpecial ? 1 : 0);
    }

    if (hasSpecial)
    {
        ret.hasSpecial = true;
        auto& w = ret.specialHeaderData;
        current += 1;
        w.Set<HipcFormat::Pid>(headerInfo.hasPid);
        w.Set<HipcFormat::CopyHandleCount>(headerInfo.copyHandleCount);
        w.Set<HipcFormat::MoveHandleCount>(headerInfo.moveHandleCount);
    }

    ret.valid = true;
    return ret;
}

inline NN_SF_DETAIL_CONSTEXPR uint32_t GetMapTransferAttribute(int bufferAttribute) NN_NOEXCEPT
{
    if (bufferAttribute & cmif::BufferAttribute_HipcMapTransferAllowsNonSecure)
    {
        return 1;
    }
    if (bufferAttribute & cmif::BufferAttribute_HipcMapTransferAllowsNonDevice)
    {
        return 3;
    }
    return 0;
}

class HipcMessageWriter
{
private:

    const HipcMessageDataInfo& m_Info;
    uint32_t* m_Buffer;

    template <typename T>
    void Write(int wordOffset, T data) NN_NOEXCEPT
    {
        std::memcpy(m_Buffer + wordOffset, &data, sizeof(data));
    }

    template <typename Tag>
    void Write(int wordOffset, sf::detail::BitPack<uint32_t, Tag> data) NN_NOEXCEPT
    {
        *(m_Buffer + wordOffset) = data.storage;
    }

public:

    explicit HipcMessageWriter(const HipcMessageDataInfo& info, void* buffer, size_t bufferSize) NN_NOEXCEPT
        : m_Info(info)
        , m_Buffer(static_cast<uint32_t*>(buffer))
    {
        NN_UNUSED(bufferSize);
        NN_SDK_ASSERT(info.dataOffsetInfo.allCount * sizeof(uint32_t) <= bufferSize, "[SF-HIPC-MessageDataIsTooLarge]");
    }

    void SetHeader(Bit16 tag) NN_NOEXCEPT
    {
        auto headerData = m_Info.headerData;
        headerData.header0.Set<HipcFormat::Tag>(tag);
        Write(0, headerData.header0);
        Write(1, headerData.header1);
        if (m_Info.hasSpecial)
        {
            Write(2, m_Info.specialHeaderData);
        }
    }

    void SetProcessId(uint64_t processId) NN_NOEXCEPT
    {
        Write(m_Info.dataOffsetInfo.pidOffset, processId);
    }

    void SetCopyHandle(int index, InternalHandleValue handle) NN_NOEXCEPT
    {
        Write(m_Info.dataOffsetInfo.copyHandleOffset + index * 1, handle);
    }

    void SetMoveHandle(int index, InternalHandleValue handle) NN_NOEXCEPT
    {
        Write(m_Info.dataOffsetInfo.moveHandleOffset + index * 1, handle);
    }

    void SetPointer(int index, InternalAddressValue addressValue, size_t size, int receiveIndex) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(size <= UINT32_MAX, "[SF-Internal]");
        auto address64 = static_cast<uint64_t>(addressValue);
        HipcFormat::PointerData data = {};
        data.data0.Set<HipcFormat::PointerIndex>(receiveIndex);
        data.data0.Set<HipcFormat::PointerAddress32>(sf::detail::ExtractBits<uint32_t>(address64, 32, HipcFormat::PointerAddress32::Width));
        data.data0.Set<HipcFormat::PointerAddress36>(sf::detail::ExtractBits<uint32_t>(address64, 36, HipcFormat::PointerAddress36::Width));
        data.data0.Set<HipcFormat::PointerSize>(static_cast<uint32_t>(size));
        data.data1.Set<HipcFormat::PointerAddress0>(static_cast<uint32_t>(address64 & 0xFFFFFFFF));
        Write(m_Info.dataOffsetInfo.pointerOffset + index * sizeof(data) / sizeof(uint32_t) + 0, data.data0);
        Write(m_Info.dataOffsetInfo.pointerOffset + index * sizeof(data) / sizeof(uint32_t) + 1, data.data1);
    }

    void SetPointer(int index, InternalAddressValue addressValue, size_t size) NN_NOEXCEPT
    {
        SetPointer(index, addressValue, size, index);
    }

private:

    void SetMapImpl(int offset, int index, InternalAddressValue addressValue, size_t size, uint32_t mapTransferAttribute) NN_NOEXCEPT
    {
        auto address64 = static_cast<uint64_t>(addressValue);
        auto size64 = static_cast<uint64_t>(size);
        auto sizeHi32 = static_cast<uint32_t>(size64 >> 32);
        auto sizeLow32 = static_cast<uint32_t>(size64 & 0xFFFFFFFF);
        HipcFormat::MapData data = {};
        data.data0.Set<HipcFormat::MapSizeLow>(sizeLow32);
        data.data1.Set<HipcFormat::MapAddress0>(static_cast<uint32_t>(address64 & 0xFFFFFFFF));
        data.data2.Set<HipcFormat::MapTransferAttribute>(mapTransferAttribute);
        data.data2.Set<HipcFormat::MapSizeHi>(sizeHi32);
        data.data2.Set<HipcFormat::MapAddress32>(sf::detail::ExtractBits<uint32_t>(address64, 32, HipcFormat::MapAddress32::Width));
        data.data2.Set<HipcFormat::MapAddress36>(sf::detail::ExtractBits<uint32_t>(address64, 36, HipcFormat::MapAddress36::Width));
        Write(offset + index * sizeof(data) / sizeof(uint32_t) + 0, data.data0);
        Write(offset + index * sizeof(data) / sizeof(uint32_t) + 1, data.data1);
        Write(offset + index * sizeof(data) / sizeof(uint32_t) + 2, data.data2);
    }

public:

    void SetSend(int index, InternalAddressValue addressValue, size_t size, uint32_t mapTransferAttribute) NN_NOEXCEPT
    {
        SetMapImpl(m_Info.dataOffsetInfo.sendOffset, index, addressValue, size, mapTransferAttribute);
    }

    void SetReceive(int index, InternalAddressValue addressValue, size_t size, uint32_t mapTransferAttribute) NN_NOEXCEPT
    {
        SetMapImpl(m_Info.dataOffsetInfo.receiveOffset, index, addressValue, size, mapTransferAttribute);
    }

    void SetExchange(int index, InternalAddressValue addressValue, size_t size, uint32_t mapTransferAttribute) NN_NOEXCEPT
    {
        SetMapImpl(m_Info.dataOffsetInfo.exchangeCount, index, addressValue, size, mapTransferAttribute);
    }

    void SetReceiveList(int index, InternalAddressValue addressValue, size_t size) NN_NOEXCEPT
    {
        NN_SDK_ASSERT(size <= UINT32_MAX, "[SF-Internal]");
        auto address64 = static_cast<uint64_t>(addressValue);
        auto addressHi32 = static_cast<uint32_t>(address64 >> 32);
        auto addressLow32 = static_cast<uint32_t>(address64 & 0xFFFFFFFF);
        HipcFormat::ReceiveListData data = {};
        data.data0.Set<HipcFormat::ReceiveListAddressLow>(addressLow32);
        data.data1.Set<HipcFormat::ReceiveListAddressHi>(addressHi32);
        data.data1.Set<HipcFormat::ReceiveListSize>(static_cast<uint32_t>(size));
        Write(m_Info.dataOffsetInfo.receiveListOffset + index * sizeof(data) / sizeof(uint32_t) + 0, data.data0);
        Write(m_Info.dataOffsetInfo.receiveListOffset + index * sizeof(data) / sizeof(uint32_t) + 1, data.data1);
    }

    uint32_t* GetRawPointer() const NN_NOEXCEPT
    {
        return m_Buffer + m_Info.dataOffsetInfo.rawOffset;
    }

};

struct HipcMessageHeaderInfo2
{
    HipcMessageHeaderInfo baseInfo;
    Bit16 tag;
    bool hasSpecial;
};

inline HipcMessageHeaderInfo2 ReadHipcMessageHeaderInfo(const uint32_t* messageBuffer) NN_NOEXCEPT
{
    HipcMessageHeaderInfo2 ret = {};

    HipcFormat::HeaderData headerData;
    std::memcpy(&headerData.header0.storage, messageBuffer + 0, sizeof(*messageBuffer));
    std::memcpy(&headerData.header1.storage, messageBuffer + 1, sizeof(*messageBuffer));

    ret.tag = headerData.header0.Get<HipcFormat::Tag>();
    ret.baseInfo.pointerCount = headerData.header0.Get<HipcFormat::PointerCount>();
    ret.baseInfo.sendCount = headerData.header0.Get<HipcFormat::SendCount>();
    ret.baseInfo.receiveCount = headerData.header0.Get<HipcFormat::ReceiveCount>();
    ret.baseInfo.exchangeCount = headerData.header0.Get<HipcFormat::ExchangeCount>();

    ret.baseInfo.rawDataByteSize = headerData.header1.Get<HipcFormat::RawCount>() * sizeof(uint32_t);
    auto receiveListCountBits = headerData.header1.Get<HipcFormat::ReceiveListCount>();
    switch (receiveListCountBits)
    {
        case 0:
        {
            ret.baseInfo.receiveBufferMode = HipcMessageReceiveBufferMode_None;
            ret.baseInfo.receiveListCount = 0;
            break;
        }
        case 1:
        {
            ret.baseInfo.receiveBufferMode = HipcMessageReceiveBufferMode_MessageBuffer;
            ret.baseInfo.receiveListCount = 0;
            break;
        }
        case 2:
        {
            ret.baseInfo.receiveBufferMode = HipcMessageReceiveBufferMode_Single;
            ret.baseInfo.receiveListCount = 1;
            break;
        }
        default:
        {
            NN_SDK_ASSERT(receiveListCountBits <= 15, "[SF-Internal]");
            ret.baseInfo.receiveBufferMode = HipcMessageReceiveBufferMode_Multi;
            ret.baseInfo.receiveListCount = receiveListCountBits - 2;
            break;
        }
    }

    auto hasSpecial = headerData.header1.Get<HipcFormat::SpecialCount>() > 0;
    if (hasSpecial)
    {
        sf::detail::BitPack<uint32_t, HipcFormat::SpecialTag> specialHeaderData;
        std::memcpy(&specialHeaderData.storage, messageBuffer + 2, sizeof(*messageBuffer));
        ret.baseInfo.hasPid = specialHeaderData.Get<HipcFormat::Pid>() > 0;
        ret.baseInfo.copyHandleCount = specialHeaderData.Get<HipcFormat::CopyHandleCount>();
        ret.baseInfo.moveHandleCount = specialHeaderData.Get<HipcFormat::MoveHandleCount>();
    }

    ret.hasSpecial = hasSpecial;
    return ret;
}

class HipcMessageReader
{
private:

    const uint32_t* m_Buffer;
    HipcMessageHeaderInfo2 m_HeaderInfo;
    HipcMessageDataOffsetInfo m_OffsetInfo;

    template <typename T>
    T Read(int wordOffset) const NN_NOEXCEPT
    {
        T ret;
        std::memcpy(&ret, m_Buffer + wordOffset, sizeof(ret));
        return ret;
    }

    template <typename Tag>
    sf::detail::BitPack<uint32_t, Tag> ReadPack(int wordOffset) const NN_NOEXCEPT
    {
        sf::detail::BitPack<uint32_t, Tag> ret;
        ret.storage = *(m_Buffer + wordOffset);
        return ret;
    }

public:

    static Bit16 ReadTag(const void* messageBuffer) NN_NOEXCEPT
    {
        sf::detail::BitPack<uint32_t, HipcFormat::Header0Tag> data;
        std::memcpy(&data, static_cast<const uint32_t*>(messageBuffer), sizeof(data));
        return data.Get<HipcFormat::Tag>();
    }

    static cmif::InlineContext ReadInlineContext(const void* messageBuffer) NN_NOEXCEPT
    {
        const auto headerInfo = detail::ReadHipcMessageHeaderInfo(static_cast<const uint32_t*>(messageBuffer));
        const auto offsetInfo = detail::MakeHipcMessageDataOffsetInfo(headerInfo.baseInfo, headerInfo.hasSpecial);
        const auto rawOffset = nn::util::align_up(offsetInfo.rawOffset, 4);
        cmif::InlineContext ret;
        std::memcpy(&ret, static_cast<const uint32_t*>(messageBuffer) + rawOffset + 3, sizeof(ret));
        return ret;
    }

    static const void* GetRawDataPointer(const void* messageBuffer) NN_NOEXCEPT
    {
        const auto headerInfo = detail::ReadHipcMessageHeaderInfo(static_cast<const uint32_t*>(messageBuffer));
        const auto offsetInfo = detail::MakeHipcMessageDataOffsetInfo(headerInfo.baseInfo, headerInfo.hasSpecial);
        const auto rawOffset = nn::util::align_up(offsetInfo.rawOffset, 4);
        return static_cast<const uint32_t*>(messageBuffer) + rawOffset;
    }

    void Initialize(const void* messageBuffer) NN_NOEXCEPT
    {
        this->m_Buffer = static_cast<const uint32_t*>(messageBuffer);
        this->m_HeaderInfo = ReadHipcMessageHeaderInfo(m_Buffer);
        this->m_OffsetInfo = MakeHipcMessageDataOffsetInfo(m_HeaderInfo.baseInfo, m_HeaderInfo.hasSpecial);
    }

    size_t GetMessageByteSize() const NN_NOEXCEPT
    {
        return m_OffsetInfo.allCount * sizeof(uint32_t);
    }

    HipcMessageHeaderInfo2 GetHipcMessageHeaderInfo() const NN_NOEXCEPT
    {
        return m_HeaderInfo;
    }

    const HipcMessageHeaderInfo2& GetHeaderInfo() const NN_NOEXCEPT
    {
        return m_HeaderInfo;
    }

    size_t GetMessageSize() const NN_NOEXCEPT
    {
        return m_OffsetInfo.allCount * sizeof(uint32_t);
    }

    Bit16 GetTag() const NN_NOEXCEPT
    {
        return m_HeaderInfo.tag;
    }

    uint64_t GetProcessId() const NN_NOEXCEPT
    {
        return Read<uint64_t>(m_OffsetInfo.pidOffset);
    }

    hipc::detail::InternalHandleValue GetCopyHandle(int index) const NN_NOEXCEPT
    {
        return Read<hipc::detail::InternalHandleValue>(m_OffsetInfo.copyHandleOffset + index * 1);
    }

    InternalHandleValue GetMoveHandle(int index) const NN_NOEXCEPT
    {
        return Read<hipc::detail::InternalHandleValue>(m_OffsetInfo.moveHandleOffset + index * 1);
    }

    AddressValueAndSize GetPointer(int index) const NN_NOEXCEPT
    {
        auto data = Read<HipcFormat::PointerData>(m_OffsetInfo.pointerOffset + index * 2);
        uint32_t size = data.data0.Get<HipcFormat::PointerSize>();
        uint64_t address36 = data.data0.Get<HipcFormat::PointerAddress36>();
        uint64_t address32 = data.data0.Get<HipcFormat::PointerAddress32>();
        uint64_t address0 = data.data1.Get<HipcFormat::PointerAddress0>();
        AddressValueAndSize ret;
        ret.pointer = static_cast<uintptr_t>(0
            | address36 << 36
            | address32 << 32
            | address0 << 0
        );
        ret.size = size;
        return ret;
    }

    int GetPointerReceiveIndex(int index) const NN_NOEXCEPT
    {
        auto data = Read<HipcFormat::PointerData>(m_OffsetInfo.pointerOffset + index * 2);
        return data.data0.Get<HipcFormat::PointerIndex>();
    }

private:

    MapTransferBufferInfo GetMapImpl(int offset) const NN_NOEXCEPT
    {
        auto data = Read<HipcFormat::MapData>(offset);
        uint64_t sizeLow = data.data0.Get<HipcFormat::MapSizeLow>();
        uint64_t address0 = data.data1.Get<HipcFormat::MapAddress0>();
        auto mapTransferAttribute = data.data2.Get<HipcFormat::MapTransferAttribute>();
        uint64_t sizeHi = data.data2.Get<HipcFormat::MapSizeHi>();
        uint64_t address32 = data.data2.Get<HipcFormat::MapAddress32>();
        uint64_t address36 = data.data2.Get<HipcFormat::MapAddress36>();
        MapTransferBufferInfo ret;
        ret.pointer = static_cast<uintptr_t>(0
            | address36 << 36
            | address32 << 32
            | address0 << 0
        );
        ret.size = static_cast<size_t>(sizeHi << 32 | sizeLow);
        ret.mapTransferAttribute = mapTransferAttribute;
        return ret;
    }

public:

    MapTransferBufferInfo GetSend(int index) const NN_NOEXCEPT
    {
        return GetMapImpl(m_OffsetInfo.sendOffset + index * 3);
    }

    MapTransferBufferInfo GetReceive(int index) const NN_NOEXCEPT
    {
        return GetMapImpl(m_OffsetInfo.receiveOffset + index * 3);
    }

    MapTransferBufferInfo GetExchange(int index) const NN_NOEXCEPT
    {
        return GetMapImpl(m_OffsetInfo.exchangeCount + index * 3);
    }

    uintptr_t GetRawPointer() const NN_NOEXCEPT
    {
        return reinterpret_cast<uintptr_t>(m_Buffer) + m_OffsetInfo.rawOffset * sizeof(uint32_t);
    }

    size_t GetRawSize() const NN_NOEXCEPT
    {
        return m_HeaderInfo.baseInfo.rawDataByteSize;
    }

    AddressValueAndSize GetReceiveList(int index) const NN_NOEXCEPT
    {
        auto data = Read<HipcFormat::ReceiveListData>(m_OffsetInfo.receiveListOffset + index * 2);
        uint64_t addressLow = data.data0.Get<HipcFormat::ReceiveListAddressLow>();
        uint64_t addressHi = data.data1.Get<HipcFormat::ReceiveListAddressHi>();
        uint32_t size = data.data1.Get<HipcFormat::ReceiveListSize>();
        AddressValueAndSize ret;
        ret.pointer = static_cast<uintptr_t>(addressHi << 32 | addressLow);
        ret.size = size;
        return ret;
    }

};

}}}}
