﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/cmif/server/sf_CmifServerMessage.h>
#include <nn/sf/cmif/server/detail/sf_CmifProcessFunctionTable.h>

#include <utility> // std::move

namespace nn { namespace sf { namespace cmif { namespace server {

/**
    @brief cmif が通信レイヤーに提供するオブジェクト情報

    @details
    コピー不可、ムーブ可能な情報として扱うことが可能。
    ProcessServerMessage 呼び出しによって、ServerMessage の処理を行う。

    なお、実装に関しては今後変更される可能性があるが、
    以下に関しては、保証される。

    - sizeof(CmifServerObjectInfo) == sizeof(void*) * 2
      - これに関しては予告ありで変更する可能性がある
    - alignof(CmifServerObjectInfo) == alignof(void*)
    - デフォルト構築可能
    - 破棄可能
    - ムーブ構築可能
    - ムーブ代入可能
    - swap 可能
    - bool 変換可能
*/
class CmifServerObjectInfo
{
private:

    SharedPointer<IServiceObject> m_P;
    const detail::CmifProcessFunctionTable* m_FunctionTable;

    // operator bool 用ヘルパ
    typedef void (CmifServerObjectInfo::*BoolType)() const;
    void FunctionForBoolType() const NN_NOEXCEPT {}

    // 内部用コピーコンストラクタ
    CmifServerObjectInfo(const CmifServerObjectInfo& other) NN_NOEXCEPT
        : m_P(other.m_P)
        , m_FunctionTable(other.m_FunctionTable)
    {
    }

    // operator= の禁止
    CmifServerObjectInfo& operator=(const CmifServerObjectInfo& rhs);

public:

    /**
        @brief デフォルトコンストラクタ
    */
    CmifServerObjectInfo() NN_NOEXCEPT
        : m_P(nullptr, false)
        , m_FunctionTable(nullptr)
    {
    }

    /**
        @brief ムーブコンストラクタ
    */
    CmifServerObjectInfo(CmifServerObjectInfo&& other) NN_NOEXCEPT
        : m_P(std::move(other.m_P))
        , m_FunctionTable(std::move(other.m_FunctionTable))
    {
    }

    /**
        @brief ムーブ代入演算子
    */
    CmifServerObjectInfo& operator=(CmifServerObjectInfo&& rhs) NN_NOEXCEPT
    {
        CmifServerObjectInfo tmp(std::move(rhs));
        tmp.swap(*this);
        return *this;
    }

    /**
        @brief swap
    */
    void swap(CmifServerObjectInfo& other) NN_NOEXCEPT
    {
        this->m_P.swap(other.m_P);
        std::swap(this->m_FunctionTable, other.m_FunctionTable);
    }

    /**
        @brief リセット

        @post static_cast<bool>(*this) == false
    */
    void Reset() NN_NOEXCEPT
    {
        this->m_P = nullptr;
        this->m_FunctionTable = nullptr;
    }

    /**
        @brief bool 変換: bool への明示的な変換演算子です。

        @return 有効なポインタであれば true を、そうでない場合には false を返します。
    */
    NN_IMPLICIT operator BoolType() const NN_NOEXCEPT
    {
        return m_P != nullptr ? &CmifServerObjectInfo::FunctionForBoolType : nullptr;
    }

    /**
        @brief 否定演算子: 否定演算子です。

        @return !static_cast<bool>(*this) を返します。
    */
    bool operator!() const NN_NOEXCEPT
    {
        return m_P == nullptr;
    }

    /**
        @brief 与えられた message を処理する。

        @pre static_cast<bool>(*this) == true
    */
    nn::Result ProcessServerMessage(CmifServerMessage* message, const PointerAndSize& rawData) NN_NOEXCEPT
    {
        return m_FunctionTable->processServerMessage(m_P.Get(), message, rawData);
    }

    /**
        @brief コピーを作成します。
    */
    CmifServerObjectInfo Clone() const NN_NOEXCEPT
    {
        return CmifServerObjectInfo(*this);
    }

    /**
        @brief 実装用のユニークな値を返すユーティリティ。

        @pre static_cast<bool>(*this) == true
    */
    std::uintptr_t GetUniqueValue() const NN_NOEXCEPT
    {
        return reinterpret_cast<std::uintptr_t>(m_P.Get());
    }

    /**
        @brief 内部用のコンストラクタ(実装外からの使用禁止)
    */
    template <typename Interface>
    explicit CmifServerObjectInfo(SharedPointer<Interface>&& p) NN_NOEXCEPT
        : m_P(std::move(p))
        , m_FunctionTable(&detail::CmifProcessFunctionTableGetter<Interface>::s_Table)
    {
    }

    /**
        @brief 型チェック付きのキャスト
    */
    template <typename Interface>
    Interface* GetServiceObject() NN_NOEXCEPT
    {
        if (!(m_FunctionTable == &detail::CmifProcessFunctionTableGetter<Interface>::s_Table))
        {
            return nullptr;
        }
        return static_cast<Interface*>(this->m_P.Get());
    }

};

static_assert(sizeof(CmifServerObjectInfo) == sizeof(void*) * 2, "[SF-Internal]");
static_assert(NN_ALIGNOF(CmifServerObjectInfo) == NN_ALIGNOF(void*), "[SF-Internal]");

inline void swap(CmifServerObjectInfo& x, CmifServerObjectInfo& y) NN_NOEXCEPT
{
    x.swap(y);
}

}}}}
