﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_SdkAssert.h>
#include <nn/sf/cmif/sf_CmifMessageCommon.h>
#include <cstdint>
#include <nn/sf/sf_NativeHandleFwd.h>

#include <nn/sf/sf_Result.h>
#include <nn/result/result_HandlingUtility.h>

namespace nn { namespace sf { namespace cmif { namespace server {

class CmifServerObjectInfo;

struct CmifServerMessageBuffersInfo
{
    PointerAndSize* inBuffers; //!< 入力バッファデータ(要素数 inBufferCount)
    PointerAndSize* outBuffers; //!< 出力バッファデータ(要素数 outBufferCount)
};

/**
    @brief サーバ用呼び出しメッセージ

    @details
    状態は以下を取る。
      - not-validated (初期状態)
      - validated
      - preparing-for-reply
      - prepared-for-reply
*/
class CmifServerMessage
{
public:

    /**
        @brief メッセージが指定されたメタ情報に合致しているかどうかをチェックし、

        @param[in] metaInfo メッセージメタ情報

        @pre not-validated 状態である
        @pre metaInfo の内容が対応する CmifClientMessage::PrepareForSyncRequest に与えられた内容と同一である

        @details
        この関数内では以下を行う必要がある。
        - メッセージが、metaInfo の入力関連情報を格納できているか → できていなかったらエラー
        - metaInfo の出力関連情報のに必要なリソースの確保

        @post validated 状態である
        @post (成功時) このメッセージが持つ入力が metaInfo の情報を満たす

        @retresult
        @endretresult
    */
    virtual nn::Result PrepareForProcess(const CmifMessageMetaInfo& metaInfo) NN_NOEXCEPT = 0;

    /**
        @brief 指定されたポインタの指すプロセス ID の領域を、妥当な値で上書きする。

        @param[out] pInOutPid 妥当性の不明なクライアントプロセス ID の格納先のポインタ

        @pre validated 状態である。
        @pre metaInfo の inProcessIdEnable が true である。

        @retresult
        @endretresult
     */
    virtual nn::Result OverwriteClientProcessId(nn::Bit64 *pInOutPid) const NN_NOEXCEPT
    {
        NN_UNUSED(pInOutPid); // 基底クラスではどんな値も妥当である、とする。
        NN_RESULT_SUCCESS;
    }

    virtual nn::Result GetBuffers(PointerAndSize* buffers) const NN_NOEXCEPT = 0;

    /**
        @brief 入力ハンドル情報を取得する。

        @details
        info.move/InNativeHandles 配列に入出力バッファの情報を書き込む。
        info.move/InNativeHandles の配列長は GetMessageInfo で取得した pOut->info.move/InNativeHandleCount 以上である必要がある。

        @param[out] info 入力

        @pre validated 状態である
        @pre [info.inNativeHandles, info.inNativeHandles + metaInfo.inNativeHandleCount) の領域に書き込むことができる
        @pre [info.moveInNativeHandles, info.moveInNativeHandles + metaInfo.moveInNativeHandles) の領域に書き込むことができる

        @post 0 <= i < metaInfo.inNativeHandleCount の i に対し、info.inNativeHandles[i] に i 番目のコピーハンドル情報が書き込まれている
        @post 0 <= j < metaInfo.moveInNativeHandles の j に対し、info.moveInNativeHandles[i] に j 番目のムーブハンドル情報が書き込まれている

        @retresult
        @endretresult
    */
    virtual nn::Result GetInNativeHandles(nn::sf::NativeHandle handles[]) const NN_NOEXCEPT
    {
        NN_UNUSED(handles);
        NN_RESULT_THROW(nn::sf::ResultNotSupported());
    }

    /**
        @brief 入力オブジェクトを取得する。

        @param[out] inObjects 入力オブジェクトを書き込む先

        @pre validated 状態である
        @pre [inObjects, inObjects + metaInfo.inObjectCount) に書き込むことができる

        @post 0 <= i < metaInfo.inObjectCount の i に対し、inObjects[i] に i 番目の入力オブジェクト情報が書き込まれている

        @retresult
        @endretresult
    */
    virtual nn::Result GetInObjects(CmifServerObjectInfo* inObjects) const NN_NOEXCEPT
    {
        NN_UNUSED(inObjects);
        NN_RESULT_THROW(nn::sf::ResultNotSupported());
    }

    /**
        @brief 返答に向けた準備をする。

        @param[in] metaInfo メッセージメタ情報

        @pre validated 状態である
        @pre metaInfo の入力に関する内容が PrepareForProcess に与えられた内容と同一である

        @post preparing-for-reply 状態である
    */
    virtual void BeginPreparingForReply(PointerAndSize* pOutRawData) NN_NOEXCEPT = 0;

    virtual void SetBuffers(PointerAndSize buffers[]) NN_NOEXCEPT
    {
        NN_UNUSED(buffers);
    }

    /**
        @brief 出力オブジェクトを設定する。

        @param[in] outObjects 出力オブジェクト配列
        @param[in] outObjectCount 出力オブジェクト数

        @pre preparing-for-reply 状態である
        @pre [outObjects, outObjects + metaInfo.outObjectCount) に読むことができる
        @pre 0 <= i < metaInfo.outObjectCount の i に対し、outObjects[i] に i 番目の入力オブジェクト情報が書き込まれている

        @post サーバの返信メッセージに適切に出力オブジェクト情報が書き込まれている
        @post 以降の呼び出しで、返信メッセージで渡された情報を使ってオブジェクトを参照できる
    */
    virtual void SetOutObjects(CmifServerObjectInfo* outObjects) NN_NOEXCEPT
    {
        NN_UNUSED(outObjects);
    }

    /**
        @brief 出力ハンドルを設定する。

        @param[in] outNativeHandles 出力オブジェクト配列
        @param[in] outNativeHandleCount 出力オブジェクト数

        @pre preparing-for-reply 状態である
        @pre [outNativeHandles, outNativeHandles + metaInfo.outNativeHandleCount) に読むことができる
        @pre 0 <= i < metaInfo.outNativeHandleCount の i に対し、outNativeHandles[i] に i 番目の出力ハンドルが書き込まれている

        @post サーバの返信メッセージに適切に出力ハンドル情報が書き込まれている
    */
    virtual void SetOutNativeHandles(nn::sf::NativeHandle outNativeHandles[]) NN_NOEXCEPT
    {
        NN_UNUSED(outNativeHandles);
    }

    virtual void BeginPreparingForErrorReply(PointerAndSize* pOutRawData, size_t outRawSize) NN_NOEXCEPT = 0;

    /**
        @brief 返答に向けた準備をする。

        @pre preparing-for-reply 状態である
        @pre 必要な情報が書き込まれている

        @post prepared-for-reply 状態である
    */
    virtual void EndPreparingForReply() NN_NOEXCEPT
    {
    }

protected:

    ~CmifServerMessage() NN_NOEXCEPT
    {
    }

};

}}}}
