﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/sf/sf_Types.h>
#include <nn/sf/cmif/sf_CmifMessageCommon.h>
#include <cstdint>
#include <utility>
#include <nn/sf/sf_NativeHandleFwd.h>
#include <nn/sf/cmif/detail/sf_CmifMethodInfos.h>
#include <nn/sf/detail/sf_CmifProxyInfoAccessor.h>
#include <nn/sf/cmif/sf_CmifResult.h>
#include <nn/sf/sf_NativeHandle.h>
#include <nn/sf/cmif/client/sf_CmifProxyFactory.h>

namespace nn { namespace sf { namespace cmif { namespace client {

/**
    実装用ベースクラス
*/
class CmifBaseObject
{
};

/**
    CmifClientMessage::PrepareForSyncRequest に渡す情報型
*/
struct CmifClientMessageInfo
{
    const PointerAndSize* buffers; //!< バッファデータ(要素数 bufferCount)
    const CmifBaseObject** inObjects; // 入力オブジェクト配列(要素数 inObjectCount)
    NativeHandle* inNativeHandles;
};

/**
    CmifClientMessage::PrepareForSyncRequest の返り値型
*/
struct CmifClientMessageInInfo
{
    std::uintptr_t inRawDataPointer; //!< 入力 raw data の先頭へのポインタ
};

/**
    CmifClientMessage::GetCmifClientMessageOutInfo の返り値型
*/
struct CmifClientMessageOutInfo
{
    std::uintptr_t outRawDataPointer; //!< 出力 raw data の先頭へのポインタ
};

/**
    @brief クライアント用呼び出しメッセージ

    @details
    PrepareForSyncRequest によって初期化されたのちに他のメンバが呼ばれる。

    状態は以下を取る。
      - not-prepared (初期状態)
      - prepared
      - completed
*/
class CmifClientMessage
{
public:

    /**
        @brief info に従って同期通信用にメッセージを用意する

        @details
        用意されたメッセージのデータを pOut に返す。
        ユーザは、これによって返された情報を raw に適切に埋めたのちに、次の SendSyncRequest を呼ぶ。

        pOut に格納される raw data のアラインは in, out ともに 16B とする。(仮想関数実装への要求)

        @param[out] pOut 出力
        @param[in] info 入力
        @param[in] metaInfo メッセージメタ情報

        @pre not-prepared 状態である
        @pre metaInfo.inRawDataSize % 4 == 0
        @pre metaInfo.outRawDataSize % 4 == 0
        @pre 0 <= metaInfo.inObjectCount <= 8 (暫定的に 8 個までとする)
        @pre 0 <= metaInfo.outObjectCount <= 8 (暫定的に 8 個までとする)
        @pre [info.inObjects, info.inObjects + metaInfo.inObjectCount) の領域を読むことができ、各オブジェクトが主体オブジェクトと同一のドメインに含まれる
        @pre [info.moveInNativeHandles, info.moveInNativeHandles + metaInfo.moveIsnNativeHandleCount) の領域を読むことができ、各要素が有効なハンドルである
        @pre [info.inNativeHandles, info.inNativeHandles + metaInfo.inNativeHandleCount) の領域を読むことができ、各要素が有効なハンドルである

        @post prepared 状態である
        @post info, metaInfo で与えられた情報のとおりに、必要なメモリ等が確保され、適切にメッセージバッファに書き込まれている

        @retresult
        @endretresult
    */
    virtual nn::Result PrepareForSyncRequest(CmifClientMessageInfo&& info, const CmifMessageMetaInfo& metaInfo) NN_NOEXCEPT = 0;

    /**
        @brief 受信前の情報を取得する

        @return CmifClientMessageInInfo

        @pre prepared 状態である

        @post retval->inRawDataPointer == 出力 raw data へのポインタ
        @post retval->inRawDataPointer % 16 == 0
    */
    virtual CmifClientMessageInInfo GetCmifClientMessageInInfo() const NN_NOEXCEPT = 0;

    /**
        @brief 設定された情報で同期通信する。

        @pre prepared 状態である
        @pre ユーザによって入力 raw data に適切にデータが書き込まれている

        @post completed 状態である
        @post 出力 raw data や設定したバッファに適切にデータが書き込まれている

        @retresult
        @endretresult
    */
    virtual nn::Result SendSyncRequest() NN_NOEXCEPT = 0;

    /**
        @brief 受信後の情報を取得する

        @return CmifClientMessageOutInfo

        @pre completed 状態である

        @post retval->outRawDataPointer == 出力 raw data へのポインタ
        @post retval->outRawDataPointer % 16 == 0
    */
    virtual CmifClientMessageOutInfo GetCmifClientMessageOutInfo() const NN_NOEXCEPT = 0;

    /**
        @brief 出力されたオブジェクトを与えられたインスタンスにアタッチする

        @param [in] outObjects アタッチする対象のインスタンスへのポインタの配列

        @pre completed 状態である
        @pre outObjects
        @pre [outObjects, outObjects + metaInfo.outObjectCount) の領域を読むことができる
        @pre 0 <= i < metaInfo.outObjectCount の i に対し、*(outObjects[i]) は this がアタッチできるオブジェクトであり、まだアタッチされていない

        @post 0 <= i < metaInfo.outObjectCount の i に対し、*(outObjects[i]) には、i 番目の出力オブジェクトがアタッチされている
    */
    virtual void AttachOutObjects(CmifBaseObject* outObjects[]) const NN_NOEXCEPT
    {
        // コンパイルエラー防止のためのデフォルト実装
        // TODO: 上位クラスで実装されたら純粋仮想に変更する。
        NN_UNUSED(outObjects);
    }

    /**
        @brief 出力ハンドルを取得する

        @param [out] outNativeHandles 出力ハンドルを書き出す配列

        @pre completed 状態である
        @pre [outNativeHandles, outNativeHandles + metaInfo.outNativeHandleCount) の領域に書き込むことができる

        @post 0 <= i < metaInfo.outNativeHandleCount の i に対し、outNativeHandles[i] には、i 番目の出力ハンドルが設定されている
    */
    virtual void GetOutNativeHandles(NativeHandle outNativeHandles[]) const NN_NOEXCEPT
    {
        // コンパイルエラー防止のためのデフォルト実装
        // TODO: 上位クラスで実装されたら純粋仮想に変更する。
        NN_UNUSED(outNativeHandles);
    }

protected:

    /**
        @brief メッセージを破棄する。

        @details
        必要に応じて、確保されているリソースを解放する。
        PrepareForSyncRequest や SendSyncRequest で失敗を返した場合にも呼ばれる。
    */
    ~CmifClientMessage() NN_NOEXCEPT
    {
    }

};

struct ClientArgumentTypeBase
{
    template <typename ArgumentInfo>
    sf::detail::MonoState PrepareBuffer(cmif::PointerAndSize pPas[]) NN_NOEXCEPT
    {
        NN_UNUSED(pPas);
        return {};
    }

    template <typename ArgumentInfo>
    sf::detail::MonoState PrepareInObject(const CmifBaseObject* pInObjects[]) NN_NOEXCEPT
    {
        NN_UNUSED(pInObjects);
        return {};
    }

    template <typename ArgumentInfo>
    sf::detail::MonoState PrepareInHandle(sf::NativeHandle inHandles[]) NN_NOEXCEPT
    {
        NN_UNUSED(inHandles);
        return {};
    }

    template <typename ArgumentInfo>
    sf::detail::MonoState PrepareInRaw(char* inRawPointer) NN_NOEXCEPT
    {
        NN_UNUSED(inRawPointer);
        return {};
    }

    template <typename ArgumentInfo>
    sf::detail::MonoState GetOutRaw(const char* outRawPointer) NN_NOEXCEPT
    {
        NN_UNUSED(outRawPointer);
        return {};
    }

    template <typename ArgumentInfo>
    sf::detail::MonoState GetOutHandle(sf::NativeHandle outHandles[]) NN_NOEXCEPT
    {
        NN_UNUSED(outHandles);
        return {};
    }

    template <typename ArgumentInfo>
    sf::detail::MonoState GetOutObject(SharedPointer<IServiceObject> pOutObjects[]) NN_NOEXCEPT
    {
        NN_UNUSED(pOutObjects);
        return {};
    }
};

template <typename ArgumentInfo, typename Enabled = void>
struct GetClientArgumentType;

struct RawClientArgumentInternalValue
{
    void* p;
};

template <size_t Size, size_t Alignment, typename Enabled = void>
struct InRawClientArgumentType;

template <size_t Alignment>
struct InRawClientArgumentType<0, Alignment>
    : public ClientArgumentTypeBase
{
    template <typename T>
    NN_IMPLICIT InRawClientArgumentType(const T&) NN_NOEXCEPT
    {
    }

    NN_IMPLICIT InRawClientArgumentType(RawClientArgumentInternalValue) NN_NOEXCEPT
    {
    }

    template <typename ArgumentInfo>
    sf::detail::MonoState PrepareInRaw(char*) NN_NOEXCEPT
    {
        return {};
    }
};

template <size_t Size, size_t Alignment>
struct InRawClientArgumentType<Size, Alignment, typename std::enable_if<(Size > 8)>::type>
    : public ClientArgumentTypeBase
{
    using Storage = typename std::aligned_storage<Size, Alignment>::type;
    const Storage* p;

    template <typename T>
    NN_IMPLICIT InRawClientArgumentType(const T& x) NN_NOEXCEPT
        : p(reinterpret_cast<const Storage*>(&x))
    {
        static_assert(sizeof(T) == Size, "[SF-Unexpected] size mismatch");
        static_assert(NN_ALIGNOF(T) == Alignment, "[SF-Unexpected] alignment mismatch");
    }

    NN_IMPLICIT InRawClientArgumentType(RawClientArgumentInternalValue v) NN_NOEXCEPT
        : p(reinterpret_cast<const Storage*>(v.p))
    {
    };

    template <typename ArgumentInfo>
    sf::detail::MonoState PrepareInRaw(char* inRawPointer) NN_NOEXCEPT
    {
        static_assert(Size == ArgumentInfo::Size::value, "[SF-Unexpected] size mismatch");
        static_assert(Alignment == ArgumentInfo::Alignment::value, "[SF-Unexpected] alignment mismatch");
        std::memcpy(inRawPointer + ArgumentInfo::RawOffset::value, p, Size);
        return {};
    }
};

template <size_t Size, size_t Alignment>
struct InRawClientArgumentType<Size, Alignment, typename std::enable_if<(Size <= 8)>::type>
    : public ClientArgumentTypeBase
{
    using Storage = typename std::aligned_storage<Size, Alignment>::type;
    Storage m;

    template <typename T>
    NN_IMPLICIT InRawClientArgumentType(T x) NN_NOEXCEPT
    {
        static_assert(sizeof(T) == Size, "[SF-Unexpected] size mismatch");
        static_assert(NN_ALIGNOF(T) == Alignment, "[SF-Unexpected] alignment mismatch");
        std::memcpy(&m, &x, Size);
    }

    NN_IMPLICIT InRawClientArgumentType(RawClientArgumentInternalValue v) NN_NOEXCEPT
    {
        std::memcpy(&m, v.p, Size);
    };

    template <typename ArgumentInfo>
    sf::detail::MonoState PrepareInRaw(char* inRawPointer) NN_NOEXCEPT
    {
        static_assert(Size == ArgumentInfo::Size::value, "[SF-Unexpected] size mismatch");
        static_assert(Alignment == ArgumentInfo::Alignment::value, "[SF-Unexpected] alignment mismatch");
        std::memcpy(inRawPointer + ArgumentInfo::RawOffset::value, &m, Size);
        return {};
    }
};

template <typename ArgumentInfo>
struct GetClientArgumentType<ArgumentInfo, typename std::enable_if<ArgumentInfo::Kind::value == sf::cmif::detail::ArgumentKind::InRaw>::type>
{
    using type = InRawClientArgumentType<ArgumentInfo::Size::value, ArgumentInfo::Alignment::value>;
};

template <size_t Size, size_t Alignment>
struct OutRawClientArgumentType
    : public ClientArgumentTypeBase
{
    using Storage = typename std::aligned_storage<Size, Alignment>::type;
    Storage* p;

    template <typename T>
    NN_IMPLICIT OutRawClientArgumentType(Out<T> pOut) NN_NOEXCEPT
        : p(reinterpret_cast<Storage*>(pOut.GetPointer()))
    {
        static_assert(sizeof(T) == Size, "[SF-Unexpected] size mismatch");
        static_assert(NN_ALIGNOF(T) == Alignment, "[SF-Unexpected] alignment mismatch");
    }

    NN_IMPLICIT OutRawClientArgumentType(RawClientArgumentInternalValue v) NN_NOEXCEPT
        : p(reinterpret_cast<Storage*>(v.p))
    {
    };

    template <typename ArgumentInfo>
    sf::detail::MonoState GetOutRaw(const char* outRawPointer) NN_NOEXCEPT
    {
        static_assert(Size == ArgumentInfo::Size::value, "[SF-Unexpected] size mismatch");
        static_assert(Alignment == ArgumentInfo::Alignment::value, "[SF-Unexpected] alignment mismatch");
        std::memcpy(p, outRawPointer + ArgumentInfo::RawOffset::value, Size);
        return {};
    }
};

template <size_t Alignment>
struct OutRawClientArgumentType<0, Alignment>
    : public ClientArgumentTypeBase
{
    template <typename T>
    NN_IMPLICIT OutRawClientArgumentType(Out<T>) NN_NOEXCEPT
    {
    }

    NN_IMPLICIT OutRawClientArgumentType(RawClientArgumentInternalValue) NN_NOEXCEPT
    {
    }

    template <typename ArgumentInfo>
    sf::detail::MonoState GetOutRaw(const char*) NN_NOEXCEPT
    {
        return {};
    }
};

template <typename ArgumentInfo>
struct GetClientArgumentType<ArgumentInfo, typename std::enable_if<ArgumentInfo::Kind::value == sf::cmif::detail::ArgumentKind::OutRaw>::type>
{
    using type = OutRawClientArgumentType<ArgumentInfo::Size::value, ArgumentInfo::Alignment::value>;
};

struct BufferClientArgumentType
    : public ClientArgumentTypeBase
{
    cmif::PointerAndSize pointerAndSize;

    NN_IMPLICIT BufferClientArgumentType(const InBuffer& buffer) NN_NOEXCEPT
    {
        pointerAndSize.pointer = reinterpret_cast<decltype(pointerAndSize.pointer)>(buffer.GetPointerUnsafe());
        pointerAndSize.size = buffer.GetSize();
    }

    NN_IMPLICIT BufferClientArgumentType(const OutBuffer& buffer) NN_NOEXCEPT
    {
        pointerAndSize.pointer = reinterpret_cast<decltype(pointerAndSize.pointer)>(buffer.GetPointerUnsafe());
        pointerAndSize.size = buffer.GetSize();
    }

    template <typename T>
    NN_IMPLICIT BufferClientArgumentType(const T& x) NN_NOEXCEPT
    {
        pointerAndSize.pointer = reinterpret_cast<decltype(pointerAndSize.pointer)>(&x);
        pointerAndSize.size = sizeof(typename std::decay<T>::type);
    }

    template <typename T>
    NN_IMPLICIT BufferClientArgumentType(const InArray<T>& x) NN_NOEXCEPT
    {
        pointerAndSize.pointer = reinterpret_cast<decltype(pointerAndSize.pointer)>(x.GetData());
        pointerAndSize.size = sizeof(T) * x.GetLength();
    }

    template <typename T>
    NN_IMPLICIT BufferClientArgumentType(const OutArray<T>& x) NN_NOEXCEPT
    {
        pointerAndSize.pointer = reinterpret_cast<decltype(pointerAndSize.pointer)>(x.GetData());
        pointerAndSize.size = sizeof(T) * x.GetLength();
    }

    template <typename T>
    NN_IMPLICIT BufferClientArgumentType(Out<T> x) NN_NOEXCEPT
    {
        pointerAndSize.pointer = reinterpret_cast<decltype(pointerAndSize.pointer)>(x.GetPointer());
        pointerAndSize.size = sizeof(typename std::decay<T>::type);
    }

    template <typename ArgumentInfo>
    sf::detail::MonoState PrepareBuffer(cmif::PointerAndSize pPas[]) NN_NOEXCEPT
    {
        pPas[ArgumentInfo::Index::value] = pointerAndSize;
        return {};
    }
};

template <typename ArgumentInfo>
struct GetClientArgumentType<ArgumentInfo, typename std::enable_if<ArgumentInfo::Kind::value == sf::cmif::detail::ArgumentKind::Buffer>::type>
{
    using type = BufferClientArgumentType;
};

struct InObjectClientArgumentType
    : public ClientArgumentTypeBase
{
    IServiceObject* pInObject;

    template <typename I>
    NN_IMPLICIT InObjectClientArgumentType(sf::SharedPointer<I> p) NN_NOEXCEPT
        : pInObject(p.Get())
    {
    }

    template <typename ArgumentInfo>
    sf::detail::MonoState PrepareInObject(const CmifBaseObject* pInObjects[]) NN_NOEXCEPT
    {
        pInObjects[ArgumentInfo::Index::value] = sf::detail::CmifProxyInfoAccessor::GetCmifBaseObject(pInObject);
        return {};
    }
};

template <typename ArgumentInfo>
struct GetClientArgumentType<ArgumentInfo, typename std::enable_if<ArgumentInfo::Kind::value == sf::cmif::detail::ArgumentKind::InObject>::type>
{
    using type = InObjectClientArgumentType;
};

struct InHandleClientArgumentType
    : public ClientArgumentTypeBase
{
    sf::NativeHandle* pHandle;

    NN_IMPLICIT InHandleClientArgumentType(sf::NativeHandle&& handle_) NN_NOEXCEPT
        : pHandle(&handle_)
    {
    }

    template <typename ArgumentInfo>
    sf::detail::MonoState PrepareInHandle(sf::NativeHandle inHandles[]) NN_NOEXCEPT
    {
        inHandles[ArgumentInfo::Index::value] = std::move(*pHandle);
        return {};
    }
};

template <typename ArgumentInfo>
struct GetClientArgumentType<ArgumentInfo, typename std::enable_if<ArgumentInfo::Kind::value == sf::cmif::detail::ArgumentKind::InHandle>::type>
{
    using type = InHandleClientArgumentType;
};

struct OutHandleClientArgumentType
    : public ClientArgumentTypeBase
{
    Out<NativeHandle> pHandle;

    NN_IMPLICIT OutHandleClientArgumentType(Out<NativeHandle> pHandle_) NN_NOEXCEPT
        : pHandle(pHandle_)
    {
    }

    template <typename ArgumentInfo>
    sf::detail::MonoState GetOutHandle(sf::NativeHandle outHandles[]) NN_NOEXCEPT
    {
        *pHandle = std::move(outHandles[ArgumentInfo::Index::value]);
        return {};
    }
};

template <typename ArgumentInfo>
struct GetClientArgumentType<ArgumentInfo, typename std::enable_if<ArgumentInfo::Kind::value == sf::cmif::detail::ArgumentKind::OutHandle>::type>
{
    using type = OutHandleClientArgumentType;
};

struct OutObjectClientArgumentType
    : public ClientArgumentTypeBase
{
    sf::Out<SharedPointer<IServiceObject>> pOut;

    template <typename I>
    NN_IMPLICIT OutObjectClientArgumentType(Out<SharedPointer<I>> pOut_) NN_NOEXCEPT
        : pOut(pOut_.template DownCast<IServiceObject>())
    {
    }

    template <typename ArgumentInfo>
    sf::detail::MonoState GetOutObject(SharedPointer<IServiceObject> pOutObjects[]) NN_NOEXCEPT
    {
        pOut.Set(std::move(pOutObjects[ArgumentInfo::Index::value]));
        return {};
    }
};

template <typename ArgumentInfo>
struct GetClientArgumentType<ArgumentInfo, typename std::enable_if<ArgumentInfo::Kind::value == sf::cmif::detail::ArgumentKind::OutObject>::type>
{
    using type = OutObjectClientArgumentType;
};

/*
    ClientCoreProcessor: ProxyKind を使用し、CoreMethodInfo に応じた CMIF 呼び出しを行うためのクラステンプレート

    以下の関数を持つ。(ただし ClientArgumentTypes... = GetClientArgumentType<ArgumentInfos>::type...)

    - NN_NOINLINE static Result Process(ProxyBaseObject* p, ClientArgumentTypes... args, MethodId methodId)
    - NN_NOINLINE static Result ProcessWithOutObjects(ProxyBaseObject* p, ClientArgumentTypes... args, MethodId methodId, SharedPointer<IServiceObject> pOutObjects[])
    - NN_FORCEINLINE static Result ProcessImpl(const ProcessorArgument& pa, typename cmif::client::GetClientArgumentType<ArgumentInfos>::type... args, cmif::MethodId methodId, SharedPointer<IServiceObject> pOutObjects[], size_t inRawPadding, size_t outRawPadding)

    Process() は出力オブジェクトがないときに呼ばれ、ProcessWithOutObjects() は出力オブジェクトのあるときに呼ばれる。
    これらの関数は一般に NN_NOINLINE をつけることで、同じ形の引数を持つ関数群が、一つのインスタンスを共有できるようになっている。

    ClientArgumentTypes の各型は、実際の引数型からの暗黙変換ができるようになっており、
    ClientProcessor::Process() 内からの ClientCoreProcessor::Process() 呼び出しの際には、
    この暗黙変換によって実際の引数渡しが行われる。

    ProcessImpl() は、プロキシ構造を重ねる際に、上の ClientCoreProcessor から呼ばれるもので、
    以下の追加引数を持つ。

    - processorArgument: ProxyBaseObject* から引数として渡すことができる情報を取り出したもの
    - inRawPadding: CMIF 入力ヘッダのずれ
    - outRawPadding: CMIF 出力ヘッダのずれ

    rawPadding 変数は上位のヘッダなどを CMIF ヘッダより前に配置する際に使用する。
*/
template <typename ProxyKind, typename CoreMethodInfo, typename ArgumentInfos = typename CoreMethodInfo::ArgumentInfos>
struct ClientCoreProcessor;

/*
    ClientCoreProcessor の汎用実装

    ProxyKind::CmifClientMessage を使用した汎用実装で、一般的にはこれが使用される。
    マクロによる旧コードとほぼ同等のコードが生成される。

    HIPC などでは本汎用実装は使われず、ProxyKind::CmifClientMessage を使用しない特殊実装が使われる。

    この場合の特殊実装においては、さらに追加で以下の関数が必要となる。

    - NN_FORCEINLINE static Result ProcessImpl(const ProcessorArgument& pa, typename cmif::client::GetClientArgumentType<ArgumentInfos_>::type... args, cmif::MethodId methodId, SharedPointer<IServiceObject> pOutObjects[], size_t inRawPadding, size_t outRawPadding)
*/
template <typename ProxyKind, typename CoreMethodInfo, typename... ArgumentInfos_>
struct ClientCoreProcessor<ProxyKind, CoreMethodInfo, std::tuple<ArgumentInfos_...>>
{
private:

    using ProxyBaseObject = typename ProxyKind::ProxyBaseObject;
    using ClientMessage = typename ProxyKind::CmifClientMessage;

    template <typename... Args>
    static void Discard(Args...) NN_NOEXCEPT
    {
    }

    template <typename ArgumentInfo>
    struct GetAttribute
    {
        using type = typename ArgumentInfo::Attribute;
    };

    template <typename Attribute>
    class AttributeSetter
    {
    private:

        Attribute* m_Attributes;

    public:

        NN_SF_DETAIL_CONSTEXPR AttributeSetter(Attribute attributes[]) NN_NOEXCEPT
            : m_Attributes(attributes)
        {
        }

        template <typename ArgumentInfo>
        NN_SF_DETAIL_CONSTEXPR void operator()(sf::detail::TypeInvokeTag<ArgumentInfo>) NN_NOEXCEPT
        {
            m_Attributes[ArgumentInfo::Index::value] = ArgumentInfo::Attribute::value;
        }
    };

    NN_SF_DETAIL_CONSTEXPR static cmif::CmifMessageMetaInfo MakeCmifMessageMetaInfo() NN_NOEXCEPT
    {
        cmif::CmifMessageMetaInfo ret = {};
        ret.inProcessIdEnable = CoreMethodInfo::InProcessIdEnable::value;
        ret.inRawDataSize = sizeof(CmifInHeader) + CoreMethodInfo::InRawSize::value;
        ret.outRawDataSize = sizeof(CmifOutHeader) + CoreMethodInfo::OutRawSize::value;
        ret.bufferCount = CoreMethodInfo::BufferCount::value;
        ret.inObjectCount = CoreMethodInfo::InObjectCount::value;
        ret.outObjectCount = CoreMethodInfo::OutObjectCount::value;
        ret.inNativeHandleCount = CoreMethodInfo::InHandleCount::value;
        ret.outNativeHandleCount = CoreMethodInfo::OutHandleCount::value;
        sf::detail::TypeForeachInvoke<typename CoreMethodInfo::Buffers>(AttributeSetter<int>(ret.bufferAttributes));
        sf::detail::TypeForeachInvoke<typename CoreMethodInfo::InHandles>(AttributeSetter<int>(ret.inNativeHandleAttributes));
        sf::detail::TypeForeachInvoke<typename CoreMethodInfo::OutHandles>(AttributeSetter<int>(ret.outNativeHandleAttributes));
        return ret;
    }

    NN_FORCEINLINE static Result ProcessImpl(ProxyBaseObject* p, typename GetClientArgumentType<ArgumentInfos_>::type... args, MethodId methodId, SharedPointer<IServiceObject> pOutObjects[]) NN_NOEXCEPT
    {
        ClientMessage message(p);
        auto pMessage = &message;

        // prepare
        std::array<NativeHandle, CoreMethodInfo::InHandleCount::value> inHandles;
        {
            std::array<cmif::PointerAndSize, CoreMethodInfo::BufferCount::value> buffers;
            std::array<const CmifBaseObject*, CoreMethodInfo::InObjectCount::value> inObjects;
            CmifClientMessageInfo info = {};

            // in buffers
            NN_UNUSED(buffers);
            if (NN_STATIC_CONDITION(CoreMethodInfo::BufferCount::value > 0))
            {
                Discard(args.template PrepareBuffer<ArgumentInfos_>(buffers.data())...);
                info.buffers = buffers.data();
            }

            // in objects
            NN_UNUSED(inObjects);
            if (NN_STATIC_CONDITION(CoreMethodInfo::InObjectCount::value > 0))
            {
                Discard(args.template PrepareInObject<ArgumentInfos_>(inObjects.data())...);
                info.inObjects = inObjects.data();
            }

            // in handles
            NN_UNUSED(inHandles);
            if (NN_STATIC_CONDITION(CoreMethodInfo::InHandleCount::value > 0))
            {
                Discard(args.template PrepareInHandle<ArgumentInfos_>(inHandles.data())...);
                info.inNativeHandles = inHandles.data();
            }

            /// prepare
            {
                NN_SF_DETAIL_CONSTEXPR auto specificMetaInfo = ClientMessage::MakeSpecificMethodMetaInfo(MakeCmifMessageMetaInfo());
                NN_SF_RESULT_DO(pMessage->PrepareForSyncRequest(std::move(info), specificMetaInfo));
            }
            auto inInfo = pMessage->GetCmifClientMessageInInfo();
            auto inHeader = cmif::CmifInHeader::Create(methodId);
            std::memcpy(reinterpret_cast<char*>(inInfo.inRawDataPointer), &inHeader, sizeof(inHeader));

            // in raw
            if (NN_STATIC_CONDITION(CoreMethodInfo::InRawSize::value > 0))
            {
                auto pInRaw = reinterpret_cast<char*>(inInfo.inRawDataPointer) + sizeof(inHeader);
                std::memset(pInRaw, 0, CoreMethodInfo::InRawSize::value);
                Discard(args.template PrepareInRaw<ArgumentInfos_>(pInRaw)...);
            }
        }

        NN_SF_RESULT_DO(pMessage->SendSyncRequest());

        // post process
        {
            auto outInfo = pMessage->GetCmifClientMessageOutInfo();
            {
                cmif::CmifOutHeader outHeader;
                std::memcpy(&outHeader, reinterpret_cast<void*>(outInfo.outRawDataPointer), sizeof(outHeader));
                NN_SF_RESULT_THROW_UNLESS(outHeader.Check(), cmif::ResultInvalidCmifOutHeader());
                NN_SF_RESULT_DO(outHeader.result);
            }

            // out raw
            if (NN_STATIC_CONDITION(CoreMethodInfo::OutRawSize::value > 0))
            {
                auto pOutRaw = reinterpret_cast<const char*>(outInfo.outRawDataPointer) + sizeof(cmif::CmifOutHeader);
                NN_UNUSED(pOutRaw);
                Discard(args.template GetOutRaw<ArgumentInfos_>(pOutRaw)...);
            }

            // out handles
            if (NN_STATIC_CONDITION(CoreMethodInfo::OutHandleCount::value > 0))
            {
                std::array<sf::NativeHandle, CoreMethodInfo::OutHandleCount::value> outHandles;
                pMessage->GetOutNativeHandles(outHandles.data());
                Discard(args.template GetOutHandle<ArgumentInfos_>(outHandles.data())...);
            }

            // out objects
            if (NN_STATIC_CONDITION(CoreMethodInfo::OutObjectCount::value > 0))
            {
                std::array<sf::cmif::client::CmifBaseObject*, CoreMethodInfo::OutObjectCount::value> pBaseObjects;
                for (auto i = 0u; i < CoreMethodInfo::OutObjectCount::value; ++i)
                {
                    pBaseObjects[i] = sf::detail::CmifProxyInfoAccessor::GetCmifBaseObject(pOutObjects[i].Get());
                }
                pMessage->AttachOutObjects(pBaseObjects.data());
                for (auto i = 0u; i < CoreMethodInfo::OutObjectCount::value; ++i)
                {
                    if (!static_cast<typename ProxyKind::ProxyBaseObject*>(pBaseObjects[i])->IsValid())
                    {
                        pOutObjects[i].Reset();
                    }
                }
                Discard(args.template GetOutObject<ArgumentInfos_>(pOutObjects)...);
            }
        }

        NN_SF_RESULT_SUCCESS;
    } // NOLINT[impl/function_size]

public:

    NN_NOINLINE static Result Process(ProxyBaseObject* p, typename GetClientArgumentType<ArgumentInfos_>::type... args, MethodId methodId) NN_NOEXCEPT
    {
        return ProcessImpl(p, std::forward<typename GetClientArgumentType<ArgumentInfos_>::type>(args)..., methodId, nullptr);
    }

    NN_NOINLINE static Result ProcessWithOutObjects(ProxyBaseObject* p, typename GetClientArgumentType<ArgumentInfos_>::type... args, MethodId methodId, SharedPointer<IServiceObject> pOutObjects[]) NN_NOEXCEPT
    {
        return ProcessImpl(p, std::forward<typename GetClientArgumentType<ArgumentInfos_>::type>(args)..., methodId, pOutObjects);
    }

};

/*
    GetClientProcessor::ProxyKind を使用し、MethodInfo に応じたプロキシ呼び出しを行う ClientProcessor を提供するクラステンプレート

    SFDL からの自動生成コードは、NN_SF_CMIF_CLIENT_DETAIL_CALL_BEGIN/END マクロによって、
    ClientProcessor を取得する。

    取得できる ClientProcessor は Process 関数テンプレートを持ち、以下の引数をとる。

    - pAllocator: 出力オブジェクトを生成するためのアロケータ
    - pProxyBaseObject: 呼び出し元のプロキシオブジェクト
    - ...args: ユーザ入力引数

    MethodInfo の出力オブジェクトの有無によって、実装が分岐する。

    - ない場合
      - CoreClientProcessor::Process(p, args..., MethodInfo::MethodId) 相当の呼び出しを行う
    - ある場合
      - pAllocator を使用して出力オブジェクト(のホルダ)をアロケートし、ポインタを pOutObjects に格納する
      - CoreClientProcessor::ProcessWithOutObjects(p, args..., pOutObjects, MethodInfo::MethodId) 相当の呼び出しを行う

    いずれの場合も CoreClientProcessor へ渡すテンプレートパラメータは以下をとる。

    - ProxyKind
    - CoreMethodInfo = GetCoreMethodInfo<MethodInfo>
      - MethodInfo から、引数型をある程度抽象化したもの
      - CoreMethodInfo が同じであれば、同じ CoreClientProcessor インスタンスを用いて同じ CMIF 呼び出しをすることができる
*/
template <typename ProxyKind, typename MethodInfo, typename Enabled = void>
struct GetClientProcessor;

template <typename ProxyKind, typename MethodInfo>
struct GetClientProcessor<ProxyKind, MethodInfo, typename std::enable_if<MethodInfo::OutObjectCount == 0>::type>
{
    using ProxyBaseObject = typename ProxyKind::ProxyBaseObject;
    using ClientMessage = typename ProxyKind::CmifClientMessage;

    using CoreMethodInfo = typename sf::cmif::detail::GetCoreMethodInfo<MethodInfo>::type;
    using CoreClientProcessor = ClientCoreProcessor<ProxyKind, CoreMethodInfo, typename CoreMethodInfo::ArgumentInfos>;

    struct ClientProcessor
    {
        template <typename AllocationPolicy, typename Allocator, typename... Args>
        static Result Process(Allocator*, ProxyBaseObject* p, Args&&... args) NN_NOEXCEPT
        {
            return CoreClientProcessor::Process(p, std::forward<Args>(args)..., MethodInfo::MethodId);
        }
    };

    using type = ClientProcessor;
};

template <typename ProxyKind>
struct GetClientProcessorWithObjectBase
{
    template <typename Type_, typename ArgumentInfo_>
    struct TypeAndArgumentInfo
    {
        using Type = Type_;
        using ArgumentInfo = ArgumentInfo_;
    };

    struct ZipTypeAndArgumentInfo
    {
        template <typename Type, typename ArgumentInfo>
        struct apply
        {
            using type = TypeAndArgumentInfo<Type, ArgumentInfo>;
        };
    };

    template <typename AllocationPolicy, typename Allocator>
    class ObjectAssigner
    {
    private:

        Allocator* m_pAllocator;
        SharedPointer<IServiceObject>* m_pObjects;

        template <typename OtherInterface>
        NN_NOINLINE void AllocateObject(int index) NN_NOEXCEPT
        {
            if (!m_pObjects)
            {
                return;
            }
            auto p = CmifProxyFactory<OtherInterface, ProxyKind, AllocationPolicy>::CreateProxyObject(m_pAllocator);
            if (!p)
            {
                this->m_pObjects = nullptr;
                return;
            }
            m_pObjects[index] = SharedPointer<OtherInterface>(p, false);
        }

    public:

        explicit ObjectAssigner(Allocator* pAllocator, SharedPointer<IServiceObject> pObjects[]) NN_NOEXCEPT
            : m_pAllocator(pAllocator)
            , m_pObjects(pObjects)
        {
        }

        template <typename OtherInterface, typename ArgumentInfo>
        void operator()(sf::detail::TypeInvokeTag<TypeAndArgumentInfo<sf::Out<sf::SharedPointer<OtherInterface>>, ArgumentInfo>>) NN_NOEXCEPT
        {
            AllocateObject<OtherInterface>(ArgumentInfo::Index::value);
        }

        bool IsValid() const
        {
            return m_pObjects != nullptr;
        }
    };
};

template <typename ProxyKind, typename MethodInfo>
struct GetClientProcessor<ProxyKind, MethodInfo, typename std::enable_if<(MethodInfo::OutObjectCount > 0)>::type>
    : public GetClientProcessorWithObjectBase<ProxyKind>
{
    using Base = GetClientProcessorWithObjectBase<ProxyKind>;
    using ZipTypeAndArgumentInfo = typename Base::ZipTypeAndArgumentInfo;

    using ProxyBaseObject = typename ProxyKind::ProxyBaseObject;
    using ClientMessage = typename ProxyKind::CmifClientMessage;

    using CoreMethodInfo = typename sf::cmif::detail::GetCoreMethodInfo<MethodInfo>::type;
    using CoreClientProcessor = ClientCoreProcessor<ProxyKind, CoreMethodInfo, typename CoreMethodInfo::ArgumentInfos>;

    struct IsOutObjectArgumentInfoWithType
    {
        template <typename ArgumentInfoWithType>
        struct predicate
            : public std::integral_constant<bool, ArgumentInfoWithType::ArgumentInfo::Kind::value == cmif::detail::ArgumentKind::OutObject>
        {
        };
    };

    struct ClientProcessor
    {

        template <typename AllocationPolicy, typename Allocator, typename... Args>
        static Result Process(Allocator* pAllocator, ProxyBaseObject* p, Args&&... args) NN_NOEXCEPT
        {
            std::array<SharedPointer<IServiceObject>, MethodInfo::OutObjectCount> pOutObjects;
            {
                using ArgumentInfosWithType = typename sf::detail::TypeZip<ZipTypeAndArgumentInfo, std::tuple<Args...>, typename MethodInfo::ArgumentInfos>::type;
                using OutObjectArgumentInfosWithType = typename sf::detail::TypeFilter<IsOutObjectArgumentInfoWithType, ArgumentInfosWithType>::type;
                static_assert(sf::detail::TypeSizeOf<OutObjectArgumentInfosWithType>::value == MethodInfo::OutObjectCount, "");
                using ObjectAssigner = typename GetClientProcessorWithObjectBase<ProxyKind>::template ObjectAssigner<AllocationPolicy, Allocator>;
                ObjectAssigner assigner(pAllocator, pOutObjects.data());
                sf::detail::TypeForeachInvoke<OutObjectArgumentInfosWithType>(assigner);
                NN_SF_RESULT_THROW_UNLESS(assigner.IsValid(), sf::cmif::ResultCmifProxyAllocationFailed());
            }
            return CoreClientProcessor::ProcessWithOutObjects(p, std::forward<Args>(args)..., MethodInfo::MethodId, pOutObjects.data());
        }
    };

    using type = ClientProcessor;
};

}}}}
