﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>

namespace nn { namespace ro {
namespace detail {

    struct NrrCertification
    {
        static const int RsaKeySize = 0x100;
        static const int SignAreaSize = 0x120;
        Bit64           programIdMask;
        Bit64           programIdPattern;
        Bit8            rserved0x10[0x10];
        Bit8            publicKey[RsaKeySize];
        Bit8            sign[RsaKeySize];
    };
    NN_STATIC_ASSERT( sizeof(NrrCertification) == 0x220 );

    class NrrHeader
    {
    public:
        static const Bit32  SignatureValue0         = 0x3052524E; // NRR0
        static const int    RsaKeySize              = 0x100;
        static const int    HashSize                = 0x20;
        static const int    SignAreaStartOffset     = 0x330;

    public:
        // バージョンが複数ある場合に、どのバージョンかを受け側で判定できるように値を返す。
        // 見つからないときは 0 を返す
        Bit32 CheckSignature() const NN_NOEXCEPT
        {
            if (m_Signature == SignatureValue0)
            {
                return m_Signature;
            }
            return 0;
        }

        uint32_t GetSize() const NN_NOEXCEPT
        {
            return m_Size;
        }

        const Bit8* GetCertificationSign() const NN_NOEXCEPT
        {
            return reinterpret_cast<const Bit8*>(m_Certification.sign);
        }

        const Bit8* GetCertificationSignArea() const NN_NOEXCEPT
        {
            return reinterpret_cast<const Bit8*>(&m_Certification);
        }

        const Bit8* GetCertificationPublicKey() const NN_NOEXCEPT
        {
            return reinterpret_cast<const Bit8*>(m_Certification.publicKey);
        }

        const Bit8* GetSign() const NN_NOEXCEPT
        {
            return reinterpret_cast<const Bit8*>(m_Sign);
        }

        const Bit8* GetSignArea() const NN_NOEXCEPT
        {
            return reinterpret_cast<const Bit8*>(&m_ProgramId);
        }

        size_t GetSignAreaSize() const NN_NOEXCEPT
        {
            return m_Size - SignAreaStartOffset;
        }

        uintptr_t GetHashList() const NN_NOEXCEPT
        {
            return reinterpret_cast<uintptr_t>(this) + m_HashListOffsetAddress;
        }

        uint32_t GetNumHash() const NN_NOEXCEPT
        {
            return m_NumHash;
        }

        uint64_t GetProgramId() const NN_NOEXCEPT
        {
            return m_ProgramId;
        }

        bool CheckProgramId() const NN_NOEXCEPT
        {
            return (m_ProgramId & m_Certification.programIdMask) == m_Certification.programIdPattern;
        }

    private:
        Bit32               m_Signature;
        Bit8                m_Reserved0[0xC] NN_IS_UNUSED_MEMBER;

        NrrCertification    m_Certification;
        Bit8                m_Sign[RsaKeySize];

        Bit64               m_ProgramId;
        uint32_t            m_Size;
        NN_PADDING4;

        uint32_t            m_HashListOffsetAddress;
        uint32_t            m_NumHash;
        Bit8                m_Reserved1[0x8] NN_IS_UNUSED_MEMBER;
    };

    NN_STATIC_ASSERT( sizeof(NrrHeader) == 0x350 );

} // detail
}} // nn::ro

