﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

#include <nn/nn_Common.h>
#include <nn/nn_Result.h>
#include <nn/nn_Abort.h>
#include <nn/util/util_ScopeExit.h>

#include <type_traits>

#if defined(NN_SDK_BUILD_DEBUG)
    #define NN_DETAIL_RESULT_HANDLING_UTILITY_RECORDS_EVENT 1 // NOLINT(preprocessor/const)
#elif defined(NN_SDK_BUILD_DEVELOP)
    #define NN_DETAIL_RESULT_HANDLING_UTILITY_RECORDS_EVENT 1 // NOLINT(preprocessor/const)
#elif defined(NN_SDK_BUILD_RELEASE)
    #define NN_DETAIL_RESULT_HANDLING_UTILITY_RECORDS_EVENT 0 // NOLINT(preprocessor/const)
#else
    #error
#endif

#ifdef NN_RESULT_HANDLING_UTILITY_SUPPRESS_RECORDING_EVENT
    #undef NN_DETAIL_RESULT_HANDLING_UTILITY_RECORDS_EVENT
    #define NN_DETAIL_RESULT_HANDLING_UTILITY_RECORDS_EVENT 0 // NOLINT(preprocessor/const)
#endif

#define NN_RESULT_HANDLING_UTILITY_TEMPORARILY_SUPPRESS_RECORDING_EVENT 0 // NOLINT(preprocessor/const)

#define NN_DETAIL_RESULT_DECLARE_SOURCE_INFO_SUPPRESS_RECORDING_0(v) const ::nn::diag::SourceInfo& v = NN_DETAIL_RESULT_MAKE_SOURCE_INFO; NN_UNUSED(v)
#define NN_DETAIL_RESULT_HANDLING_UTILITY_ON_THROW_SUPPRESS_RECORDING_0(r) ::nn::diag::OnResultThrow(r, NN_DETAIL_RESULT_MAKE_SOURCE_INFO)
#define NN_DETAIL_RESULT_HANDLING_UTILITY_ON_HANDLE_SUPPRESS_RECORDING_0(r, sourceInfo) ::nn::diag::OnResultHandle(r, sourceInfo)
#define NN_DETAIL_RESULT_HANDLING_UTILITY_ON_RETHROW_SUPPRESS_RECORDING_0(r) ::nn::diag::OnResultRethrow(r, NN_DETAIL_RESULT_MAKE_SOURCE_INFO)
#define NN_DETAIL_RESULT_HANDLING_UTILITY_ON_CATCH_SUPPRESS_RECORDING_0(r) ::nn::diag::OnResultCatch(r, NN_DETAIL_RESULT_MAKE_SOURCE_INFO)
#define NN_DETAIL_RESULT_HANDLING_UTILITY_ON_MISS_SUPPRESS_RECORDING_0(r, sourceInfo) ::nn::diag::OnResultMiss(r, sourceInfo)

#define NN_DETAIL_RESULT_DECLARE_SOURCE_INFO_SUPPRESS_RECORDING_1(v) static_cast<void>(0)
#define NN_DETAIL_RESULT_HANDLING_UTILITY_ON_THROW_SUPPRESS_RECORDING_1(r) static_cast<void>(0)
#define NN_DETAIL_RESULT_HANDLING_UTILITY_ON_HANDLE_SUPPRESS_RECORDING_1(r, sourceInfo) static_cast<void>(0)
#define NN_DETAIL_RESULT_HANDLING_UTILITY_ON_RETHROW_SUPPRESS_RECORDING_1(r) static_cast<void>(0)
#define NN_DETAIL_RESULT_HANDLING_UTILITY_ON_CATCH_SUPPRESS_RECORDING_1(r) static_cast<void>(0)
#define NN_DETAIL_RESULT_HANDLING_UTILITY_ON_MISS_SUPPRESS_RECORDING_1(r, sourceInfo) static_cast<void>(0)

#if NN_DETAIL_RESULT_HANDLING_UTILITY_RECORDS_EVENT
    #include <nn/diag/diag_ResultTraceEvent.h>
    #define NN_DETAIL_RESULT_MAKE_SOURCE_INFO { __LINE__, __FILE__, NN_CURRENT_FUNCTION_NAME }
    #define NN_DETAIL_RESULT_DECLARE_SOURCE_INFO(v) NN_MACRO_CONCATENATE(NN_DETAIL_RESULT_DECLARE_SOURCE_INFO_SUPPRESS_RECORDING_, NN_RESULT_HANDLING_UTILITY_TEMPORARILY_SUPPRESS_RECORDING_EVENT)(v)
    #define NN_DETAIL_RESULT_HANDLING_UTILITY_ON_THROW(r) NN_MACRO_CONCATENATE(NN_DETAIL_RESULT_HANDLING_UTILITY_ON_THROW_SUPPRESS_RECORDING_, NN_RESULT_HANDLING_UTILITY_TEMPORARILY_SUPPRESS_RECORDING_EVENT)(r)
    #define NN_DETAIL_RESULT_HANDLING_UTILITY_ON_HANDLE(r, sourceInfo) NN_MACRO_CONCATENATE(NN_DETAIL_RESULT_HANDLING_UTILITY_ON_HANDLE_SUPPRESS_RECORDING_, NN_RESULT_HANDLING_UTILITY_TEMPORARILY_SUPPRESS_RECORDING_EVENT)(r, sourceInfo)
    #define NN_DETAIL_RESULT_HANDLING_UTILITY_ON_RETHROW(r) NN_MACRO_CONCATENATE(NN_DETAIL_RESULT_HANDLING_UTILITY_ON_RETHROW_SUPPRESS_RECORDING_, NN_RESULT_HANDLING_UTILITY_TEMPORARILY_SUPPRESS_RECORDING_EVENT)(r)
    #define NN_DETAIL_RESULT_HANDLING_UTILITY_ON_CATCH(r) NN_MACRO_CONCATENATE(NN_DETAIL_RESULT_HANDLING_UTILITY_ON_CATCH_SUPPRESS_RECORDING_, NN_RESULT_HANDLING_UTILITY_TEMPORARILY_SUPPRESS_RECORDING_EVENT)(r)
    #define NN_DETAIL_RESULT_HANDLING_UTILITY_ON_MISS(r, sourceInfo) NN_MACRO_CONCATENATE(NN_DETAIL_RESULT_HANDLING_UTILITY_ON_MISS_SUPPRESS_RECORDING_, NN_RESULT_HANDLING_UTILITY_TEMPORARILY_SUPPRESS_RECORDING_EVENT)(r, sourceInfo)
#else
    #define NN_DETAIL_RESULT_MAKE_SOURCE_INFO {}
    #define NN_DETAIL_RESULT_DECLARE_SOURCE_INFO(v) NN_DETAIL_RESULT_DECLARE_SOURCE_INFO_SUPPRESS_RECORDING_1(v)
    #define NN_DETAIL_RESULT_HANDLING_UTILITY_ON_THROW(r) NN_DETAIL_RESULT_HANDLING_UTILITY_ON_THROW_SUPPRESS_RECORDING_1(r)
    #define NN_DETAIL_RESULT_HANDLING_UTILITY_ON_HANDLE(r, sourceInfo) NN_DETAIL_RESULT_HANDLING_UTILITY_ON_HANDLE_SUPPRESS_RECORDING_1(r, sourceInfo)
    #define NN_DETAIL_RESULT_HANDLING_UTILITY_ON_RETHROW(r) NN_DETAIL_RESULT_HANDLING_UTILITY_ON_RETHROW_SUPPRESS_RECORDING_1(r)
    #define NN_DETAIL_RESULT_HANDLING_UTILITY_ON_CATCH(r) NN_DETAIL_RESULT_HANDLING_UTILITY_ON_CATCH_SUPPRESS_RECORDING_1(r)
    #define NN_DETAIL_RESULT_HANDLING_UTILITY_ON_MISS(r, sourceInfo) NN_DETAIL_RESULT_HANDLING_UTILITY_ON_MISS_SUPPRESS_RECORDING_1(r, sourceInfo)
#endif

#define NN_DETAIL_RESULT_BEGIN_DECLARE_TEMPORARY_VARIABLE \
    NN_PRAGMA_PUSH_WARNINGS \
    NN_DISABLE_WARNING_SHADOW

#define NN_DETAIL_RESULT_END_DECLARE_TEMPORARY_VARIABLE \
    NN_PRAGMA_POP_WARNINGS

namespace nn { namespace result {

namespace detail {

    void ConvertInvalidImpl(Result result) NN_NOEXCEPT;

    template <typename ReturnType, typename = void>
    class ResultToReturnTypeConverter
    {
    public:
        static ReturnType Convert(Result result) NN_NOEXCEPT
        {
            ConvertInvalidImpl(result);
            return {};
        }
    };

    template <typename ReturnType>
    class ResultToReturnTypeConverter<ReturnType, typename std::enable_if<std::is_convertible<Result, ReturnType>::value>::type>
    {
    public:
        static ReturnType Convert(Result result) NN_NOEXCEPT
        {
            return result;
        }
    };

#if NN_DETAIL_RESULT_HANDLING_UTILITY_RECORDS_EVENT

    class ResultAutoConverterWithSourceInfo
    {
    private:
        Result m_Result;
        const diag::SourceInfo& m_SourceInfo;
    public:
        explicit ResultAutoConverterWithSourceInfo(Result result, const diag::SourceInfo& sourceInfo) NN_NOEXCEPT
            : m_Result(result)
            , m_SourceInfo(sourceInfo)
        {
        }
        template <typename ReturnType>
        NN_IMPLICIT operator ReturnType() const NN_NOEXCEPT
        {
            if (NN_STATIC_CONDITION(!(std::is_convertible<Result, ReturnType>::value)))
            {
                NN_DETAIL_RESULT_HANDLING_UTILITY_ON_MISS(m_Result, m_SourceInfo);
            }
            return ResultToReturnTypeConverter<ReturnType>::Convert(m_Result);
        }
    };

    inline ResultAutoConverterWithSourceInfo MakeResultAutoConverterWithSourceInfo(Result result, const diag::SourceInfo& sourceInfo) NN_NOEXCEPT
    {
        return ResultAutoConverterWithSourceInfo(result, sourceInfo);
    }

    #define NN_DETAIL_RESULT_THROW_IMPL(r) \
        do \
        { \
            NN_DETAIL_RESULT_BEGIN_DECLARE_TEMPORARY_VARIABLE \
            NN_DETAIL_RESULT_DECLARE_SOURCE_INFO(_nn_result_throw_impl_source_info); \
            NN_DETAIL_RESULT_END_DECLARE_TEMPORARY_VARIABLE \
            return ::nn::result::detail::MakeResultAutoConverterWithSourceInfo(r, _nn_result_throw_impl_source_info); \
        } while (NN_STATIC_CONDITION(0))

#else

    class ResultAutoConverter
    {
    private:
        Result m_Result;
    public:
        explicit ResultAutoConverter(Result result) NN_NOEXCEPT : m_Result(result) {}
        template <typename ReturnType>
        NN_IMPLICIT operator ReturnType() const NN_NOEXCEPT
        {
            return ResultToReturnTypeConverter<ReturnType>::Convert(m_Result);
        }
    };

    inline ResultAutoConverter MakeResultAutoConverter(Result result) NN_NOEXCEPT
    {
        return ResultAutoConverter(result);
    }

    #define NN_DETAIL_RESULT_THROW_IMPL(r) return ::nn::result::detail::MakeResultAutoConverter(r)

#endif

    template <typename HandleResult, typename F>
    class ResultBlockHolder
    {
        NN_DISALLOW_COPY(ResultBlockHolder);
    private:

        F m_F;
        bool m_Valid;

    public:

        explicit ResultBlockHolder(F&& f) NN_NOEXCEPT
            : m_F(std::move(f))
            , m_Valid(true)
        {
        }

        ResultBlockHolder(ResultBlockHolder&& other) NN_NOEXCEPT
            : m_F(std::move(other.m_F))
            , m_Valid(other.m_Valid)
        {
            other.m_Valid = false;
        }

        ~ResultBlockHolder() NN_NOEXCEPT
        {
            if (m_Valid)
            {
                HandleResult::Handle(m_F());
            }
        }

    };

    template <typename HandleResult>
    struct ResultBlockHolderHelper
    {
    };

    template <typename HandleResult, typename F>
    ResultBlockHolder<HandleResult, F> operator+(ResultBlockHolderHelper<HandleResult>, F&& rhs) NN_NOEXCEPT
    {
        return ResultBlockHolder<HandleResult, F>(std::forward<F>(rhs));
    }

    struct AbortForResult
    {
        static void Handle(Result result) NN_NOEXCEPT
        {
            NN_ABORT_UNLESS_RESULT_SUCCESS(result);
        }
    };

    struct IgnoreForResult
    {
        static void Handle(Result result) NN_NOEXCEPT
        {
            NN_UNUSED(result);
            NN_DETAIL_RESULT_DECLARE_SOURCE_INFO(sourceInfo);
            NN_DETAIL_RESULT_HANDLING_UTILITY_ON_HANDLE(result, sourceInfo);
        }
    };

}

}}

#if NN_DETAIL_RESULT_HANDLING_UTILITY_RECORDS_EVENT
    #define NN_DETAIL_RESULT_CATCH_PREAMBLE \
                NN_DETAIL_RESULT_HANDLING_UTILITY_ON_CATCH(_nn_result_try_temporary); \
                NN_DETAIL_RESULT_BEGIN_DECLARE_TEMPORARY_VARIABLE \
                auto _nn_result_try_rethrow = false; \
                NN_DETAIL_RESULT_DECLARE_SOURCE_INFO(_nn_result_try_catch_source_info); \
                NN_UNUSED(_nn_result_try_catch_source_info); \
                NN_DETAIL_RESULT_END_DECLARE_TEMPORARY_VARIABLE \
                NN_UTIL_SCOPE_EXIT \
                { \
                    if (!_nn_result_try_rethrow) \
                    { \
                        NN_DETAIL_RESULT_HANDLING_UTILITY_ON_HANDLE(_nn_result_try_temporary, _nn_result_try_catch_source_info); \
                    } \
                };

    #define NN_DETAIL_RESULT_MARK_AS_RETHROW \
                do \
                { \
                    _nn_result_try_rethrow = true; \
                } while (NN_STATIC_CONDITION(0))

#else
    #define NN_DETAIL_RESULT_CATCH_PREAMBLE
    #define NN_DETAIL_RESULT_MARK_AS_RETHROW static_cast<void>(0)
#endif


// 以下、インターフェイス

#define NN_RESULT_SUCCESS \
    return ::nn::ResultSuccess()

#define NN_RESULT_THROW(r) \
    do \
    { \
        NN_DETAIL_RESULT_BEGIN_DECLARE_TEMPORARY_VARIABLE \
        const auto& _nn_result_throw_temporary = r; \
        NN_DETAIL_RESULT_END_DECLARE_TEMPORARY_VARIABLE \
        NN_DETAIL_RESULT_HANDLING_UTILITY_ON_THROW(_nn_result_throw_temporary); \
        NN_DETAIL_RESULT_THROW_IMPL(_nn_result_throw_temporary); \
    } while (NN_STATIC_CONDITION(0))

#define NN_RESULT_THROW_UNLESS(condition, r) \
    do \
    { \
        if (!(condition)) \
        { \
            NN_RESULT_THROW(r); \
        } \
    } while (NN_STATIC_CONDITION(0))

#define NN_RESULT_DO(r) \
    do \
    { \
        const auto& _nn_result_do_temporary = r; \
        if (!_nn_result_do_temporary.IsSuccess()) \
        { \
            NN_DETAIL_RESULT_BEGIN_DECLARE_TEMPORARY_VARIABLE \
            NN_DETAIL_RESULT_DECLARE_SOURCE_INFO(_nn_result_do_impl_source_info); \
            NN_DETAIL_RESULT_END_DECLARE_TEMPORARY_VARIABLE \
            NN_DETAIL_RESULT_HANDLING_UTILITY_ON_MISS(_nn_result_do_temporary, _nn_result_do_impl_source_info); \
            NN_DETAIL_RESULT_THROW_IMPL(_nn_result_do_temporary); \
        } \
    } while (NN_STATIC_CONDITION(0))

#define NN_RESULT_TRY(r) \
    { \
        NN_DETAIL_RESULT_BEGIN_DECLARE_TEMPORARY_VARIABLE \
        const auto& _nn_result_try_temporary = r; \
        NN_DETAIL_RESULT_END_DECLARE_TEMPORARY_VARIABLE \
        if (!_nn_result_try_temporary.IsSuccess()) \
        { \
            if (NN_STATIC_CONDITION(false)) \
            {

#define NN_RESULT_CATCH(type) \
            } \
            else if (type::Includes(_nn_result_try_temporary)) \
            { \
                NN_DETAIL_RESULT_CATCH_PREAMBLE

#define NN_RESULT_CATCH_ALL \
            } \
            else if (NN_STATIC_CONDITION(true)) \
            { \
                NN_DETAIL_RESULT_CATCH_PREAMBLE

#define NN_RESULT_CURRENT_RESULT \
            _nn_result_try_temporary

#define NN_RESULT_MARK_AS_RETHROW NN_DETAIL_RESULT_MARK_AS_RETHROW

#define NN_RESULT_RETHROW \
                do \
                { \
                    NN_DETAIL_RESULT_MARK_AS_RETHROW; \
                    NN_DETAIL_RESULT_HANDLING_UTILITY_ON_RETHROW(_nn_result_try_temporary); \
                    NN_DETAIL_RESULT_THROW_IMPL(_nn_result_try_temporary); \
                } while (NN_STATIC_CONDITION(0))

#define NN_RESULT_CATCH_CONVERT(type, r) \
            NN_RESULT_CATCH(type) \
                NN_RESULT_MARK_AS_RETHROW; \
                NN_RESULT_THROW(r);

#define NN_RESULT_END_TRY \
            } \
            else \
            { \
                NN_DETAIL_RESULT_BEGIN_DECLARE_TEMPORARY_VARIABLE \
                NN_DETAIL_RESULT_DECLARE_SOURCE_INFO(_nn_result_end_try_impl_source_info); \
                NN_DETAIL_RESULT_END_DECLARE_TEMPORARY_VARIABLE \
                NN_DETAIL_RESULT_HANDLING_UTILITY_ON_MISS(_nn_result_try_temporary, _nn_result_end_try_impl_source_info); \
                NN_DETAIL_RESULT_THROW_IMPL(_nn_result_try_temporary); \
            } \
        } \
    }

#define NN_RESULT_ABORTING_BLOCK \
    ::nn::result::detail::ResultBlockHolderHelper<::nn::result::detail::AbortForResult>() + [&]() -> ::nn::Result

#define NN_RESULT_IGNORING_BLOCK \
    ::nn::result::detail::ResultBlockHolderHelper<::nn::result::detail::IgnoreForResult>() + [&]() -> ::nn::Result
