﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   スレッド情報の取得に関する API の宣言
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Result.h>
#include <nn/osdbg/osdbg_Result.h>
#include <nn/osdbg/osdbg_ThreadApiImpl.h>

#include <nn/svc/svc_Base.h>
#include <nn/svc/svc_Dmnt.h>
#include <nn/svc/svc_Handle.h>


namespace nn { namespace osdbg {

struct  ThreadInfo;

//! @name スレッド情報の取得に関する API
//! @{

//-----------------------------------------------------------------------------
/**
 * @brief   ThreadInfo オブジェクトを初期化します
 *
 * @param[in] pThreadInfo               初期化する ThreadInfo オブジェクトへのポインタ
 * @param[in] debugHandle               デバッグ対象の Debug を指すハンドル
 * @param[in] pDebugInfoCreateProcess   svc::DebugInfoCreateProcess オブジェクトへのポインタ
 * @param[in] pDebugInfoCreateThread    svc::DebugInfoCreateThread オブジェクトへのポインタ
 *
 * @retresult
 *   @handleresult{nn::osdbg::ResultCannotGetThreadInfo}
 * @endretresult
 *
 * @details
 *  スレッド情報取得に必要となる ThreadInfo オブジェクトを初期化します。@n
 *  以後、ThreadInfo オブジェクトを使うことで、デバッグ対象スレッドの情報を
 *  nn::osdbg::GetThreadPriority() などで取得できるようになります。
 *
 *  pDebugInfoCreateProcess および pDebugInfoCreateThread は、
 *  カーネルからデバッグイベント通知を受けた後に nn::svc::GetDebugEvent() で
 *  取得したオブジェクトへのポインタを渡して下さい。
 *
 *  これらの情報は pThreadInfo オブジェクト内にコピーされて保持されます。
 *
 */
nn::Result InitializeThreadInfo(ThreadInfo* pThreadInfo, nn::svc::Handle debugHandle, const nn::svc::DebugInfoCreateProcess* pDebugInfoCreateProcess, const nn::svc::DebugInfoCreateThread* pDebugInfoCreateThread) NN_NOEXCEPT;


//-----------------------------------------------------------------------------
/**
 * @brief   最新のスレッド情報を取得し ThreadInfo 構造体にセットします。
 *
 * @param[in] pThreadInfo   ThreadInfo オブジェクトへのポインタ
 *
 * @retresult
 *   @handleresult{nn::osdbg::ResultCannotGetThreadInfo}
 * @endretresult
 *
 * @pre
 *  - ThreadInfo オブジェクトが InitializeThreadInfo() 済みである
 *
 * @details
 *  ThreadInfo オブジェクトが指すデバッグ対象の最新のスレッド情報を
 *  再取得します。最初に InitializeThreadInfo() で取得したスレッド情報を
 *  最新の情報に更新する場合に使用します。
 *
 */
nn::Result UpdateThreadInfo(ThreadInfo* pThreadInfo) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   デバッグ対象スレッドの本来の優先度を取得します。
 *
 * @param[in] pThreadInfo   ThreadInfo オブジェクトへのポインタ
 *
 * @return  デバッグ対象スレッドの本来の優先度です。
 *
 * @pre
 *  - ThreadInfo オブジェクトが InitializeThreadInfo() 済みである
 *
 * @details
 *  対象スレッドの本来の優先度を返します。@n
 *  本来の優先度とは、ミューテックス等の優先度継承によって変化する前の
 *  スレッド優先度です。
 *
 */
int GetThreadPriority(const ThreadInfo* pThreadInfo) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   デバッグ対象スレッドの現在の優先度を取得します。
 *
 * @param[in] pThreadInfo   ThreadInfo オブジェクトへのポインタ
 *
 * @return  デバッグ対象スレッドの現在の優先度です。
 *
 * @pre
 *  - ThreadInfo オブジェクトが InitializeThreadInfo() 済みである
 *
 * @details
 *  対象スレッドの現在の優先度を返します。@n
 *  ミューテックスの優先度継承が発動している場合は、継承中の優先度を返します。
 *
 */
int GetThreadCurrentPriority(const ThreadInfo* pThreadInfo) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   デバッグ対象スレッドのスレッド名格納アドレスを取得します。
 *
 * @param[in] pThreadInfo   ThreadInfo オブジェクトへのポインタ
 *
 * @return  デバッグ対象スレッドのスレッド名格納アドレスです。
 *
 * @pre
 *  - ThreadInfo オブジェクトが InitializeThreadInfo() 済みである
 *
 * @details
 *  対象スレッドのスレッド名を格納しているアドレスを返します。@n
 *  アドレスはデバッグ対象プロセス空間内のアドレスを指しています。
 *
 *  スレッド名の長さに規定はありません。そのため、スレッド名を取得する際は
 *  終端のヌル文字までを取得するか、もしくは適当な長さで取得を中止するなど
 *  して下さい。
 *
 */
uintptr_t GetThreadNamePointer(const ThreadInfo* pThreadInfo) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   デバッグ対象スレッドのスタック領域の先頭アドレスを取得します。
 *
 * @param[in] pThreadInfo   ThreadInfo オブジェクトへのポインタ
 *
 * @return  デバッグ対象スレッドのスタック領域の先頭アドレスです。
 *
 * @pre
 *  - ThreadInfo オブジェクトが InitializeThreadInfo() 済みである
 *
 * @details
 *  対象スレッドのスタック領域の先頭アドレスを取得します。@n
 *  アドレスはデバッグ対象プロセス空間内のアドレスを指しています。
 *
 *  スタックがエイリアスされている場合、このアドレスは CreateThread() 時に
 *  渡されたスタック領域とは異なっており、エイリアスした実際に使用中の
 *  スタック領域のアドレスが返されます。
 *
 */
uintptr_t GetThreadStackAddress(const ThreadInfo* pThreadInfo) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   デバッグ対象スレッドのスタック領域のサイズを取得します。
 *
 * @param[in] pThreadInfo   ThreadInfo オブジェクトへのポインタ
 *
 * @return  デバッグ対象スレッドのスタック領域のサイズです。
 *
 * @pre
 *  - ThreadInfo オブジェクトが InitializeThreadInfo() 済みである
 *
 * @details
 *  対象スレッドのスタック領域のサイズを取得します。
 *
 */
size_t GetThreadStackSize(const ThreadInfo* pThreadInfo) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   デバッグ対象スレッドのスレッド関数エントリアドレスを取得します。
 *
 * @param[in] pThreadInfo   ThreadInfo オブジェクトへのポインタ
 *
 * @return  デバッグ対象スレッドのスレッド関数エントリアドレスです。
 *
 * @pre
 *  - ThreadInfo オブジェクトが InitializeThreadInfo() 済みである
 *
 * @details
 *  対象スレッドのスレッド関数エントリアドレスを返します。@n
 *  アドレスはデバッグ対象プロセス空間内のアドレスを指しています。
 *
 */
uintptr_t GetThreadFunction(const ThreadInfo* pThreadInfo) NN_NOEXCEPT;


//--------------------------------------------------------------------------
/**
 * @brief   デバッグ対象スレッドのスレッド関数への引数を取得します。
 *
 * @param[in] pThreadInfo   ThreadInfo オブジェクトへのポインタ
 *
 * @return  デバッグ対象スレッドのスレッド関数への引数です。
 *
 * @pre
 *  - ThreadInfo オブジェクトが InitializeThreadInfo() 済みである
 *
 * @details
 *  対象スレッドのスレッド関数に渡される引数の値を返します。
 *
 */
uintptr_t GetThreadFunctionArgument(const ThreadInfo* pThreadInfo) NN_NOEXCEPT;


//-----------------------------------------------------------------------------

//! @}

}} // namespace nn::osdbg

