﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief   共有メモリ機能に関する公開ヘッダファイル
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_SharedMemoryTypes.h>
#include <nn/os/os_SharedMemoryApi.h>

namespace nn { namespace os {

//--------------------------------------------------------------------------
/**
 * @brief   共有メモリを扱うためのクラスです。
 *
 * @details
 *  共有メモリは、プロセス間でメモリを共有するためのオブジェクトです。
 *
 *  nn::os::SharedMemory クラスは、@ref nn::os::SharedMemoryType オブジェクトと
 *  関連する API の呼び出しをラッピングしたユーティリティクラスです。
 *  本クラスが提供するメンバ関数は、@ref nn::os::SharedMemoryType 型を引数にとる
 *  @ref nn::os::AttachSharedMemory() などの API を発行する形で実装されています。
 */
class SharedMemory
{
    NN_DISALLOW_COPY( SharedMemory );
    NN_DISALLOW_MOVE( SharedMemory );

public:
    //--------------------------------------------------------------------------
    /**
     * @brief   共有メモリオブジェクトを未初期化で用意します。
     *
     * @details
     *  コンストラクタです。@n
     *  内部の共有メモリオブジェクトは未初期化状態になります。
     *  この後に Attach() することで、共有メモリが使用できるようになります。
     */
    SharedMemory() NN_NOEXCEPT
    {
        m_SharedMemory._state = SharedMemoryType::State_NotInitialized;
    }

    //--------------------------------------------------------------------------
    /**
     * @brief   共有メモリオブジェクトを作成し初期化します。
     *
     * @param[in] size              共有メモリのサイズ
     * @param[in] myPermission      自プロセスに許可されるメモリアクセス権
     * @param[in] otherPermission   他プロセスに許可されるメモリアクセス権
     *
     * @details
     *  コンストラクタです。@n
     *  内部の SharedMemoryType オブジェクトを指定されたパラメータで初期化します。@n
     *  詳細は nn::os::CreateSharedMemory() を参照して下さい。
     *
     *  ただし、本クラスでは nn::os::CreateSharedMemory とは異なり、
     *  ハンドルまたは共有メモリのリソースが足りない場合アボートします。
     *
     */
    SharedMemory(size_t size, MemoryPermission myPermission, MemoryPermission otherPermission) NN_NOEXCEPT
    {
        auto result = CreateSharedMemory(&m_SharedMemory, size, myPermission, otherPermission);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    //--------------------------------------------------------------------------
    /**
     * @brief   指定されたハンドルを使って共有メモリオブジェクトを初期化します。
     *
     * @param[in] size          共有メモリのサイズ
     * @param[in] handle        共有メモリのハンドル
     * @param[in] managed       共有メモリのハンドル管理フラグ
     *
     * @details
     *  コンストラクタです。@n
     *  指定されたハンドルを使って共有メモリオブジェクトを初期化します。@n
     *  詳細は nn::os::AttachSharedMemory() を参照して下さい。
     *
     */
    SharedMemory(size_t size, NativeHandle handle, bool managed) NN_NOEXCEPT
    {
        AttachSharedMemory(&m_SharedMemory, size, handle, managed);
    }

    //--------------------------------------------------------------------------
    /**
     * @brief  共有メモリオブジェクトを破棄します。
     *
     * @details
     *  デストラクタです。@n
     *  共有メモリオブジェクトを破棄します。@n
     *  マップされている場合には、自動的にアンマップされます。@n
     *  詳細は nn::os::DestroySharedMemory() を参照して下さい。
     *
     *  ただし、本クラスでは nn::os::DestroySharedMemory() とは異なり、
     *  共有メモリオブジェクトが未初期化の場合、何もしません。
     *
     */
    ~SharedMemory() NN_NOEXCEPT
    {
        if (m_SharedMemory._state == SharedMemoryType::State_NotInitialized)
        {
            return;
        }
        DestroySharedMemory(&m_SharedMemory);
    }


    //--------------------------------------------------------------------------
    /**
     * @brief   既存の共有メモリハンドルを SharedMemoryType オブジェクトに関連付けします。
     *
     * @param[in] size          共有メモリのサイズ
     * @param[in] handle        共有メモリのハンドル
     * @param[in] managed       共有メモリのハンドル管理フラグ
     *
     * @details
     *  指定されたハンドルを使って共有メモリオブジェクトを初期化します。@n
     *  詳細は nn::os::AttachSharedMemory() を参照して下さい。
     *
     */
    void Attach(size_t size, NativeHandle handle, bool managed) NN_NOEXCEPT
    {
        AttachSharedMemory(&m_SharedMemory, size, handle, managed);
    }


    //--------------------------------------------------------------------------
    /**
     * @brief   共有メモリをマップします。
     *
     * @param[in] myPermission  マップするメモリ領域に設定するアクセス権
     *
     * @return  共有メモリをマップしたアドレスを返します。
     *
     * @details
     *  詳細は nn::os::MapSharedMemory(SharedMemoryType*, MemoryPermission)
     *  を参照して下さい。
     *
     */
    void* Map(MemoryPermission myPermission) NN_NOEXCEPT
    {
        return MapSharedMemory(&m_SharedMemory, myPermission);
    }

    //--------------------------------------------------------------------------
    /**
     * @brief   共有メモリをアンマップします。
     *
     * @details
     *  詳細は nn::os::UnmapSharedMemory() を参照して下さい。
     *
     */
    void Unmap() NN_NOEXCEPT
    {
        UnmapSharedMemory(&m_SharedMemory);
    }


    //--------------------------------------------------------------------------
    /**
     * @brief   共有メモリオブジェクトの マップ先アドレス を返します。
     *
     * @return  共有メモリのマップ先アドレスを返します。
     *
     * @details
     *  詳細は nn::os::GetSharedMemoryAddress() を参照して下さい。
     *
     */
    void* GetMappedAddress() const NN_NOEXCEPT
    {
        return GetSharedMemoryAddress(&m_SharedMemory);
    }

    //--------------------------------------------------------------------------
    /**
     * @brief   共有メモリのサイズを返します。
     *
     * @return  共有メモリのサイズを返します。
     *
     * @details
     *  詳細は nn::os::GetSharedMemorySize() を参照して下さい。
     *
     */
    size_t GetSize() const NN_NOEXCEPT
    {
        return GetSharedMemorySize(&m_SharedMemory);
    }


    //--------------------------------------------------------------------------
    /**
     * @brief   共有メモリオブジェクトの ハンドル を返します。
     *
     * @return  共有メモリオブジェクトのハンドルを返します。
     *
     * @details
     *  詳細は nn::os::GetSharedMemoryHandle() を参照して下さい。
     *
     */
    NativeHandle GetHandle() const NN_NOEXCEPT
    {
        return GetSharedMemoryHandle(&m_SharedMemory);
    }


    //--------------------------------------------------------------------------
    /**
     * @brief   SharedMemoryType オブジェクトへの参照を返します。
     *
     * @return  SharedMemoryType オブジェクトへの参照
     *
     * @details
     *  自インスタンスが持つ SharedMemoryType オブジェクトへの参照を返します。
     *  この変換演算子があるため、SharedMemoryType& を引数にとる関数に対して、
     *  SharedMemory& を指定することが可能です。
     */
    NN_IMPLICIT operator SharedMemoryType&() NN_NOEXCEPT
    {
        return m_SharedMemory;
    }

    //--------------------------------------------------------------------------
    /**
     * @brief   SharedMemoryType オブジェクトへの const 参照を返します。
     *
     * @return  SharedMemoryType オブジェクトへの const 参照
     *
     * @details
     *  自インスタンスが持つ SharedMemoryType オブジェクトへの const 参照を返します。
     *  この変換演算子があるため、const SharedMemoryType& を引数にとる関数に対して、
     *  SharedMemory& を指定することが可能です。
     */
    NN_IMPLICIT operator const SharedMemoryType&() const NN_NOEXCEPT
    {
        return m_SharedMemory;
    }

    //--------------------------------------------------------------------------
    /**
     * @brief   自インスタンスの SharedMemoryType オブジェクトへのポインタを返します。
     *
     * @return  SharedMemoryType オブジェクトへのポインタ
     *
     * @details
     *  自インスタンスが持つ SharedMemoryType オブジェクトへのポインタを返します。
     */
    SharedMemoryType* GetBase() NN_NOEXCEPT
    {
        return &m_SharedMemory;
    }


private:
    SharedMemoryType m_SharedMemory;
};

}}  // namespace nn::os

