﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

/**
 * @file
 * @brief  SDK 用のスレッドローカルストレージに関する公開ヘッダファイル
 */

#pragma once

#include <nn/nn_Macro.h>
#include <nn/nn_Abort.h>
#include <nn/os/os_ThreadLocalStorage.h>
#include <nn/os/os_SdkThreadLocalStorageApi.h>


namespace nn { namespace os {

//--------------------------------------------------------------------------
/**
 * @brief   SDK 用のスレッドローカルストレージを扱うためのクラスです。
 *
 * @details
 *  SDK 用のスレッドローカルストレージ（以後 TLS）は、単一の TLS スロット に対して
 *  スレッド毎に異なるデータを扱うスレッド固有データを操作するための機能です。
 *
 *  nn::os::SdkThreadLocalStorage クラスは nn::os::ThreadLocalStorage クラスと
 *  基本的な使い方は同様ですが、SDK 専用の TLS スロットを使用するため、
 *  アプリケーション用の TLS スロットを消費しません。
 *
 */
class SdkThreadLocalStorage
{
    NN_DISALLOW_COPY( SdkThreadLocalStorage );
    NN_DISALLOW_MOVE( SdkThreadLocalStorage );

public:
    /**
     * @brief   TLS オブジェクトを構築し初期化します。
     *
     * @overloadlist{nn_os_ThreadLocalStorage_ThreadLocalStorage}
     *
     * @details
     *  コンストラクタです。@n
     *  TLS スロットを１つ確保します。@n
     *  各スレッドの TLS 初期値は 0 となります。@n
     *  詳細は @ref nn::os::AllocateTlsSlot を参照して下さい。
     */
    SdkThreadLocalStorage() NN_NOEXCEPT
    {
        auto result = nn::os::SdkAllocateTlsSlot(&m_TlsSlot, NULL);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    /**
     * @brief   TLS オブジェクトを構築し初期化します。
     *
     * @overloadlist{nn_os_ThreadLocalStorage_ThreadLocalStorage}
     *
     * @param[in]  function デストラクタ用コールバック関数エントリ
     *
     * @details
     *  コンストラクタです。@n
     *  TLS スロットを１つ確保し、デストラクタ関数を登録します。
     *  デストラクタ関数はスレッド終了時にコールバックされます。@n
     *  各スレッドの TLS 初期値は 0 となります。@n
     *  詳細は @ref nn::os::AllocateTlsSlot を参照して下さい。
     */
    explicit SdkThreadLocalStorage(TlsDestructor function) NN_NOEXCEPT
    {
        auto result = nn::os::SdkAllocateTlsSlot(&m_TlsSlot, function);
        NN_ABORT_UNLESS_RESULT_SUCCESS(result);
    }

    /**
     * @brief  デストラクタです。
     *
     * @details
     *  TLS スロットを返却し、オブジェクトを破棄します。@n
     *  詳細は @ref nn::os::FreeTlsSlot を参照して下さい。
     */
    ~SdkThreadLocalStorage() NN_NOEXCEPT
    {
        nn::os::FreeTlsSlot(m_TlsSlot);
    }

    /**
     * @brief   TLS スロットに値を設定します。
     *
     * @param[in] value     設定したい TLS 値
     *
     * @details
     *  自スレッドの TLS スロットに値を設定します。@n
     *  詳細は @ref nn::os::SetTlsValue を参照して下さい。
     */
    void SetValue(uintptr_t value) NN_NOEXCEPT
    {
        return nn::os::SetTlsValue(m_TlsSlot, value);
    }

    /**
     * @brief   TLS スロットにセットされている値を取得します。
     *
     * @return  TLS の値が返ります
     *
     * @details
     *  自スレッドの TLS スロットから値を取得します。@n
     *  詳細は @ref nn::os::GetTlsValue を参照して下さい。
     */
    uintptr_t GetValue() const NN_NOEXCEPT
    {
        return nn::os::GetTlsValue(m_TlsSlot);
    }


    /**
     * @brief   自インスタンスの TlsSlot を返します。
     *
     * @return  TlsSlot オブジェクトが返ります
     *
     * @details
     *  自インスタンスが持つ TlsSlot オブジェクトを返します。
     */
    TlsSlot GetTlsSlot() const NN_NOEXCEPT
    {
        return m_TlsSlot;
    }

private:
    TlsSlot m_TlsSlot;
};

//--------------------------------------------------------------------------

}} // namespace nn::os

