﻿/*--------------------------------------------------------------------------------*
  Copyright (C)Nintendo All rights reserved.

  These coded instructions, statements, and computer programs contain proprietary
  information of Nintendo and/or its licensed developers and are protected by
  national and international copyright laws. They may not be disclosed to third
  parties or copied or duplicated in any form, in whole or in part, without the
  prior written consent of Nintendo.

  The content herein is highly confidential and should be handled accordingly.
 *--------------------------------------------------------------------------------*/

#pragma once

/**
 * @file
 * @brief デバイスコード関連の API 宣言
 */

#include <nn/nn_Common.h>
#include <nn/devicecode/detail/devicecode_InnerFormat.h>

namespace nn {

namespace detail {
    typedef Bit32 DeviceCodeType;
}

//! @name デバイスコード関連機能
//! @{

/**
 * @brief デバイスコードです。
 * @detail
 *  デバイスコードは、デバイスドライバ層が扱う様々な対象を表現するための一意な識別子です。 @n
 *  デバイスコードの個別値の定義には @ref NN_DEVICECODE_MAKE_CODE() マクロを使用してください。 @n
 *  C リンケージ向けの等価なインタフェースを提供しています。 <nnc/nn_DeviceCode.h> を参照してください。
 */
class DeviceCode
{
public:
    /**
     * @brief デバイスコードのカテゴリを表す型です。カテゴリの有効値は <nn/devicecode/devicecode_Categories.h> で定義されています。
     */
    typedef Bit32 Category;

    /**
     * @brief デバイスコードの InnerID を表す型です。
     */
    typedef Bit32 InnerId;

public:
    /**
     * @brief デバイスコードのコンストラクタです。
     * @param[in] code デバイスコードの内部表現値
     */
    NN_IMPLICIT DeviceCode(detail::DeviceCodeType code) NN_NOEXCEPT :
        m_Value(code)
    {}

    /**
     * @brief 無効なデバイスコードを明示的に生成します。
     */
    static DeviceCode GetInvalidCode() NN_NOEXCEPT
    {
        DeviceCode code = { 0 };
        return code;
    }

    /**
     * @brief デバイスコードの内部表現値を返します。
     */
    detail::DeviceCodeType GetInternalValue() const NN_NOEXCEPT
    {
        return m_Value;
    }

    /**
     * @brief デバイスコードのカテゴリを返します。
     */
    Category GetCategory() const NN_NOEXCEPT
    {
        return NN_DETAIL_DEVICECODE_GET_CATEGORY(m_Value);
    }

    /**
     * @brief デバイスコードの InnerID を返します。
     */
    InnerId GetInnerId() const NN_NOEXCEPT
    {
        return NN_DETAIL_DEVICECODE_GET_INNERID(m_Value);
    }

    /**
     * @brief デバイスコードが指定したカテゴリに属するかを返します。
     * @param[in] category カテゴリ値
     */
    bool IsCategory(Category category) const NN_NOEXCEPT
    {
        return GetCategory() == category;
    }

    /**
     * @brief デバイスコードの比較演算子（一致）です。
     */
    bool operator==(const DeviceCode& rhs) const NN_NOEXCEPT
    {
        return GetInternalValue() == rhs.GetInternalValue();
    }

    /**
     * @brief デバイスコードの比較演算子（不一致）です。
     */
    bool operator!=(const DeviceCode& rhs) const NN_NOEXCEPT
    {
        return !(*this == rhs);
    }

private:
    detail::DeviceCodeType m_Value;
};

//! @}

}
